use chrono::{DateTime, Duration, Utc};
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use uuid::Uuid;

use crate::context::Context;
use crate::customer::{Address, Contact};
use crate::error::{new_application_error, new_bad_request_error, Error};
use crate::inventory::ReserveResult;
use crate::invoice::Invoice;
use crate::item::Item;
use crate::money::{Currency, Money};
use crate::shipping::{FulfillmentType, FulfillmentTypeSelection, ShippingQuote};

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum State {
    Shopping,
    ItemsConfirmed(DateTime<Utc>),
    Completed,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct Checkout {
    pub id: String,
    pub state: State,
    pub currency: Currency,
    pub promo_codes: Vec<String>,
    pub items: Vec<Item>,
    pub contact: Option<Contact>,
    pub shipping_address: Option<Address>,
    pub fulfillment_type: Option<FulfillmentType>,
    pub shipping_quotes: Vec<ShippingQuote>,
    pub invoice: Option<Invoice>,
}

// state transitions
impl Checkout {
    async fn enter_shopping_state<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        self.reset_shipping_options();
        ctx.free_items(&self.id, &self.items).await?;
        Ok(self.state = State::Shopping)
    }

    async fn enter_items_confirmed_state<C: Context + Send>(
        &mut self,
        ctx: &mut C,
    ) -> Result<(), Error> {
        self.reset_shipping_options();
        match ctx.reserve_items(&self.id, &self.items).await? {
            ReserveResult::Success(deadline) => {
                if let Some(address) = &self.shipping_address {
                    self.shipping_quotes = ctx
                        .get_shipping_quotes(
                            &self.currency,
                            &self.promo_codes,
                            &self.items,
                            address,
                        )
                        .await?;
                }

                self.update_item_prices(ctx).await?;
                self.update_invoice(ctx).await?;
                self.state = State::ItemsConfirmed(deadline);
                ctx.on_confirm_items(self).await
            }
            ReserveResult::StockIssue(stock_issues) => {
                Err(new_application_error("STOCK_ISSUE", stock_issues))
            }
        }
    }
}

// private methods
impl Checkout {
    async fn ensure_shopping_state<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        match self.state {
            State::Shopping => Ok(()),
            State::ItemsConfirmed(_) => self.enter_shopping_state(ctx).await,
            _ => Err(new_bad_request_error(
                "must be in the shopping or items_confirmed state",
            )),
        }
    }

    fn assert_items_confirmed_state(&self) -> Result<(), Error> {
        if !matches!(self.state, State::ItemsConfirmed(_)) {
            return Err(new_bad_request_error(
                "must be in the items_confirmed state",
            ));
        }

        Ok(())
    }

    async fn update_invoice<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        Ok(self.invoice = Some(ctx.generate_invoice(self).await?))
    }

    async fn update_item_prices<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        let item_prices = ctx
            .calculate_item_prices(&self.currency, &self.promo_codes, &self.items)
            .await?;

        for item_price in item_prices.iter() {
            for item in self.items.iter_mut() {
                if item.sku == item_price.sku {
                    item.price = item_price.price.clone();
                    item.discount = item_price.discount.clone();
                    break;
                }
            }
        }

        Ok(())
    }

    fn reset_shipping_options(&mut self) {
        self.fulfillment_type = None;
        self.shipping_quotes = vec![];
    }
}

// public methods
impl Checkout {
    pub async fn create<C: Context + Send>(
        ctx: &mut C,
        currency: Currency,
        contact: Option<Contact>,
        shipping_address: Option<Address>,
    ) -> Result<Self, Error> {
        let checkout = Checkout {
            id: Uuid::new_v4().to_string(),
            state: State::Shopping,
            currency,
            promo_codes: vec![],
            items: vec![],
            contact,
            shipping_address,
            fulfillment_type: None,
            shipping_quotes: vec![],
            invoice: None,
        };

        ctx.on_create(&checkout).await?;
        Ok(checkout)
    }

    pub async fn add_item<C: Context + Send>(
        &mut self,
        ctx: &mut C,
        sku: String,
        quantity: u64,
    ) -> Result<(), Error> {
        self.ensure_shopping_state(ctx).await?;

        let mut is_added = false;
        for item in self.items.iter_mut() {
            if item.sku == sku {
                item.quantity += quantity;
                is_added = true;
                break;
            }
        }

        if !is_added {
            let product = ctx.resolve_product(&self.currency, &sku).await?;
            let item = Item {
                sku: sku.to_string(),
                title: product.title.clone(),
                quantity,
                price: product.price.clone(),
                discount: Money::new(self.currency.clone(), "0"),
            };
            self.items.push(item);
        }

        self.update_item_prices(ctx).await?;
        self.update_invoice(ctx).await?;
        ctx.on_add_item(self, &sku, quantity).await
    }

    pub async fn remove_item<C: Context + Send>(
        &mut self,
        ctx: &mut C,
        sku: String,
        quantity: u64,
    ) -> Result<(), Error> {
        self.ensure_shopping_state(ctx).await?;

        let mut index_to_remove: Option<usize> = None;
        for (index, item) in self.items.iter_mut().enumerate() {
            if item.sku == sku {
                if item.quantity <= quantity {
                    index_to_remove = Some(index);
                    break;
                }

                item.quantity -= quantity;
                break;
            }
        }

        if let Some(index) = index_to_remove {
            self.items.remove(index);
        }

        self.update_item_prices(ctx).await?;
        self.update_invoice(ctx).await?;
        ctx.on_remove_item(self, &sku, quantity).await
    }

    pub async fn confirm_items<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        match self.state {
            State::ItemsConfirmed(deadline) => {
                if deadline - Utc::now() >= Duration::minutes(30) {
                    return Ok(());
                }

                ctx.free_items(&self.id, &self.items).await?;
                self.enter_items_confirmed_state(ctx).await?;
            }
            State::Shopping => {
                self.enter_items_confirmed_state(ctx).await?;
            }
            _ => {
                return Err(new_bad_request_error(
                    "must be in the shopping or items_confirmed state",
                ))
            }
        }

        Ok(ctx.on_confirm_items(self).await?)
    }

    pub async fn update_contact<C: Context + Send>(
        &mut self,
        ctx: &mut C,
        contact: Contact,
    ) -> Result<(), Error> {
        self.contact = Some(contact);
        ctx.on_update_contact(self).await
    }

    pub async fn update_shipping_address<C: Context + Send>(
        &mut self,
        ctx: &mut C,
        address: Address,
    ) -> Result<(), Error> {
        self.reset_shipping_options();
        self.shipping_address = Some(address);
        if let State::ItemsConfirmed(_) = self.state {
            self.shipping_quotes = ctx
                .get_shipping_quotes(
                    &self.currency,
                    &self.promo_codes,
                    &self.items,
                    self.shipping_address.as_ref().unwrap(),
                )
                .await?;
        }

        ctx.on_update_shipping_address(self).await
    }

    pub async fn update_fulfillment_type<C: Context + Send>(
        &mut self,
        ctx: &mut C,
        fulfillment_type: FulfillmentTypeSelection,
    ) -> Result<(), Error> {
        self.assert_items_confirmed_state()?;

        match fulfillment_type {
            FulfillmentTypeSelection::Pickup => {
                self.fulfillment_type = Some(FulfillmentType::Pickup);
            }
            FulfillmentTypeSelection::Shipping(quote_id) => {
                let mut updated = false;

                for quote in self.shipping_quotes.iter() {
                    if quote.id == quote_id {
                        self.fulfillment_type = Some(FulfillmentType::Shipping(quote.clone()));
                        updated = true;
                        break;
                    }
                }

                if !updated {
                    return Err(new_bad_request_error(
                        "the requested quote is not available",
                    ));
                }
            }
        }

        self.update_invoice(ctx).await?;
        ctx.on_update_fulfillment_type(self).await
    }
}
