use std::time::Duration;
use tokio::{task, time};

use async_trait::async_trait;
use cloud_pubsub::Client;
use dropshot::{ApiDescription, ServerContext};

pub use checkout_server_derive::Server;

use crate::context::{ClientContext, ClientContextProvider};
use crate::error::Error;

#[async_trait]
pub trait Server<'a, C: ClientContext>: ClientContextProvider<'a, C> {
    type Context: ServerContext;
    fn api_description() -> ApiDescription<Self::Context>;
    async fn start(self) -> Result<(), Error>;
}

pub fn spawn_token_renew(client: &Client, interval: Duration) {
    let mut client = client.clone();
    let c = client.clone();
    let renew_token_task = async move {
        let mut int = time::interval(interval);
        loop {
            if c.is_running() {
                int.tick().await;
                let refresh = client.refresh_token();
                match time::timeout(Duration::from_secs(30), refresh).await {
                    Ok(result) => {
                        if let Err(e) = result {
                            eprintln!("Failed to update token: {}", e);
                        }
                    }
                    Err(_) => {
                        eprintln!("no refresh token response in 30 seconds")
                    }
                }
            }
        }
    };

    task::spawn(renew_token_task);
}
