use cloud_pubsub::{Client, Subscription, Topic};
use std::sync::Arc;

use crate::error::{new_server_error, Error};

#[derive(Clone)]
pub struct InternalContext {
    pub pubsub_client: Arc<Client>,
    pub payment_callbacks_topic: Arc<Topic>,
    pub payment_callbacks_subscription: Arc<Subscription>,
    pub order_events_topic: Arc<Topic>,
    pub order_events_subscription: Arc<Subscription>,
}

impl InternalContext {
    pub async fn new(
        google_application_credentials: String,
        payment_callbacks_topic_name: String,
        order_events_topic_name: String,
    ) -> Result<Self, Error> {
        let pubsub_client = Arc::new(
            cloud_pubsub::Client::new(google_application_credentials)
                .await
                .map_err(|_| new_server_error("failed to create pubsub client"))?,
        );

        let payment_callbacks_topic =
            Arc::new(pubsub_client.topic(payment_callbacks_topic_name.clone()));

        let payment_callbacks_subscription =
            Arc::new(pubsub_client.subscribe(payment_callbacks_topic_name));

        let order_events_topic = Arc::new(pubsub_client.topic(order_events_topic_name.clone()));

        let order_events_subscription = Arc::new(pubsub_client.subscribe(order_events_topic_name));

        Ok(Self {
            pubsub_client,
            payment_callbacks_topic,
            payment_callbacks_subscription,
            order_events_topic,
            order_events_subscription,
        })
    }
}
