use serde::de::DeserializeOwned;
use serde::Serialize;

use crate::checkout::Checkout;
use crate::context::Context;
use crate::error::new_application_error;
use crate::error::new_not_found_error;
use crate::error::Error;
use crate::internal_context::InternalContext;
use crate::payment::cancel as cancel_payment;
use crate::payment::Initiator;
use crate::payment::Payment;
use crate::payment::PaymentProcessor;

use super::LookupKey;
use super::Order;
use super::OrderState;

pub async fn confirm_handler<C: Context + Send, P: Sync + Send + Serialize + DeserializeOwned>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    order_id: &str,
) -> Result<Order<P>, Error> {
    let maybe_od = ctx.get_order_for_update(&order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_od.unwrap();
    od.confirm(internal_ctx, ctx).await?;

    Ok(od)
}

pub async fn cancel_handler<C: Context + Send, P: Sync + Send + Serialize + DeserializeOwned>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    order_id: &str,
) -> Result<Order<P>, Error> {
    // first retrieve the associated payment for update to avoid deadlock
    let maybe_payment = get_order_payment_for_update(ctx, order_id).await?;

    let maybe_od = ctx.get_order_for_update(&order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_od.unwrap();

    if let OrderState::PaymentInProgress = od.state {
        if maybe_payment.is_none() {
            return Err(new_not_found_error(
                "the requested payment could not be found",
            ));
        }

        let mut payment = maybe_payment.unwrap();
        cancel_payment(
            internal_ctx,
            ctx,
            &mut payment,
            Initiator::<P>::Order(&mut od),
        )
        .await?;
    } else {
        od.cancel(internal_ctx, ctx).await?;
    }

    Ok(od)
}

pub async fn reconcile_handler<C: Context + Send, P: Sync + Send + Serialize + DeserializeOwned>(
    ctx: &mut C,
    lookup_key: LookupKey,
) -> Result<(), Error> {
    let maybe_order: Option<Order<P>> = match &lookup_key {
        LookupKey::CheckoutId(checkout_id) => get_order_by_checkout_id(ctx, checkout_id).await?,
        LookupKey::OrderId(order_id) => ctx.get_order_for_update(order_id).await?,
    };

    if maybe_order.is_none() {
        // this isn't supposed to happen
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_order.unwrap();

    od.reconcile(ctx).await
}

async fn get_order_by_checkout_id<
    C: Context + Send,
    P: Sync + Send + Serialize + DeserializeOwned,
>(
    ctx: &mut C,
    checkout_id: &str,
) -> Result<Option<Order<P>>, Error> {
    let maybe_checkout: Option<
        Checkout<
            Payment<
                <C as PaymentProcessor>::PrivateData,
                <C as PaymentProcessor>::PublicData,
                <C as PaymentProcessor>::ProcessArgs,
            >,
        >,
    > = ctx.get_checkout_for_update(checkout_id).await?;
    if maybe_checkout.is_none() {
        return Err(new_not_found_error(
            "the associated checkout could not be found",
        ));
    }

    let checkout = maybe_checkout.unwrap();
    if checkout.order_id.is_none() {
        return Err(new_application_error(
            "INVALID_ASSOCIATED_CHECKOUT",
            "there is no order associated with the checkout",
        ));
    }
    ctx.get_order_for_update(&checkout.order_id.unwrap()).await
}

async fn get_order_payment_for_update<C: Context + Send>(
    ctx: &mut C,
    order_id: &str,
) -> Result<
    Option<
        Payment<
            <C as PaymentProcessor>::PrivateData,
            <C as PaymentProcessor>::PublicData,
            <C as PaymentProcessor>::ProcessArgs,
        >,
    >,
    Error,
> {
    let maybe_od: Option<
        Order<
            Payment<
                <C as PaymentProcessor>::PrivateData,
                <C as PaymentProcessor>::PublicData,
                <C as PaymentProcessor>::ProcessArgs,
            >,
        >,
    > = ctx.get_order(order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let od = maybe_od.unwrap();
    ctx.get_payment_for_update(od.payment_id.as_ref().map_or("", |payment_id| payment_id))
        .await
}
