use async_trait::async_trait;
use cloud_pubsub;
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use serde_json;
use uuid::Uuid;

use crate::{
    checkout::Checkout,
    context::Context,
    customer::{Address, Contact},
    error::{new_application_error, new_invalid_state_error, Error},
    internal_context::InternalContext,
    invoice::Invoice,
    item::Item,
    payment::{handlers as payment_handlers, Initiator},
    payment::{Payment, PaymentProcessor},
    shipping::FulfillmentType,
};

pub mod handlers;

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct Order<C: Context + Send> {
    pub id: String,
    pub state: OrderState,
    pub state_reconciled: bool,
    pub contact: Contact,
    pub shipping_address: Address,
    pub items: Vec<Item>,
    pub fulfillment_type: FulfillmentType,
    pub invoice: Invoice,
    pub payment_id: Option<String>,
    // associations
    pub payment: Option<
        Payment<
            <C as PaymentProcessor>::PrivateData,
            <C as PaymentProcessor>::PublicData,
            <C as PaymentProcessor>::ProcessArgs,
        >,
    >,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum OrderState {
    PaymentInProgress, // initial state if the order is created via a checkout without payment; transitions to PendingConfirmation or Cancelled
    PendingConfirmation, // initial state if the order is created via a checkout with payment; transitions to Confirmed or Cancelled
    Confirmed, // can only enter this state from PendingConfirmation; transitions to Fulfilled or Cancelled
    Fulfilled, // a terminal state
    Cancelled, // a terminal state
}

#[derive(Serialize, Deserialize)]
pub enum LookupKey {
    CheckoutId(String),
    OrderId(String),
}

impl cloud_pubsub::FromPubSubMessage for LookupKey {
    fn from(message: cloud_pubsub::EncodedMessage) -> Result<Self, cloud_pubsub::error::Error> {
        match message.decode() {
            Ok(bytes) => {
                let val: LookupKey = serde_json::from_slice(&bytes).unwrap();
                Ok(val)
            }
            Err(e) => Err(cloud_pubsub::error::Error::from(e)),
        }
    }
}

#[async_trait]
pub trait OrderStore {
    async fn create_order<C: Context + Send>(&mut self, order: &Order<C>) -> Result<(), Error>;
    async fn update_order<C: Context + Send>(&mut self, order: &Order<C>) -> Result<(), Error>;
    async fn get_order_for_update<C: Context + Send>(
        &mut self,
        id: &str,
    ) -> Result<Option<Order<C>>, Error>;
    async fn get_order<C: Context + Send>(&mut self, id: &str) -> Result<Option<Order<C>>, Error>;
}

#[async_trait]
pub trait OrderEvents {
    async fn on_payment_in_progress<C: Context + Send>(
        &mut self,
        _order: &Order<C>,
    ) -> Result<(), Error> {
        Ok(())
    }

    async fn on_pending_confirmation<C: Context + Send>(
        &mut self,
        _order: &Order<C>,
    ) -> Result<(), Error> {
        Ok(())
    }

    async fn on_confirmed<C: Context + Send>(&mut self, _order: &Order<C>) -> Result<(), Error> {
        Ok(())
    }

    async fn on_fulfilled<C: Context + Send>(&mut self, _order: &Order<C>) -> Result<(), Error> {
        Ok(())
    }

    async fn on_cancelled<C: Context + Send>(&mut self, _order: &Order<C>) -> Result<(), Error> {
        Ok(())
    }
}

impl<C: Context + Send> Order<C> {
    pub async fn new_from_checkout(
        internal_ctx: &InternalContext,
        ctx: &mut C,
        checkout: &Checkout<C>,
    ) -> Result<Self, Error> {
        let order = Self {
            id: Uuid::new_v4().to_string(),
            state_reconciled: false,
            state: OrderState::PaymentInProgress,
            contact: checkout.contact.as_ref().unwrap().clone(),
            shipping_address: checkout.shipping_address.as_ref().unwrap().clone(),
            items: checkout.items.clone(),
            fulfillment_type: checkout.fulfillment_type.as_ref().unwrap().clone(),
            invoice: checkout.invoice.as_ref().unwrap().clone(),
            payment_id: checkout.payment_id.clone(),
            payment: None,
        };

        ctx.create_order(&order).await?;

        publish_state_change_event(internal_ctx, LookupKey::CheckoutId(checkout.id.clone()))
            .await?;

        Ok(order)
    }

    pub async fn new_from_checkout_missing_payment(
        internal_ctx: &InternalContext,
        ctx: &mut C,
        checkout: &Checkout<C>,
        payment_args: <C as PaymentProcessor>::ProcessArgs,
    ) -> Result<Self, Error> {
        let mut order = Order {
            id: Uuid::new_v4().to_string(),
            state_reconciled: false,
            state: OrderState::PaymentInProgress,
            contact: checkout.contact.as_ref().unwrap().clone(),
            shipping_address: checkout.shipping_address.as_ref().unwrap().clone(),
            items: checkout.items.clone(),
            fulfillment_type: checkout.fulfillment_type.as_ref().unwrap().clone(),
            invoice: checkout.invoice.as_ref().unwrap().clone(),
            payment_id: None,
            payment: None,
        };

        ctx.create_order(&order).await?;

        let charge_amount = order.invoice.initial_charge_amount.clone();
        let payment = payment_handlers::create_handler(
            ctx,
            Initiator::Order(&mut order),
            charge_amount,
            payment_args,
        )
        .await?;

        order.payment_id = Some(payment.id.clone());

        ctx.update_order(&order).await?;

        publish_state_change_event(internal_ctx, LookupKey::CheckoutId(checkout.id.clone()))
            .await?;

        Ok(order)
    }
}

impl<C: Context + Send> Order<C> {
    async fn assert_reconciled(&self) -> Result<(), Error> {
        match self.state_reconciled {
            true => Ok(()),
            false => Err(new_application_error(
                "ORDER_STATE_RECONCILING",
                "the order state is reconciling. please try again in a few moments",
            )),
        }
    }

    async fn transition_state(
        &mut self,
        internal_ctx: &InternalContext,
        to: OrderState,
    ) -> Result<(), Error> {
        self.assert_reconciled().await?;

        self.state = to;
        self.state_reconciled = false;

        publish_state_change_event(internal_ctx, LookupKey::OrderId(self.id.clone())).await
    }

    async fn ensure_pending_confirmation_state(
        &mut self,
        internal_ctx: &InternalContext,
    ) -> Result<(), Error> {
        match self.state {
            OrderState::PendingConfirmation => Ok(()),
            OrderState::PaymentInProgress => {
                self.transition_state(internal_ctx, OrderState::PendingConfirmation)
                    .await
            }
            _ => Err(new_invalid_state_error(
                "should be in the PaymentInProgress state",
            )),
        }
    }

    async fn ensure_confirmed_state(
        &mut self,
        internal_ctx: &InternalContext,
    ) -> Result<(), Error> {
        match self.state {
            OrderState::Confirmed => Ok(()),
            OrderState::PendingConfirmation => {
                self.transition_state(internal_ctx, OrderState::Confirmed)
                    .await
            }
            _ => Err(new_invalid_state_error(
                "should be in the PendingConfirmation state",
            )),
        }
    }

    async fn ensure_fulfilled_state(
        &mut self,
        internal_ctx: &InternalContext,
    ) -> Result<(), Error> {
        match self.state {
            OrderState::Fulfilled => Ok(()),
            OrderState::Confirmed => {
                self.transition_state(internal_ctx, OrderState::Fulfilled)
                    .await
            }
            _ => Err(new_invalid_state_error("should be in the Confirmed state")),
        }
    }

    async fn ensure_cancelled_state(
        &mut self,
        internal_ctx: &InternalContext,
    ) -> Result<(), Error> {
        match self.state {
            OrderState::Cancelled => Ok(()),
            OrderState::PaymentInProgress
            | OrderState::PendingConfirmation
            | OrderState::Confirmed => {
                self.transition_state(internal_ctx, OrderState::Cancelled)
                    .await
            }
            _ => Err(new_invalid_state_error(
                "should be in the PaymentInProgress, PendingConfirmation or Confirmed states",
            )),
        }
    }
}

impl<C: Context + Send> Order<C> {
    pub async fn prompt_confirmation(
        &mut self,
        internal_ctx: &InternalContext,
        ctx: &mut C,
    ) -> Result<(), Error> {
        self.ensure_pending_confirmation_state(internal_ctx).await?;
        ctx.update_order(self).await
    }

    pub async fn confirm(
        &mut self,
        internal_ctx: &InternalContext,
        ctx: &mut C,
    ) -> Result<(), Error> {
        self.ensure_confirmed_state(internal_ctx).await?;
        ctx.update_order(self).await
    }

    pub async fn fulfill(
        &mut self,
        internal_ctx: &InternalContext,
        ctx: &mut C,
    ) -> Result<(), Error> {
        self.ensure_fulfilled_state(internal_ctx).await?;
        ctx.update_order(self).await
    }

    pub async fn cancel(
        &mut self,
        internal_ctx: &InternalContext,
        ctx: &mut C,
    ) -> Result<(), Error> {
        self.ensure_cancelled_state(internal_ctx).await?;
        ctx.update_order(self).await
    }

    pub async fn reconcile(&mut self, ctx: &mut C) -> Result<(), Error> {
        if self.state_reconciled == true {
            return Ok(());
        }

        match self.state {
            OrderState::PaymentInProgress => ctx.on_payment_in_progress(self).await?,
            OrderState::PendingConfirmation => ctx.on_pending_confirmation(self).await?,
            OrderState::Confirmed => ctx.on_confirmed(self).await?,
            OrderState::Fulfilled => ctx.on_fulfilled(self).await?,
            OrderState::Cancelled => ctx.on_cancelled(self).await?,
        }

        self.state_reconciled = true;
        ctx.update_order(self).await
    }

    // utility method to be called by handlers
    pub async fn populate_associated_fields(&mut self, ctx: &mut C) -> Result<(), Error> {
        if let Some(ref payment_id) = self.payment_id {
            self.payment = ctx.get_payment(payment_id).await?;
        }

        Ok(())
    }
}

async fn publish_state_change_event(
    internal_ctx: &InternalContext,
    lookup_key: LookupKey,
) -> Result<(), Error> {
    internal_ctx
        .order_events_topic
        .clone()
        .publish(lookup_key)
        .await
        .map_err(|_| {
            new_application_error(
                "UPSTREAM_ERROR",
                "couldn't send message to order events topic",
            )
        })?;

    Ok(())
}
