use core::future::Future;
use sqlx::{postgres::PgPoolOptions, Executor as _, Pool, Postgres};
use std::env;
use std::pin::Pin;

pub struct Manager {
    pool: Pool<Postgres>,
}

#[derive(Debug)]
pub enum TransactionError<E> {
    ApplicationError(E),
    DBError(sqlx::Error),
}

impl Manager {
    pub async fn new() -> Result<Self, sqlx::Error> {
        let db_uri = env::var("DB_URI").unwrap();
        let db_max_conn: u32 = env::var("DB_MAX_CONN").unwrap().parse::<u32>().unwrap();

        let pool = PgPoolOptions::new()
            .max_connections(db_max_conn)
            .after_connect(|conn| {
                Box::pin(async move {
                    conn.execute("SET default_transaction_isolation TO 'serializable'")
                        .await?;
                    Ok(())
                })
            })
            .connect(&db_uri)
            .await?;

        Ok(Self { pool })
    }

    pub async fn with_tx<E, F>(&self, f: F) -> Result<(), TransactionError<E>>
    where
        for<'tx> F: FnOnce(
            &'tx mut sqlx::Transaction<sqlx::Postgres>,
        ) -> Pin<Box<dyn Future<Output = Result<(), E>> + 'tx>>,
    {
        let mut tx = self
            .pool
            .begin()
            .await
            .map_err(|e| TransactionError::DBError(e))?;

        // TODO: set tx isolation level

        if let Err(e) = f(&mut tx).await {
            tx.rollback()
                .await
                .map_err(|e| TransactionError::DBError(e))?;
            return Err(TransactionError::ApplicationError(e));
        }

        tx.commit().await.map_err(|e| TransactionError::DBError(e))
    }
}
