use crate::land::*;
use std::{ops, slice};

/// A map object
///
/// # Layout
///
/// The lands of a [`Map`] is stored on an one-dimensional [`Vec`], so you can access them directly with a single number, called 'land_id'.
///
/// For convenience, [`Map::land_id_to_xy`] and [`Map::xy_to_land_id`] are implemented to cast between land_ids and two-tuples(x, y).
///
/// # Indexing
///
/// The [`Map`] type allows to access lands by index, because it implements the
/// [`ops::Index`] trait. An example will be more explicit:
///
/// ```
/// use checkmate::Map;
///
/// let mut map = Map::with_size(10);
/// let mut land = &mut map[0];
/// assert_eq!(land.amount, 0);
///
/// land.amount = 161; // changes the amount of the land
///
/// assert_eq!(land.amount, 161);
/// ```
///
/// However be careful: if you try to access an index which isn't in the [`Map`],
/// your software will panic! You cannot do this:
///
/// ```should_panic
/// use checkmate::Map;
///
/// let map = Map::new();
/// println!("{:?}", map[0]); // it will panic!
/// ```
///
/// Use [`Map::get`] and [`Map::get_mut`] if you want to check whether the index is in
/// the [`Map`].
#[derive(Debug, Clone, Default)]
pub struct Map {
    size: usize,
    gm: Vec<Land>,
}

impl Map {
    /// Creates a new empty [`Map`].
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::Map;
    ///
    /// let map = Map::new();
    /// assert!(map.is_empty());
    /// ```
    #[inline]
    pub fn new() -> Self {
        Self::default()
    }

    /// Creates a new empty [`Map`] with a particular size.
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::Map;
    ///
    /// let map = Map::with_size(10);
    /// assert_eq!(map.size(), 10);
    /// ```
    #[inline]
    pub fn with_size(size: usize) -> Self {
        Self {
            size,
            gm: vec![Land::new(); size * size + 1],
        }
    }

    /// Returns the size of the [`Map`].
    #[inline]
    pub fn size(&self) -> usize {
        self.size
    }

    /// Returns `true` if the [`Map`] has a size of 0.
    #[inline]
    pub fn is_empty(&self) -> bool {
        self.size() == 0
    }

    /// Returns a reference to a land or subslice depending on the type of index.
    ///
    /// - If given a position, returns a reference to the land at that position or [`None`] if out of bounds.
    /// - If given a range, returns the subslice corresponding to that range, or [`None`] if out of bounds.
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::{Map, Land};
    ///
    /// let map = Map::with_size(3);
    /// assert_eq!(Some(&Land::new()), map.get(1));
    /// assert_eq!(Some(&[Land::new(), Land::new()][..]), map.get(0..2));
    /// assert_eq!(None, map.get(10));
    /// assert_eq!(None, map.get(0..20));
    /// ```
    #[inline]
    pub fn get<I>(&self, index: I) -> Option<&I::Output>
    where
        I: slice::SliceIndex<[Land]>,
    {
        self.gm.get(index)
    }

    /// Returns a mutable reference to a land or subslice depending on the type of index (see [`Map::get`]) or [`None`] if the index is out of bounds.
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::Map;
    ///
    /// let mut map = Map::with_size(10);
    ///
    /// if let Some(land) = map.get_mut(1) {
    ///     land.amount = 161;
    /// }
    /// assert_eq!(map[1].amount, 161);
    /// ```
    #[inline]
    pub fn get_mut<I>(&mut self, index: I) -> Option<&mut I::Output>
    where
        I: slice::SliceIndex<[Land]>,
    {
        self.gm.get_mut(index)
    }

    /// Returns an iterator over the [`Map`].
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::{Map, Land};
    ///
    /// let map = Map::with_size(2);
    /// let mut iterator = map.iter();
    ///
    /// assert_eq!(iterator.next(), Some(&Land::new()));
    /// assert_eq!(iterator.next(), Some(&Land::new()));
    /// assert_eq!(iterator.next(), Some(&Land::new()));
    /// assert_eq!(iterator.next(), Some(&Land::new()));
    /// assert_eq!(iterator.next(), None);
    /// ```
    #[inline]
    pub fn iter(&self) -> slice::Iter<Land> {
        let mut iter = self.gm.iter();
        iter.next();
        iter
    }

    /// Returns an iterator that allows modifying each land.
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::Map;
    ///
    /// let mut map = Map::with_size(2);
    /// for land in map.iter_mut() {
    ///     land.amount = 161;
    /// }
    /// assert_eq!(map[0].amount, 161);
    /// assert_eq!(map[1].amount, 161);
    /// ```
    #[inline]
    pub fn iter_mut(&mut self) -> slice::IterMut<Land> {
        self.gm.iter_mut()
    }

    /// Returns `true` if the two-tuple(x, y) is in the [`Map`].
    ///
    /// # Examples
    /// ```
    /// use checkmate::Map;
    ///
    /// let map = Map::with_size(10);
    /// assert!(map.is_valid_pos(1, 1));
    /// assert!(!map.is_valid_pos(11, 1));
    #[inline]
    pub fn is_valid_pos(&self, x: usize, y: usize) -> bool {
        x >= 1 && x <= self.size() && y >= 1 && y <= self.size()
    }

    /// Converts a two-tuple(x, y) into a land_id.
    ///
    /// If the position is out of bounds, it will return [`None`].
    ///
    /// # Examples
    /// ```
    /// use checkmate::Map;
    ///
    /// let map = Map::with_size(10);
    /// assert_eq!(map.xy_to_land_id(1, 1), Some(1));
    /// assert_eq!(map.xy_to_land_id(11, 11), None);
    /// ```
    #[inline]
    pub fn xy_to_land_id(&self, x: usize, y: usize) -> Option<usize> {
        if self.is_valid_pos(x, y) {
            Some((x - 1) * self.size() + y)
        } else {
            None
        }
    }

    /// Converts a land_id into a two-tuple(x, y).
    ///
    /// If the position is out of bounds, it will return [`None`].
    ///
    /// # Examples
    /// ```
    /// use checkmate::Map;
    ///
    /// let map = Map::with_size(10);
    /// assert_eq!(map.land_id_to_xy(1), Some((1, 1)));
    /// assert_eq!(map.land_id_to_xy(101), None);
    /// ```
    #[inline]
    pub fn land_id_to_xy(&self, id: usize) -> Option<(usize, usize)> {
        if id >= 1 && id <= self.size() * self.size() {
            Some(((id - 1) / self.size() + 1, (id - 1) % self.size() + 1))
        } else {
            None
        }
    }

    /// Gets the Manhattan Distance between two positions.
    ///
    /// If any of the given positions is out of bounds, it will return [`None`].
    ///
    /// # Examples
    /// ```
    /// use checkmate::Map;
    ///
    /// let map = Map::with_size(10);
    /// assert_eq!(map.dist(10, 1), Some(9));
    /// assert_eq!(map.dist(10, 101), None);
    /// ```
    #[inline]
    pub fn dist(&self, p1: usize, p2: usize) -> Option<usize> {
        let (x1, y1) = self.land_id_to_xy(p1)?;
        let (x2, y2) = self.land_id_to_xy(p2)?;

        Some(x1.abs_diff(x2) + y1.abs_diff(y2))
    }

    /// Gets the **accessible** neighbors of a position.
    ///
    /// If the position is out of bounds, it will return [`None`].
    ///
    /// **NOTE**: The outputs of this function are in a certain order. You may add some randomness on your own.
    ///
    /// # Examples
    /// ```
    /// use checkmate::{Map, Type};
    ///
    /// let mut map = Map::with_size(3);
    /// map[2].amount = 161;
    /// map[4].tp = Type::Mountain;
    /// map[8].tp = Type::Mountain;
    ///
    /// /*
    /// The map will be like:
    /// [   ] [161] [   ]
    /// [###] [   ] [   ]
    /// [   ] [###] [   ]
    /// */
    ///
    /// assert_eq!(map.get_neighbors(5), Some(vec![2, 6])); // Mountains are not included
    /// assert_eq!(map.get_neighbors(10), None);
    /// ```
    #[inline]
    pub fn get_neighbors(&self, cur: usize) -> Option<Vec<usize>> {
        let (x, y) = self.land_id_to_xy(cur)?;

        Some(
            [(-1, 0), (0, 1), (1, 0), (0, -1)]
                .iter()
                .filter_map(
                    #[inline]
                    |(dx, dy)| {
                        let px = (x as i32 + dx) as usize;
                        let py = (y as i32 + dy) as usize;

                        let p = self.xy_to_land_id(px, py)?;

                        if self[p].tp != Type::Mountain {
                            Some(p)
                        } else {
                            None
                        }
                    },
                )
                .collect(),
        )
    }
}

impl ops::Index<usize> for Map {
    type Output = Land;

    #[inline]
    fn index(&self, id: usize) -> &Self::Output {
        &self.gm[id]
    }
}

impl ops::IndexMut<usize> for Map {
    #[inline]
    fn index_mut(&mut self, id: usize) -> &mut Self::Output {
        &mut self.gm[id]
    }
}
