use crate::Error;

/// Type of a [`super::Land`]
#[derive(Copy, Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum Type {
    Empty,
    Crown,
    Land,
    City,
    Mountain,
    EmptyCity,
    Obstacle,
    Heart,
    Sword,
    Gas,
    Kit,
    Armor,
    Wifi,
}

impl Type {
    /// Converts an integer into [`Type`].
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::land::Type;
    ///
    /// let tp = Type::from(1).unwrap();
    /// assert_eq!(tp, Type::Crown);
    /// ```
    ///
    /// # Errors
    ///
    /// If the conversion fails, [`Error::WrongLandType`] will be returned.
    #[inline]
    pub fn from(value: impl Into<i32>) -> Result<Self, Error> {
        //'gas', 'kit', 'armor', 'wifi'];
        match value.into() {
            0 => Ok(Self::Empty),
            1 => Ok(Self::Crown),
            2 => Ok(Self::Land),
            3 => Ok(Self::City),
            4 => Ok(Self::Mountain),
            5 => Ok(Self::EmptyCity),
            6 => Ok(Self::Obstacle),
            7 => Ok(Self::Heart),
            8 => Ok(Self::Sword),
            9 => Ok(Self::Gas),
            10 => Ok(Self::Kit),
            11 => Ok(Self::Armor),
            12 => Ok(Self::Wifi),
            other => Err(Error::WrongLandType(other)),
        }
    }
}

impl Default for Type {
    #[inline]
    fn default() -> Self {
        Self::Empty
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn convert_into_type() {
        assert_eq!(Type::from(0), Ok(Type::Empty));
        assert_eq!(Type::from(1), Ok(Type::Crown));
        assert_eq!(Type::from(2), Ok(Type::Land));
        assert_eq!(Type::from(3), Ok(Type::City));
        assert_eq!(Type::from(4), Ok(Type::Mountain));
        assert_eq!(Type::from(5), Ok(Type::EmptyCity));
        assert_eq!(Type::from(6), Ok(Type::Obstacle));
        assert_eq!(Type::from(7), Ok(Type::Heart));
        assert_eq!(Type::from(8), Ok(Type::Sword));
        assert_eq!(Type::from(9), Ok(Type::Gas));
        assert_eq!(Type::from(10), Ok(Type::Kit));
        assert_eq!(Type::from(11), Ok(Type::Armor));
        assert_eq!(Type::from(12), Ok(Type::Wifi));
        assert_eq!(Type::from(13), Err(Error::WrongLandType(13)));
    }

    #[test]
    fn default_type() {
        assert_eq!(Type::default(), Type::Empty);
    }
}
