use crate::Error;

/// Color of a [`super::Land`]
#[derive(Copy, Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum Color {
    Grey,
    Blue,
    Red,
    Green,
    Orange,
    Pink,
    Purple,
    Chocolate,
    Maroon,
}

impl Color {
    /// Converts an integer into [`Color`].
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::Color;
    ///
    /// let color = Color::from(1).unwrap();
    /// assert_eq!(color, Color::Blue);
    /// ```
    ///
    /// # Errors
    ///
    /// If the conversion fails, [`Error::WrongLandColor`] will be returned.
    pub fn from(value: impl Into<i32>) -> Result<Self, Error> {
        match value.into() {
            0 => Ok(Self::Grey),
            1 => Ok(Self::Blue),
            2 => Ok(Self::Red),
            3 => Ok(Self::Green),
            4 => Ok(Self::Orange),
            5 => Ok(Self::Pink),
            6 => Ok(Self::Purple),
            7 => Ok(Self::Chocolate),
            8 => Ok(Self::Maroon),
            other => Err(Error::WrongLandColor(other)),
        }
    }
}

impl Default for Color {
    fn default() -> Self {
        Self::Grey
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn convert_into_color() {
        assert_eq!(Color::from(0), Ok(Color::Grey));
        assert_eq!(Color::from(1), Ok(Color::Blue));
        assert_eq!(Color::from(2), Ok(Color::Red));
        assert_eq!(Color::from(3), Ok(Color::Green));
        assert_eq!(Color::from(4), Ok(Color::Orange));
        assert_eq!(Color::from(5), Ok(Color::Pink));
        assert_eq!(Color::from(6), Ok(Color::Purple));
        assert_eq!(Color::from(7), Ok(Color::Chocolate));
        assert_eq!(Color::from(8), Ok(Color::Maroon));
        assert_eq!(Color::from(9), Err(Error::WrongLandColor(9)));
    }

    #[test]
    fn default_color() {
        assert_eq!(Color::default(), Color::Grey);
    }
}
