use crate::error::Error;

/// Color of a [`Land`]
#[derive(Copy, Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum Color {
    Grey,
    Blue,
    Red,
    Green,
    Orange,
    Pink,
    Purple,
    Chocolate,
    Maroon,
}

impl Color {
    /// Convert an integer into [`Color`].
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::land::Color;
    ///
    /// let color = Color::from(1).unwrap();
    /// assert_eq!(color, Color::Blue);
    /// ```
    ///
    /// # Errors
    ///
    /// If the conversion fails, [`Error::WrongLandColor`] will be returned.
    pub fn from(value: impl Into<i32>) -> Result<Self, Error> {
        match value.into() {
            0 => Ok(Color::Grey),
            1 => Ok(Color::Blue),
            2 => Ok(Color::Red),
            3 => Ok(Color::Green),
            4 => Ok(Color::Orange),
            5 => Ok(Color::Pink),
            6 => Ok(Color::Purple),
            7 => Ok(Color::Chocolate),
            8 => Ok(Color::Maroon),
            other => Err(Error::WrongLandColor(other)),
        }
    }
}

/// Type of a [`Land`]
#[derive(Copy, Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum Type {
    Empty,
    Crown,
    Land,
    City,
    Mountain,
    EmptyCity,
    Obstacle,
    Heart,
    Sword,
    Gas,
    Kit,
    Armor,
    Wifi,
}

impl Type {
    /// Convert an integer into [`Type`].
    ///
    /// # Examples
    ///
    /// ```
    /// use checkmate::land::Type;
    ///
    /// let tp = Type::from(1).unwrap();
    /// assert_eq!(tp, Type::Crown);
    /// ```
    ///
    /// # Errors
    ///
    /// If the conversion fails, [`Error::WrongLandType`] will be returned.
    pub fn from(value: impl Into<i32>) -> Result<Self, Error> {
        //'gas', 'kit', 'armor', 'wifi'];
        match value.into() {
            0 => Ok(Type::Empty),
            1 => Ok(Type::Crown),
            2 => Ok(Type::Land),
            3 => Ok(Type::City),
            4 => Ok(Type::Mountain),
            5 => Ok(Type::EmptyCity),
            6 => Ok(Type::Obstacle),
            7 => Ok(Type::Heart),
            8 => Ok(Type::Sword),
            9 => Ok(Type::Gas),
            10 => Ok(Type::Kit),
            11 => Ok(Type::Armor),
            12 => Ok(Type::Wifi),
            other => Err(Error::WrongLandType(other)),
        }
    }
}

/// A land object
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub struct Land {
    pub color: Color,
    pub tp: Type,
    pub amount: i32,
}
