use std::{fmt::Display, iter::once, marker::PhantomData};

use either::Either;

use crate::{
    error::{Builder as Eb, CustomBuilder as Cb},
    prim::{pure, Pure},
    util::run_drop,
    ICont, IOk, IResult, IReturn, Input, LazyError, Parser, ParserOnce,
};

/// If the first argument is Some, return its value; if None, execute the parser of the second argument.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(pure_or(Some("first"), str("second").to("second")).parse_ok("second"), Some("first"));
/// assert_eq!(pure_or(None, str("second").to("second")).parse_ok("second"), Some("second"))
/// ```
pub fn pure_or<O, I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>>(o: Option<O>, p: P) -> Either<Pure<O>, P> {
    match o {
        Some(o) => Either::Left(pure(o)),
        None => Either::Right(p),
    }
}

/// Replace the parser result with the specified value.
#[derive(Clone, Copy)]
pub struct Value<P, O>(pub(crate) P, pub(crate) O);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>, O> ParserOnce<I, C, S, M> for Value<P, O> {
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.0.run_once(cont).map(|(_, ok)| (self.1, ok))
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>, O: Clone> Parser<I, C, S, M> for Value<P, O> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.0.run(cont).map(|(_, ok)| (self.1.clone(), ok))
    }
}

/// Process parser results with functions.
#[derive(Clone, Copy)]
pub struct Map<P, F>(pub(crate) P, pub(crate) F);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>, F: FnOnce(P::Output) -> O, O> ParserOnce<I, C, S, M>
    for Map<P, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.0.run_once(cont).map(|(o, ok)| (self.1(o), ok))
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>, F: Fn(P::Output) -> O, O> Parser<I, C, S, M> for Map<P, F> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.0.run(cont).map(|(o, ok)| (self.1(o), ok))
    }
}

/// Add "expecting [specified label]" to the parser error display.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').label("special a").parse_easy("b"), Err("unexpected b, expecting special a at 0..1".to_string()));
/// ```
#[derive(Clone, Copy)]
pub struct Label<P, L>(pub(crate) P, pub(crate) L);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>, L: Display + 'static> ParserOnce<I, C, S, M> for Label<P, L> {
    type Output = P::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        if cont.ok.cutted {
            (cont.drop)()
        }
        self.0.run_once(cont).map_err(|e| e.label(self.1))
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>, L: Display + 'static + Clone> Parser<I, C, S, M> for Label<P, L> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        if cont.ok.cutted {
            (cont.drop)()
        }
        self.0.run(cont).map_err(|e| e.label(self.1.clone()))
    }
}

/// Add "expecting [specified label]" to the parser error display. The label will be evaluated lazily.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').label_with(||"special a").parse_easy("b"), Err("unexpected b, expecting special a at 0..1".to_string()));
/// ```
#[derive(Clone, Copy)]
pub struct LabelWith<P, F>(pub(crate) P, pub(crate) F);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>, L: Display, F: Fn() -> L + 'static> ParserOnce<I, C, S, M>
    for LabelWith<P, F>
{
    type Output = P::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        self.0.run_once(cont).map_err(|e| e.label_with(self.1))
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>, L: Display, F: Fn() -> L + 'static + Clone> Parser<I, C, S, M>
    for LabelWith<P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        self.0.run(cont).map_err(|e| e.label_with(self.1.clone()))
    }
}

/// Execute the next parser generated by accepting the result of the parser. If either of them fails, the whole thing will fail.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.bind(|c| char(c)).parse_ok("aa"), Some('a'));
/// assert_eq!(char('b').bind(|_| char('a')).parse_ok("aa"), None);
/// assert_eq!(any.bind(|c| char(c)).parse_ok("ab"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Bind<P, F>(pub(crate) P, pub(crate) F);
impl<I: Input, C, S, M: Cb, P1: ParserOnce<I, C, S, M>, P2: ParserOnce<I, C, S, M>, F: FnOnce(P1::Output) -> P2>
    ParserOnce<I, C, S, M> for Bind<P1, F>
{
    type Output = P2::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P2::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run_once(ICont { ok, config, drop }).and_then(|(o, ok)| self.1(o).run_once(ok.to_cont(config, drop)))
    }
}
impl<I: Input, C, S, M: Cb, P1: Parser<I, C, S, M>, P2: ParserOnce<I, C, S, M>, F: Fn(P1::Output) -> P2>
    Parser<I, C, S, M> for Bind<P1, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P2::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run(ICont { ok, config, drop }).and_then(|(o, ok)| self.1(o).run_once(ok.to_cont(config, drop)))
    }
}

/// Execute the two parsers in succession and return them as tuples.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.and(any).parse_ok("aa"), Some(('a','a')))
/// ```
#[derive(Clone, Copy)]
pub struct And<P1, P2>(pub(crate) P1, pub(crate) P2);
impl<I: Input, C, S, M: Cb, P1: ParserOnce<I, C, S, M>, P2: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M>
    for And<P1, P2>
{
    type Output = (P1::Output, P2::Output);
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<(P1::Output, P2::Output), I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0
            .run_once(ICont { ok, config, drop })
            .and_then(|(o1, ok)| self.1.run_once(ok.to_cont(config, drop)).map(|(o2, ok)| ((o1, o2), ok)))
    }
}
impl<I: Input, C, S, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>> Parser<I, C, S, M> for And<P1, P2> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<(P1::Output, P2::Output), I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0
            .run(ICont { ok, config, drop })
            .and_then(|(o1, ok)| self.1.run(ok.to_cont(config, drop)).map(|(o2, ok)| ((o1, o2), ok)))
    }
}

/// Runs two parsers in succession, returning only the first value. If either of them fails, the whole thing will fail.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.left(any).parse_ok("ab"), Some('a'));
/// assert_eq!(any.left(char('a')).parse_ok("ab"), None);
/// assert_eq!(str("chasa").to( ()).left(char(':')).parse_ok("chasa: parser combinator"), Some(()));
/// ```
#[derive(Clone, Copy)]
pub struct Left<P1, P2>(pub(crate) P1, pub(crate) P2);
impl<I: Input, C, S, M: Cb, P1: ParserOnce<I, C, S, M>, P2: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M>
    for Left<P1, P2>
{
    type Output = P1::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P1::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0
            .run_once(ICont { ok, config, drop })
            .and_then(|(o, ok)| self.1.run_once(ok.to_cont(config, drop)).map(|(_, ok)| (o, ok)))
    }
}
impl<I: Input, C, S, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>> Parser<I, C, S, M> for Left<P1, P2> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P1::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0
            .run(ICont { ok, config, drop })
            .and_then(|(o, ok)| self.1.run(ok.to_cont(config, drop)).map(|(_, ok)| (o, ok)))
    }
}

/// Runs two parsers in succession, returning only the second value. If either of them fails, the whole thing will fail.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.right(any).parse_ok("ab"), Some('b'));
/// assert_eq!(char('b').right(any).parse_ok("ab"), None);
/// assert_eq!(ws.right(str("chasa").to( ())).parse_ok("   chasa"), Some(()));
/// ```
#[derive(Clone, Copy)]
pub struct Right<P1, P2>(pub(crate) P1, pub(crate) P2);
impl<I: Input, C, S, M: Cb, P1: ParserOnce<I, C, S, M>, P2: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M>
    for Right<P1, P2>
{
    type Output = P2::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P2::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run_once(ICont { ok, config, drop }).and_then(|(_, ok)| self.1.run_once(ok.to_cont(config, drop)))
    }
}
impl<I: Input, C, S, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>> Parser<I, C, S, M> for Right<P1, P2> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P2::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run(ICont { ok, config, drop }).and_then(|(_, ok)| self.1.run(ok.to_cont(config, drop)))
    }
}
/// Place the parser between two parsers. The results at both ends will be ignored.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.between(char('('), char(')')).parse_ok("(a)"), Some('a'));
/// assert_eq!(char('a').between(char('('), char(')')).parse_ok("(a"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Between<P1, P2, P3>(pub(crate) P1, pub(crate) P2, pub(crate) P3);
impl<I: Input, C, S, M: Cb, P1: ParserOnce<I, C, S, M>, P2: ParserOnce<I, C, S, M>, P3: ParserOnce<I, C, S, M>>
    ParserOnce<I, C, S, M> for Between<P1, P2, P3>
{
    type Output = P2::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P2::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run_once(ICont { ok, config, drop }).and_then(|(_, ok)| {
            self.1
                .run_once(ok.to_cont(config, drop))
                .and_then(|(o, ok)| self.2.run_once(ok.to_cont(config, drop)).map(|(_, ok)| (o, ok)))
        })
    }
}
impl<I: Input, C, S, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, P3: Parser<I, C, S, M>> Parser<I, C, S, M>
    for Between<P1, P2, P3>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P2::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run(ICont { ok, config, drop }).and_then(|(_, ok)| {
            self.1
                .run(ok.to_cont(config, drop))
                .and_then(|(o, ok)| self.2.run(ok.to_cont(config, drop)).map(|(_, ok)| (o, ok)))
        })
    }
}

/// Pass a value to chain the parser together with the parser result, and let the parser continue.
/// Even if `bind` returns a lot of parsers of different types, `case` does not need to use `Either` artificially
/// # Example
/// ```
/// use chasa::*;
/// fn parser<I:Input<Item=char>>() -> impl ParserOnce<I,(),(),Nil,Output=usize> {
///     one_of("abc").case(|char, k| match char {
///         'a' => k.to(10),
///         'b' => k.then(parser).and(parser).map(|(a,b)| a + b),
///         'c' => k.then(parser),
///         _ => unreachable!()
///     })
/// }
/// assert_eq!(parser.parse_ok("abc"), Some(10));
/// assert_eq!(parser.parse_ok("bcaa"), Some(20));
/// assert_eq!(parser.parse_ok("bcabacca"), Some(30));
/// assert_eq!(parser.parse_ok("ba"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Case<P, F>(pub(crate) P, pub(crate) F);
impl<
        I: Input,
        C,
        S,
        M: Cb,
        P: ParserOnce<I, C, S, M>,
        O,
        F: FnOnce(P::Output, ICont<I, C, S, M>) -> IReturn<O, I, C, S, M>,
    > ParserOnce<I, C, S, M> for Case<P, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0
            .run_once(ICont { ok, config, drop })
            .and_then(|(o, ok)| self.1(o, ok.to_cont(config, drop)).0.map(|(o, k)| (o, k.ok)))
    }
}
impl<
        I: Input,
        C,
        S,
        M: Cb,
        P: Parser<I, C, S, M>,
        O,
        F: Fn(P::Output, ICont<I, C, S, M>) -> IReturn<O, I, C, S, M>,
    > Parser<I, C, S, M> for Case<P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0
            .run(ICont { ok, config, drop })
            .and_then(|(o, ok)| self.1(o, ok.to_cont(config, drop)).0.map(|(o, k)| (o, k.ok)))
    }
}

/// Sift through the parser results. If a token is sifted out, it does not consume input.
/// # Example
/// ```
/// use chasa::*;
/// let p = any.and_then(|c| match c {
///     'a' => Ok(true),
///     _ => Err(message("hello"))
/// });
/// assert_eq!(p.to_ref().parse_ok("abc"), Some(true));
/// assert_eq!(p.to_ref().parse_ok("cba"), None);
/// ```
#[derive(Clone, Copy)]
pub struct AndThen<P, F>(pub(crate) P, pub(crate) F);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>, O, F: FnOnce(P::Output) -> Result<O, Eb<M>>>
    ParserOnce<I, C, S, M> for AndThen<P, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        let pos = ok.input.pos();
        self.0.run_once(ICont { ok, config, drop }).and_then(|(o, ok)| match self.1(o) {
            Ok(o) => Ok((o, ok)),
            Err(err) => Err(err.at::<I>(ok.input.index(), pos, Some(ok.input.pos())).or_merge(ok.err)),
        })
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>, O, F: Fn(P::Output) -> Result<O, Eb<M>>> Parser<I, C, S, M>
    for AndThen<P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        let pos = ok.input.pos();
        self.0.run(ICont { ok, config, drop }).and_then(|(o, ok)| match self.1(o) {
            Ok(o) => Ok((o, ok)),
            Err(err) => Err(err.at::<I>(ok.input.index(), pos, Some(ok.input.pos())).or_merge(ok.err)),
        })
    }
}

/// If the first parser fails without consuming any input, try the next parser.
/// It is more efficient to assume that the syntax is determined when the first parser consumes input.
/// See also `cut` for input consumption.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').or(char('b')).parse_ok("aa"), Some('a'));
/// assert_eq!(char('a').or(char('b')).parse_ok("bb"), Some('b'));
/// assert_eq!(char('a').right(char('b')).or(char('b').right(char('b'))).parse_ok("bb"), Some('b'));
/// assert_eq!(char('b').right(char('b')).or(char('b').right(char('a'))).parse_ok("ba"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Or<P1, P2>(pub(crate) P1, pub(crate) P2);
impl<I: Input, C, S: Clone, M: Cb, P1: ParserOnce<I, C, S, M>, P2: ParserOnce<I, C, S, M, Output = P1::Output>>
    ParserOnce<I, C, S, M> for Or<P1, P2>
{
    type Output = P1::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P1::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        if ok.cutted {
            drop()
        }
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.0, ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => Ok((o, ok)),
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => {
                self.1.run_once(IOk { input, state, err: Some(e), cutted }.to_cont(config, drop))
            },
        }
    }
}
impl<I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M, Output = P1::Output>>
    Parser<I, C, S, M> for Or<P1, P2>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P1::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        if ok.cutted {
            drop()
        }
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.0.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => Ok((o, ok)),
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => {
                self.1.run(IOk { input, state, err: Some(e), cutted }.to_cont(config, drop))
            },
        }
    }
}

/// It returns `Some` if the parser succeeds, returns `None` if the parser does not consume any input and fails, and fails if the parser consumes some input and fails.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').or_not().parse_ok("aa"), Some(Some('a')));
/// assert_eq!(char('a').or_not().parse_ok("bb"), Some(None));
/// assert_eq!(char('a').right(char('b')).or_not().parse_ok("bb"), Some(None));
/// assert_eq!(char('b').right(char('b')).or_not().parse_ok("ba"), None);
/// ```
#[derive(Clone, Copy)]
pub struct OrNot<P>(pub(crate) P);
impl<I: Input, C, S: Clone, M: Cb, P: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M> for OrNot<P> {
    type Output = Option<P::Output>;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<Option<P::Output>, I, S, M> {
        let ICont { ok, config, drop } = cont;
        if ok.cutted {
            drop()
        }
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.0, ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => Ok((Some(o), ok)),
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => Ok((None, IOk { input, state, err: Some(e), cutted })),
        }
    }
}
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>> Parser<I, C, S, M> for OrNot<P> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<Option<P::Output>, I, S, M> {
        let ICont { ok, config, drop } = cont;
        if ok.cutted {
            drop()
        }
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.0.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => Ok((Some(o), ok)),
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => Ok((None, IOk { input, state, err: Some(e), cutted })),
        }
    }
}

/// If the parser fails, it will not consume any input. This is useful for parsing tokens that have multiple parts combined.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('b').right(char('b')).or(char('b').right(char('a'))).parse_ok("ba"), None);
/// assert_eq!(char('b').right(char('b')).cut().or(char('b').right(char('a'))).parse_ok("ba"), Some('a'));
/// ```
#[derive(Clone, Copy)]
pub struct Cut<P>(pub(crate) P);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M> for Cut<P> {
    type Output = P::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        if ok.cutted {
            drop()
        }
        match run_drop(self.0, ICont { ok, config, drop: &mut || {} }, ()) {
            (Ok((o, ok)), d) => {
                if d.is_some() {
                    drop()
                }
                Ok((o, IOk { cutted: ok.cutted && !d.is_some(), ..ok }))
            },
            (Err(e), _) => Err(e),
        }
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>> Parser<I, C, S, M> for Cut<P> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        if ok.cutted {
            drop()
        }
        match run_drop(self.0.to_ref(), ICont { ok, config, drop }, ()) {
            (Ok((o, ok)), d) => {
                if d.is_some() {
                    drop()
                }
                Ok((o, IOk { cutted: ok.cutted && !d.is_some(), ..ok }))
            },
            (Err(e), _) => Err(e),
        }
    }
}

/// Returns the parser result with the position before and after the parse.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').ranged().parse_ok("a"), Some(('a',0,1)));
/// assert_eq!(str("abcd").to(()).ranged().parse_ok("abcd"), Some(((),0,4)));
/// assert_eq!(ws.right(str("abcd").to(()).ranged()).parse_ok("    abcd"), Some(((),4,8)))
/// ```
#[derive(Clone, Copy)]
pub struct Ranged<P>(pub(crate) P);
impl<I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M> for Ranged<P> {
    type Output = (P::Output, I::Pos, I::Pos);
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<Self::Output, I, S, M> {
        let pos = cont.ok.input.pos();
        self.0.run_once(cont).map(|(o, ok)| ((o, pos, ok.input.pos()), ok))
    }
}
impl<I: Input, C, S, M: Cb, P: Parser<I, C, S, M>> Parser<I, C, S, M> for Ranged<P> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<Self::Output, I, S, M> {
        let pos = cont.ok.input.pos();
        self.0.run(cont).map(|(o, ok)| ((o, pos, ok.input.pos()), ok))
    }
}

/// Returns together with the string accepted by the parser.
pub struct GetString<P, O>(pub(crate) P, pub(crate) PhantomData<fn() -> O>);
impl<O: FromIterator<I::Item>, I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M>
    for GetString<P, O>
{
    type Output = (P::Output, O);
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<(P::Output, O), I, S, M> {
        let (mut input, begin) = (cont.ok.input.clone(), cont.ok.input.index());
        self.0.run_once(cont).map(|(o, ok)| {
            let end = ok.input.index();
            let str = InputIter { input: &mut input, begin, end }.collect();
            ((o, str), ok)
        })
    }
}
impl<O: FromIterator<I::Item>, I: Input, C, S, M: Cb, P: Parser<I, C, S, M>> Parser<I, C, S, M> for GetString<P, O> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<(P::Output, O), I, S, M> {
        let (mut input, begin) = (cont.ok.input.clone(), cont.ok.input.index());
        self.0.run(cont).map(|(o, ok)| {
            let end = ok.input.index();
            let str = InputIter { input: &mut input, begin, end }.collect();
            ((o, str), ok)
        })
    }
}
struct InputIter<'a, I: Input> {
    input: &'a mut I,
    begin: usize,
    end: usize,
}
impl<'a, I: Input> Iterator for InputIter<'a, I> {
    type Item = I::Item;
    #[inline]
    fn next(&mut self) -> Option<I::Item> {
        if self.input.index() < self.end {
            self.input.next()?.ok()
        } else {
            None
        }
    }
    #[inline]
    fn size_hint(&self) -> (usize, Option<usize>) {
        let len = self.end - self.begin;
        (len, Some(len))
    }
}
impl<'a, I: Input> ExactSizeIterator for InputIter<'a, I> {}

pub struct GetStringExtend<P, O>(pub(crate) P, pub(crate) O);
#[inline]
pub fn extend_with_str<O, P>(str: O, parser: P) -> GetStringExtend<P, O> {
    GetStringExtend(parser, str)
}
impl<O: Extend<I::Item>, I: Input, C, S, M: Cb, P: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M>
    for GetStringExtend<P, O>
{
    type Output = (P::Output, O);
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<(P::Output, O), I, S, M> {
        let (mut input, begin) = (cont.ok.input.clone(), cont.ok.input.index());
        self.0.run_once(cont).map(|(o, ok)| {
            let end = ok.input.index();
            let mut str = self.1;
            str.extend(InputIter { input: &mut input, begin, end });
            ((o, str), ok)
        })
    }
}
impl<O: Extend<I::Item> + Clone, I: Input, C, S, M: Cb, P: Parser<I, C, S, M>> Parser<I, C, S, M>
    for GetStringExtend<P, O>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<(P::Output, O), I, S, M> {
        let (mut input, begin) = (cont.ok.input.clone(), cont.ok.input.index());
        self.0.run(cont).map(|(o, ok)| {
            let end = ok.input.index();
            let mut str = self.1.clone();
            str.extend(InputIter { input: &mut input, begin, end });
            ((o, str), ok)
        })
    }
}
/// If successful, it does not consume input. The subsequent parser will read the same part again.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').and(char('a')).parse_ok("a"), None);
/// assert_eq!(before(char('a')).and(char('a')).parse_ok("a"), Some(('a','a')));
/// assert_eq!(char('a').and(char('b')).parse_ok("ab"), Some(('a','b')));
/// assert_eq!(before(char('a')).and(char('b')).parse_ok("ab"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Before<P>(pub(crate) P);
#[inline]
pub fn before<P>(parser: P) -> Before<P> {
    Before(parser)
}

impl<I: Input, C, S: Clone, M: Cb, P: ParserOnce<I, C, S, M>> ParserOnce<I, C, S, M> for Before<P> {
    type Output = P::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        let (input, state) = (cont.ok.input.clone(), cont.ok.state.clone());
        self.0.run_once(cont).map(|(o, IOk { err, .. })| (o, IOk { input, state, err, cutted: false }))
    }
}
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>> Parser<I, C, S, M> for Before<P> {
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P::Output, I, S, M> {
        let (input, state) = (cont.ok.input.clone(), cont.ok.state.clone());
        self.0.run(cont).map(|(o, IOk { err, .. })| (o, IOk { input, state, err, cutted: false }))
    }
}

/// Swaps parser successes with failures that do not consume any input. Results will be discarded.
/// If the original parser consumes input and fails, the whole thing will fail.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(char('a').and(char('a')).parse_ok("aa"), Some(('a','a')));
/// assert_eq!(not_followed_by(char('a'),'a').and(char('a')).parse_ok("aa"), None);
/// assert_eq!(not_followed_by(char('b'),'b').and(char('a')).parse_ok("a"), Some(((),'a')));
/// assert_eq!(not_followed_by(char('b'),'b').and(any).parse_ok("b"), None);
/// assert_eq!(not_followed_by(char('b').and(char('a')),'b').and(any).parse_ok("bb"), None);
/// ```
#[derive(Clone, Copy)]
pub struct NotFollowedBy<P, L>(pub(crate) P, pub(crate) L);
#[inline]
pub fn not_followed_by<P, L: Display + 'static>(parser: P, label: L) -> NotFollowedBy<P, L> {
    NotFollowedBy(parser, label)
}
impl<I: Input, C, S: Clone, M: Cb, P: ParserOnce<I, C, S, M>, L: Display + 'static> ParserOnce<I, C, S, M>
    for NotFollowedBy<P, L>
{
    type Output = ();
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<(), I, S, M> {
        let ICont { ok, config, drop } = cont;
        let (input, state, pos) = (ok.input.clone(), ok.state.clone(), ok.input.pos());
        match run_drop(self.0, ICont { ok, config, drop: &mut || {} }, (input, state)) {
            (Ok((_, ok)), _) => {
                Err(Eb::unexpected(self.1).at::<I>(ok.input.index(), pos, Some(ok.input.pos())).or_merge(ok.err))
            },
            (Err(e), None) => {
                drop();
                Err(e)
            },
            (Err(e), Some((input, state))) => Ok(((), IOk { input, state, err: Some(e), cutted: false })),
        }
    }
}
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, L: Display + Clone + 'static> Parser<I, C, S, M>
    for NotFollowedBy<P, L>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<(), I, S, M> {
        let ICont { ok, config, drop } = cont;
        let (input, state, pos) = (ok.input.clone(), ok.state.clone(), ok.input.pos());
        match run_drop(self.0.to_ref(), ICont { ok, config, drop: &mut || {} }, (input, state)) {
            (Ok((_, ok)), _) => Err(Eb::unexpected(self.1.clone())
                .at::<I>(ok.input.index(), pos, Some(ok.input.pos()))
                .or_merge(ok.err)),
            (Err(e), None) => {
                drop();
                Err(e)
            },
            (Err(e), Some((input, state))) => Ok(((), IOk { input, state, err: Some(e), cutted: false })),
        }
    }
}

fn run_fold<O, I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>>(
    o: O, p: P, cont: ICont<I, C, S, M>, f: impl Fn(O, P::Output) -> O,
) -> IResult<O, I, S, M> {
    let ICont { ok, config, drop } = cont;
    let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
    match run_drop(p.to_ref(), ICont { ok, config, drop }, (input, state)) {
        (Ok((o2, ok)), _) => run_fold(f(o, o2), p, ok.to_cont(config, drop), f),
        (Err(e), None) => Err(e),
        (Err(e), Some((input, state))) => Ok((o, IOk { input, state, cutted, err: Some(e) })),
    }
}

/// Run the parser greedily as many times as possible and fold the results.
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of("0123456789").and_then(|c: char| c.to_string().parse::<usize>().map_err(message));
/// let d = d.to_ref();
/// assert_eq!(d.fold(0,|a,b| a+b).parse_ok("12345"), Some(15));
/// assert_eq!(d.fold(0,|a,b| a+b).parse_ok(""), Some(0));
/// assert_eq!(d.fold(0,|a,b| a+b).and(d).parse_ok("12345"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Fold<T, P, F>(T, P, F);
#[inline]
pub fn fold<T, P, F>(init: T, parser: P, succ: F) -> Fold<T, P, F> {
    Fold(init, parser, succ)
}
impl<I: Input, C, S: Clone, M: Cb, T, P: Parser<I, C, S, M>, F: Fn(T, P::Output) -> T> ParserOnce<I, C, S, M>
    for Fold<T, P, F>
{
    type Output = T;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<T, I, S, M> {
        run_fold(self.0, self.1, cont, self.2)
    }
}
impl<I: Input, C, S: Clone, M: Cb, T: Clone, P: Parser<I, C, S, M>, F: Fn(T, P::Output) -> T> Parser<I, C, S, M>
    for Fold<T, P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<T, I, S, M> {
        run_fold(self.0.clone(), self.1.to_ref(), cont, &self.2)
    }
}

/// Run the parser greedily as many times as possible and fold the results. Require more than one success.
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of("0123456789").and_then(|c: char| c.to_string().parse::<usize>().map_err(message));
/// let d = d.to_ref();
/// assert_eq!(d.fold1(0,|a,b| a+b).parse_ok("12345"), Some(15));
/// assert_eq!(d.fold1(0,|a,b| a+b).parse_ok(""), None);
/// assert_eq!(d.fold1(0,|a,b| a+b).and(d).parse_ok("12345"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Fold1<T, P, F>(T, P, F);
#[inline]
pub fn fold1<T, P, F>(init: T, parser: P, succ: F) -> Fold1<T, P, F> {
    Fold1(init, parser, succ)
}
impl<I: Input, C, S: Clone, M: Cb, T, P: Parser<I, C, S, M>, F: Fn(T, P::Output) -> T> ParserOnce<I, C, S, M>
    for Fold1<T, P, F>
{
    type Output = T;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<T, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.1
            .run(ICont { ok, config, drop })
            .and_then(|(o, ok)| run_fold(self.2(self.0, o), self.1, ok.to_cont(config, drop), self.2))
    }
}
impl<I: Input, C, S: Clone, M: Cb, T: Clone, P: Parser<I, C, S, M>, F: Fn(T, P::Output) -> T> Parser<I, C, S, M>
    for Fold1<T, P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<T, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.1
            .run(ICont { ok, config, drop })
            .and_then(|(o, ok)| run_fold(self.2(self.0.clone(), o), self.1.to_ref(), ok.to_cont(config, drop), &self.2))
    }
}

// sepが成功した後に本体がempty failしても元に戻る仕組み
fn run_left_sep<
    I: Input,
    C,
    S: Clone,
    M: Cb,
    T,
    P1: Parser<I, C, S, M>,
    P2: Parser<I, C, S, M>,
    F: Fn(T, P1::Output) -> T,
>(
    t: T, p: P1, sep: P2, succ: F, cont: ICont<I, C, S, M>,
) -> IResult<T, I, S, M> {
    let ICont { ok, config, drop } = cont;
    let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
    if cutted {
        drop()
    }
    match run_drop(sep.to_ref(), ICont { ok, config, drop }, ()) {
        (Err(e), None) => Err(e),
        (Err(e), Some(())) => Ok((t, IOk { input, state, cutted, err: Some(e) })),
        (Ok((_, ok)), _) => {
            if ok.cutted {
                drop()
            }
            match run_drop(p.to_ref(), ok.to_cont(config, drop), (input, state)) {
                (Err(e), None) => Err(e),
                (Err(e), Some((input, state))) => Ok((t, IOk { input, state, cutted, err: Some(e) })),
                (Ok((o, ok)), _) => run_left_sep(succ(t, o), p, sep, succ, ok.to_cont(config, drop)),
            }
        },
    }
}

/// The parser will be separated by another parser (the result will be discarded) and folded.
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of("0123456789").and_then(|c: char| c.to_string().parse::<usize>().map_err(message));
/// let d = d.to_ref();
/// assert_eq!(d.sep_fold(0, char(','),|a,b| a+b).parse_ok("1,2,3,4,5"), Some(15));
/// assert_eq!(d.sep_fold(0, char(','),|a,b| a+b).parse_ok(""), Some(0));
/// assert_eq!(d.sep_fold(0, char(','),|a,b| a+b).and(char(',').right(d)).parse_ok("1,2,3,4,5,6"), None);
/// assert_eq!(d.sep_fold(0, char(','),|a,b| a+b).and(char(',')).parse_ok("1,2,3,4,"), Some((10, ',')));
/// ```
#[derive(Clone, Copy)]
pub struct SepFold<T, P1, P2, F> {
    pub(crate) init: T,
    pub(crate) p: P1,
    pub(crate) sep: P2,
    pub(crate) succ: F,
}

impl<I: Input, C, S: Clone, M: Cb, T, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, F: Fn(T, P1::Output) -> T>
    ParserOnce<I, C, S, M> for SepFold<T, P1, P2, F>
{
    type Output = T;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<T, I, S, M> {
        let ICont { ok, config, drop } = cont;
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.p.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => {
                run_left_sep((self.succ)(self.init, o), self.p, self.sep, self.succ, ok.to_cont(config, drop))
            },
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => Ok((self.init, IOk { input, state, cutted, err: Some(e) })),
        }
    }
}
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        T: Clone,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: Fn(T, P1::Output) -> T,
    > Parser<I, C, S, M> for SepFold<T, P1, P2, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<T, I, S, M> {
        let ICont { ok, config, drop } = cont;
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.p.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => run_left_sep(
                (self.succ)(self.init.clone(), o),
                self.p.to_ref(),
                self.sep.to_ref(),
                &self.succ,
                ok.to_cont(config, drop),
            ),
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => Ok((self.init.clone(), IOk { input, state, cutted, err: Some(e) })),
        }
    }
}

// sepが成功した後に本体がempty failしても元に戻る仕組み
fn run_left_sep1<
    I: Input,
    C,
    S: Clone,
    M: Cb,
    P1: Parser<I, C, S, M>,
    P2: Parser<I, C, S, M>,
    F: Fn(P1::Output, P2::Output, P1::Output) -> P1::Output,
>(
    t: P1::Output, p: P1, sep: P2, succ: F, cont: ICont<I, C, S, M>,
) -> IResult<P1::Output, I, S, M> {
    let ICont { ok, config, drop } = cont;
    let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
    if cutted {
        drop()
    }
    match run_drop(sep.to_ref(), ICont { ok, config, drop }, ()) {
        (Err(e), None) => Err(e),
        (Err(e), Some(())) => Ok((t, IOk { input, state, cutted, err: Some(e) })),
        (Ok((op, ok)), _) => {
            if ok.cutted {
                drop()
            }
            match run_drop(p.to_ref(), ok.to_cont(config, drop), (input, state)) {
                (Err(e), None) => Err(e),
                (Err(e), Some((input, state))) => Ok((t, IOk { input, state, cutted, err: Some(e) })),
                (Ok((o, ok)), _) => run_left_sep1(succ(t, op, o), p, sep, succ, ok.to_cont(config, drop)),
            }
        },
    }
}

/// The parser will be separated by another parser (the result will be discarded) and folded. Require more than one success.
/// It can be used for folding without discarding the separator.
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of("0123456789").and_then(|c: char| c.to_string().parse::<isize>().map_err(message));
/// let p = d.sep_fold1(char(','),|a,_,b| a+b);
/// assert_eq!(p.parse_ok("1,2,3,4,5"), Some(15));
/// assert_eq!(p.parse_ok(""), None);
/// assert_eq!(p.and(char(',').right(d)).parse_ok("1,2,3,4,5,6"), None);
/// assert_eq!(p.and(char(',')).parse_ok("1,2,3,4,"), Some((10, ',')));
///
/// let op = char('+').to(1).or(char('-').to(-1));
/// let p = d.sep_fold1(op, |a,sign,b| a + sign*b);
/// assert_eq!(p.parse_ok("1+2+3-4+5"), Some(7));
/// ```
#[derive(Clone, Copy)]
pub struct SepFold1<P1, P2, F> {
    pub(crate) p: P1,
    pub(crate) sep: P2,
    pub(crate) succ: F,
}
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: Fn(P1::Output, P2::Output, P1::Output) -> P1::Output,
    > ParserOnce<I, C, S, M> for SepFold1<P1, P2, F>
{
    type Output = P1::Output;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<P1::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.p
            .run(ICont { ok, config, drop })
            .and_then(|(o, ok)| run_left_sep1(o, self.p, self.sep, self.succ, ok.to_cont(config, drop)))
    }
}
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: Fn(P1::Output, P2::Output, P1::Output) -> P1::Output,
    > Parser<I, C, S, M> for SepFold1<P1, P2, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<P1::Output, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.p.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            run_left_sep1(o, self.p.to_ref(), self.sep.to_ref(), &self.succ, ok.to_cont(config, drop))
        })
    }
}

fn run_extend<O: Extend<P::Output>, I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>>(
    mut o: O, p: P, cont: ICont<I, C, S, M>,
) -> IResult<O, I, S, M> {
    let ICont { ok, config, drop } = cont;
    let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
    match run_drop(p.to_ref(), ICont { ok, config, drop }, (input, state)) {
        (Ok((o2, ok)), _) => {
            o.extend(once(o2));
            run_extend(o, p, ok.to_cont(config, drop))
        },
        (Err(e), None) => Err(e),
        (Err(e), Some((input, state))) => Ok((o, IOk { input, state, cutted, err: Some(e) })),
    }
}

/// Repeat extend with parser results.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.extend(String::new()).parse_ok("abcde"), Some("abcde".to_string()));
/// assert_eq!(char('a').extend(String::new()).parse_ok("aaabbb"), Some("aaa".to_string()));
/// assert_eq!(char('a').extend(String::new()).parse_ok("b"), Some("".to_string()));
/// ```
#[derive(Clone, Copy)]
pub struct ExtendParser<O, P>(pub(crate) O, pub(crate) P);
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, O: Extend<P::Output>> ParserOnce<I, C, S, M>
    for ExtendParser<O, P>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        run_extend(self.0, self.1, cont)
    }
}
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, O: Extend<P::Output> + Clone> Parser<I, C, S, M>
    for ExtendParser<O, P>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        run_extend(self.0.clone(), self.1.to_ref(), cont)
    }
}

/// Repeat extend with parser results. Require more than one success.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.extend1(String::new()).parse_ok("abcde"), Some("abcde".to_string()));
/// assert_eq!(char('a').extend1(String::new()).parse_ok("aaabbb"), Some("aaa".to_string()));
/// assert_eq!(char('a').extend1(String::new()).parse_ok("b"), None);
/// ```
#[derive(Clone, Copy)]
pub struct Extend1Parser<O, P>(pub(crate) O, pub(crate) P);
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, O: Extend<P::Output>> ParserOnce<I, C, S, M>
    for Extend1Parser<O, P>
{
    type Output = O;
    #[inline]
    fn run_once(mut self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.1.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            self.0.extend(once(o));
            run_extend(self.0, self.1, ok.to_cont(config, drop))
        })
    }
}
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, O: Extend<P::Output> + Clone> Parser<I, C, S, M>
    for Extend1Parser<O, P>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.1.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            let mut init = self.0.clone();
            init.extend(once(o));
            run_extend(init, self.1.to_ref(), ok.to_cont(config, drop))
        })
    }
}

// sepが成功した後に本体がempty failしても元に戻る仕組み
fn run_left_sep_extend<
    I: Input,
    C,
    S: Clone,
    M: Cb,
    T: Extend<P1::Output>,
    P1: Parser<I, C, S, M>,
    P2: Parser<I, C, S, M>,
>(
    mut t: T, p: P1, sep: P2, cont: ICont<I, C, S, M>,
) -> IResult<T, I, S, M> {
    let ICont { ok, config, drop } = cont;
    let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
    if cutted {
        drop()
    }
    match run_drop(sep.to_ref(), ICont { ok, config, drop }, ()) {
        (Err(e), None) => Err(e),
        (Err(e), Some(())) => Ok((t, IOk { input, state, cutted, err: Some(e) })),
        (Ok((_, ok)), _) => {
            if ok.cutted {
                drop()
            }
            match run_drop(p.to_ref(), ok.to_cont(config, drop), (input, state)) {
                (Err(e), None) => Err(e),
                (Err(e), Some((input, state))) => Ok((t, IOk { input, state, cutted, err: Some(e) })),
                (Ok((o, ok)), _) => {
                    t.extend(once(o));
                    run_left_sep_extend(t, p, sep, ok.to_cont(config, drop))
                },
            }
        },
    }
}

/// Repeat the parser, separating it with other parsers, and extend it.
#[derive(Clone, Copy)]
pub struct SepExtend<O, P1, P2> {
    pub(crate) init: O,
    pub(crate) p: P1,
    pub(crate) sep: P2,
}
impl<I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, O: Extend<P1::Output>>
    ParserOnce<I, C, S, M> for SepExtend<O, P1, P2>
{
    type Output = O;
    #[inline]
    fn run_once(mut self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.p.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => {
                self.init.extend(once(o));
                run_left_sep_extend(self.init, self.p, self.sep, ok.to_cont(config, drop))
            },
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => Ok((self.init, IOk { input, state, cutted, err: Some(e) })),
        }
    }
}
impl<I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, O: Extend<P1::Output> + Clone>
    Parser<I, C, S, M> for SepExtend<O, P1, P2>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
        match run_drop(self.p.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => {
                let mut init = self.init.clone();
                init.extend(once(o));
                run_left_sep_extend(init, self.p.to_ref(), self.sep.to_ref(), ok.to_cont(config, drop))
            },
            (Err(e), None) => Err(e),
            (Err(e), Some((input, state))) => Ok((self.init.clone(), IOk { input, state, cutted, err: Some(e) })),
        }
    }
}

/// Repeat the parser, separating it with other parsers, and extend it. Require more than one success.
#[derive(Clone, Copy)]
pub struct SepExtend1<O, P1, P2> {
    pub(crate) init: O,
    pub(crate) p: P1,
    pub(crate) sep: P2,
}
impl<I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, O: Extend<P1::Output>>
    ParserOnce<I, C, S, M> for SepExtend1<O, P1, P2>
{
    type Output = O;
    #[inline]
    fn run_once(mut self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.p.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            self.init.extend(once(o));
            run_left_sep_extend(self.init, self.p, self.sep, ok.to_cont(config, drop))
        })
    }
}
impl<I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, O: Extend<P1::Output> + Clone>
    Parser<I, C, S, M> for SepExtend1<O, P1, P2>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.p.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            let mut init = self.init.clone();
            init.extend(once(o));
            run_left_sep_extend(init, self.p.to_ref(), self.sep.to_ref(), ok.to_cont(config, drop))
        })
    }
}

/// A recursive parser. It takes `T` and processes the continuation with `Err(T)` and the output with `Ok(U)`.
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of("0123456789").and_then(|c: char| c.to_string().parse::<usize>().map_err(message));
/// let d = d.to_ref();
/// let p = char('(').right(tail_rec(0, move |n| d.map(move |m| Err(m+n)).or(char(')').to(Ok(n)))));
/// assert_eq!(p.to_ref().parse_ok("(12345)"), Some(15));
/// assert_eq!(p.to_ref().parse_ok("(12)345)"), Some(3));
/// assert_eq!(p.to_ref().parse_ok("()"), Some(0));
/// assert_eq!(p.to_ref().parse_ok("(12a345)"), None);
/// ```
#[derive(Clone, Copy)]
pub struct TailRec<O, F>(pub(crate) O, pub(crate) F);
#[inline]
pub fn tail_rec<I: Input, C, S, M: Cb, O1, O2, P: ParserOnce<I, C, S, M, Output = Result<O2, O1>>, F: Fn(O1) -> P>(
    init: O1, f: F,
) -> TailRec<O1, F> {
    TailRec(init, f)
}
impl<I: Input, C, S, M: Cb, O1, O2, P: ParserOnce<I, C, S, M, Output = Result<O2, O1>>, F: Fn(O1) -> P>
    ParserOnce<I, C, S, M> for TailRec<O1, F>
{
    type Output = O2;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O2, I, S, M> {
        run_tail_rec(self.0, self.1, cont)
    }
}
impl<I: Input, C, S, M: Cb, O1: Clone, O2, P: ParserOnce<I, C, S, M, Output = Result<O2, O1>>, F: Fn(O1) -> P>
    Parser<I, C, S, M> for TailRec<O1, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O2, I, S, M> {
        run_tail_rec(self.0.clone(), &self.1, cont)
    }
}
fn run_tail_rec<I: Input, C, S, M: Cb, O1, O2, P: ParserOnce<I, C, S, M, Output = Result<O2, O1>>, F: Fn(O1) -> P>(
    o: O1, f: F, cont: ICont<I, C, S, M>,
) -> IResult<O2, I, S, M> {
    let ICont { ok, config, drop } = cont;
    f(o).run_once(ICont { ok, config, drop }).and_then(|(o, ok)| match o {
        Err(o) => run_tail_rec(o, f, ok.to_cont(config, drop)),
        Ok(o) => Ok((o, ok)),
    })
}

/// Collects successive runs of the parser as if they were iterators. However, a failure that consumes input will cause the whole thing to fail.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.many().parse_ok("1234"), Some(vec!['1','2','3','4']));
/// assert_eq!(any.many().parse_ok("1234"), Some("1234".to_string()));
/// assert_eq!(char('a').many().parse_ok("aaabaaba"), Some("aaa".to_string()));
/// assert_eq!(char('a').many().parse_ok(""), Some("".to_string()));
/// ```
pub struct Many<P, O>(P, PhantomData<fn() -> O>);
impl<P: Clone, O> Clone for Many<P, O> {
    #[inline]
    fn clone(&self) -> Self {
        Self(self.0.clone(), PhantomData)
    }
}
impl<P: Copy, O> Copy for Many<P, O> {}
#[inline]
pub fn many<O, P>(parser: P) -> Many<P, O> {
    Many(parser, PhantomData)
}
impl<I: Input, C, S: Clone, M: Cb, O: FromIterator<P::Output>, P: Parser<I, C, S, M>> ParserOnce<I, C, S, M>
    for Many<P, O>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.run(cont)
    }
}
impl<I: Input, C, S: Clone, M: Cb, O: FromIterator<P::Output>, P: Parser<I, C, S, M>> Parser<I, C, S, M>
    for Many<P, O>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let mut ret = Some(Ok(cont));
        let iter = ParserIterator { parser: &self.0, ret: &mut ret };
        let o = iter.collect::<O>();
        match ret.unwrap() {
            Ok(cont) => Ok((o, cont.ok)),
            Err(e) => Err(e),
        }
    }
}
/// Collects successive runs of the parser as if they were iterators. However, if more than one input is consumed and fails, or none of them succeeds, the whole thing will fail.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(any.many1::<Vec<_>>().parse_ok("1234"), Some(vec!['1','2','3','4']));
/// assert_eq!(any.many1::<String>().parse_ok("1234"), Some("1234".to_string()));
/// assert_eq!(char('a').many1::<String>().parse_ok("aaabaaba"), Some("aaa".to_string()));
/// assert_eq!(char('a').many1::<String>().parse_ok(""), None);
/// ```
pub struct Many1<P, O>(P, PhantomData<fn() -> O>);
impl<P: Clone, O> Clone for Many1<P, O> {
    #[inline]
    fn clone(&self) -> Self {
        Self(self.0.clone(), PhantomData)
    }
}
impl<P: Copy, O> Copy for Many1<P, O> {}
#[inline]
pub fn many1<O, P>(parser: P) -> Many1<P, O> {
    Many1(parser, PhantomData)
}
impl<I: Input, C, S: Clone, M: Cb, O: FromIterator<P::Output>, P: Parser<I, C, S, M>> ParserOnce<I, C, S, M>
    for Many1<P, O>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.run(cont)
    }
}
impl<I: Input, C, S: Clone, M: Cb, O: FromIterator<P::Output>, P: Parser<I, C, S, M>> Parser<I, C, S, M>
    for Many1<P, O>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            let mut ret = Some(Ok(ICont { ok, config, drop }));
            let iter = ParserIterator { parser: &self.0, ret: &mut ret };
            let o = once(o).chain(iter).collect::<O>();
            match ret.unwrap() {
                Ok(cont) => Ok((o, cont.ok)),
                Err(e) => Err(e),
            }
        })
    }
}

/// It iterates through a sequence of parses, or does not parse if dropped.
/// A failure that does not consume input will cause the iterator to terminate, a failure that consumes it will cause the whole parser to fail.
/// The required [`ParserIterator<P>`] argument to `F` is simply an iterator that returns `P::Output`.
/// # Example
/// ```
/// use chasa::*;
/// assert_eq!(
///     any.many_with(|iter| iter.enumerate().collect()).parse_ok("abcde"),
///     Some(vec![(0,'a'),(1,'b'),(2,'c'),(3,'d'),(4,'e')])
/// );
/// assert_eq!(
///     any.many_with(|iter| iter.take(2).collect())
///         .and(char('c'))
///     .parse_ok("abcde"), Some(("ab".to_string(), 'c'))
/// );
/// assert_eq!(
///     any.and(char('a')).many_with(|iter| iter.collect()).parse_ok("baca"),
///     Some(vec![('b','a'),('c','a')])
/// );
/// assert_eq!(
///     any.and(char('a')).many_with(|iter| iter.collect::<Vec<_>>()).parse_ok("bacahh"),
///     None
/// );
/// ```
#[derive(Clone, Copy)]
pub struct ManyWith<P, F>(pub(crate) P, pub(crate) F);
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, F: FnOnce(ParserIterator<P, I, C, S, M>) -> O, O>
    ParserOnce<I, C, S, M> for ManyWith<P, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let mut ret = Some(Ok(cont));
        let iter = ParserIterator { parser: &self.0, ret: &mut ret };
        let o = self.1(iter);
        match ret.unwrap() {
            Ok(cont) => Ok((o, cont.ok)),
            Err(e) => Err(e),
        }
    }
}
impl<I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>, F: Fn(ParserIterator<P, I, C, S, M>) -> O, O>
    Parser<I, C, S, M> for ManyWith<P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let mut ret = Some(Ok(cont));
        let iter = ParserIterator { parser: &self.0, ret: &mut ret };
        let o = self.1(iter);
        match ret.unwrap() {
            Ok(cont) => Ok((o, cont.ok)),
            Err(e) => Err(e),
        }
    }
}

/// Repeat a series of parses and do not parse if dropped.
/// Failures that do not consume input terminate the iterator, while failures that consume input cause the entire parser to fail.
/// A function can return an error::Builder<M> as its return value, which also causes the entire parser to fail.
/// The mandatory [`ParserIterator<P>`] argument to `F` is simply an iterator returning `P::Output`.
pub struct ManyThen<P, F>(pub(crate) P, pub(crate) F);
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P: Parser<I, C, S, M>,
        F: FnOnce(ParserIterator<P, I, C, S, M>) -> Result<O, Eb<M>>,
        O,
    > ParserOnce<I, C, S, M> for ManyThen<P, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let pos = cont.ok.input.pos();
        let mut ret = Some(Ok(cont));
        let iter = ParserIterator { parser: &self.0, ret: &mut ret };
        let o = self.1(iter);
        match ret.unwrap() {
            Ok(cont) => match o {
                Ok(o) => Ok((o, cont.ok)),
                Err(e) => Err(e.at::<I>(cont.ok.input.index(), pos, Some(cont.ok.input.pos())).or_merge(cont.ok.err)),
            },
            Err(e) => Err(e),
        }
    }
}
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P: Parser<I, C, S, M>,
        F: Fn(ParserIterator<P, I, C, S, M>) -> Result<O, Eb<M>>,
        O,
    > Parser<I, C, S, M> for ManyThen<P, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let pos = cont.ok.input.pos();
        let mut ret = Some(Ok(cont));
        let iter = ParserIterator { parser: &self.0, ret: &mut ret };
        let o = self.1(iter);
        match ret.unwrap() {
            Ok(cont) => match o {
                Ok(o) => Ok((o, cont.ok)),
                Err(e) => Err(e.at::<I>(cont.ok.input.index(), pos, Some(cont.ok.input.pos())).or_merge(cont.ok.err)),
            },
            Err(e) => Err(e),
        }
    }
}

/// an iterator that returns `P::Output`. It can only be used locally within [`ManyWith`].
pub struct ParserIterator<'a, 'b, P: Parser<I, C, S, M>, I: Input, C, S, M: Cb> {
    parser: &'b P,
    ret: &'b mut Option<Result<ICont<'a, I, C, S, M>, LazyError<I, M>>>,
}
impl<'a, 'b, I: Input, C, S: Clone, M: Cb, P: Parser<I, C, S, M>> Iterator for ParserIterator<'a, 'b, P, I, C, S, M> {
    type Item = P::Output;
    #[inline]
    fn next(&mut self) -> Option<P::Output> {
        match self.ret.take()? {
            Err(_) => None,
            Ok(ICont { ok, config, drop }) => {
                let (input, state, cutted) = (ok.input.clone(), ok.state.clone(), ok.cutted);
                if cutted {
                    drop()
                }
                match run_drop(self.parser.to_ref(), ICont { ok, config, drop }, (input, state)) {
                    (Ok((o, ok)), _) => {
                        *self.ret = Some(Ok(ok.to_cont(config, drop)));
                        Some(o)
                    },
                    (Err(e), None) => {
                        *self.ret = Some(Err(e));
                        None
                    },
                    (Err(e), Some((input, state))) => {
                        *self.ret = Some(Ok(IOk { input, state, err: Some(e), cutted }.to_cont(config, drop)));
                        None
                    },
                }
            },
        }
    }
}

/// Arrange several parsers in a row and separate them with another parser. The result is a [`std::iter::Iterator::collect`].
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of('0'..'9').and_then(|c: char| c.to_string().parse::<isize>().map_err(message));
/// let p = d.to_ref().sep(char(','));
/// assert_eq!(p.parse_ok("1,2,3,4,5"), Some(vec![1,2,3,4,5]));
/// assert_eq!(p.parse_ok(""), Some(vec![]));
/// assert_eq!(p.and(char(',').right(d.to_ref())).parse_ok("1,2,3,4,5,6"), None);
/// assert_eq!(p.and(char(',')).parse_ok("1,2,3,4,"), Some((vec![1,2,3,4], ',')));
/// ```
pub struct Sep<P1, P2, O>(pub(crate) P1, pub(crate) P2, pub(crate) PhantomData<fn() -> O>);
impl<P1: Clone, P2: Clone, O> Clone for Sep<P1, P2, O> {
    #[inline]
    fn clone(&self) -> Self {
        Sep(self.0.clone(), self.1.clone(), PhantomData)
    }
}
impl<P1: Copy, P2: Copy, O> Copy for Sep<P1, P2, O> {}
impl<O: FromIterator<P1::Output>, I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>>
    ParserOnce<I, C, S, M> for Sep<P1, P2, O>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.run(cont)
    }
}
impl<O: FromIterator<P1::Output>, I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>>
    Parser<I, C, S, M> for Sep<P1, P2, O>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let mut ret = Some(Ok(cont));
        let iter = ParserSepIterator { parser: &self.0, sep: &self.1, is_first: true, ret: &mut ret };
        let o = iter.collect();
        match ret.unwrap() {
            Ok(cont) => Ok((o, cont.ok)),
            Err(e) => Err(e),
        }
    }
}

/// Arrange several parsers in a row and separate them with another parser. The result is a [`std::iter::Iterator::collect`]. More than one success is requested.
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of('0'..'9').and_then(|c: char| c.to_string().parse::<isize>().map_err(message));
/// let p = d.to_ref().sep1(char(','));
/// assert_eq!(p.parse_ok("1,2,3,4,5"), Some(vec![1,2,3,4,5]));
/// assert_eq!(p.parse_ok(""), None);
/// assert_eq!(p.and(char(',').right(d.to_ref())).parse_ok("1,2,3,4,5,6"), None);
/// assert_eq!(p.and(char(',')).parse_ok("1,2,3,4,"), Some((vec![1,2,3,4], ',')));
/// ```
pub struct Sep1<P1, P2, O>(pub(crate) P1, pub(crate) P2, pub(crate) PhantomData<fn() -> O>);
impl<P1: Clone, P2: Clone, O> Clone for Sep1<P1, P2, O> {
    #[inline]
    fn clone(&self) -> Self {
        Sep1(self.0.clone(), self.1.clone(), PhantomData)
    }
}
impl<P1: Copy, P2: Copy, O> Copy for Sep1<P1, P2, O> {}
impl<O: FromIterator<P1::Output>, I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>>
    ParserOnce<I, C, S, M> for Sep1<P1, P2, O>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        self.run(cont)
    }
}
impl<O: FromIterator<P1::Output>, I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>>
    Parser<I, C, S, M> for Sep1<P1, P2, O>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let ICont { ok, config, drop } = cont;
        self.0.run(ICont { ok, config, drop }).and_then(|(o, ok)| {
            let mut ret = Some(Ok(ICont { ok, config, drop }));
            let iter = ParserSepIterator { parser: &self.0, sep: &self.1, is_first: false, ret: &mut ret };
            let o = once(o).chain(iter).collect();
            match ret.unwrap() {
                Ok(cont) => Ok((o, cont.ok)),
                Err(e) => Err(e),
            }
        })
    }
}

/// Manipulate the iterator to repeat the parse while breathing on another parser, and get the result. If it is dropped, no parsing is done.
///
/// A failure that does not consume input will terminate the iterator, a failure that consumes input will cause the entire iterator to fail.
///
/// The [`ParserSepIterator<P1,P2>`] argument, required for `F`, is simply an iterator that returns `P1:Output`.
///
/// See also [`ManyWith`] and [`Sep`]
/// # Example
/// ```
/// use chasa::*;
/// let d = one_of('0'..'9').and_then(|c: char| c.to_string().parse::<isize>().map_err(message));
/// let d = d.to_ref();
/// assert_eq!(
///     d.sep_with(char(','), |iter| iter.take(2).collect())
///         .and(str(",3,4,5").to(true))
///         .parse_ok("1,2,3,4,5"),
///     Some((vec![1,2], true))
/// );
/// assert_eq!(
///     d.sep_with(char(','), |iter| iter.collect())
///         .parse_ok("1,2,3,4,"),
///     Some(vec![1,2,3,4])
/// );
/// ```
pub struct SepWith<P1, P2, F>(pub(crate) P1, pub(crate) P2, pub(crate) F);
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: FnOnce(ParserSepIterator<P1, P2, I, C, S, M>) -> O,
        O,
    > ParserOnce<I, C, S, M> for SepWith<P1, P2, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let mut ret = Some(Ok(cont));
        let iter = ParserSepIterator { parser: &self.0, sep: &self.1, is_first: true, ret: &mut ret };
        let o = self.2(iter);
        match ret.unwrap() {
            Ok(cont) => Ok((o, cont.ok)),
            Err(e) => Err(e),
        }
    }
}
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: Fn(ParserSepIterator<P1, P2, I, C, S, M>) -> O,
        O,
    > Parser<I, C, S, M> for SepWith<P1, P2, F>
{
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let mut ret = Some(Ok(cont));
        let iter = ParserSepIterator { parser: &self.0, sep: &self.1, is_first: true, ret: &mut ret };
        let o = self.2(iter);
        match ret.unwrap() {
            Ok(cont) => Ok((o, cont.ok)),
            Err(e) => Err(e),
        }
    }
}

pub struct SepThen<P1, P2, F>(pub(crate) P1, pub(crate) P2, pub(crate) F);
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: FnOnce(ParserSepIterator<P1, P2, I, C, S, M>) -> Result<O, Eb<M>>,
        O,
    > ParserOnce<I, C, S, M> for SepThen<P1, P2, F>
{
    type Output = O;
    #[inline]
    fn run_once(self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let pos = cont.ok.input.pos();
        let mut ret = Some(Ok(cont));
        let iter = ParserSepIterator { parser: &self.0, sep: &self.1, is_first: true, ret: &mut ret };
        let o = self.2(iter);
        match ret.unwrap() {
            Ok(cont) => match o {
                Ok(o) => Ok((o, cont.ok)),
                Err(e) => Err(e.at::<I>(cont.ok.input.index(), pos, Some(cont.ok.input.pos())).or_merge(cont.ok.err)),
            },
            Err(e) => Err(e),
        }
    }
}
impl<
        I: Input,
        C,
        S: Clone,
        M: Cb,
        P1: Parser<I, C, S, M>,
        P2: Parser<I, C, S, M>,
        F: Fn(ParserSepIterator<P1, P2, I, C, S, M>) -> Result<O, Eb<M>>,
        O,
    > Parser<I, C, S, M> for SepThen<P1, P2, F>
{
    #[inline]
    fn run(&self, cont: ICont<I, C, S, M>) -> IResult<O, I, S, M> {
        let pos = cont.ok.input.pos();
        let mut ret = Some(Ok(cont));
        let iter = ParserSepIterator { parser: &self.0, sep: &self.1, is_first: true, ret: &mut ret };
        let o = self.2(iter);
        match ret.unwrap() {
            Ok(cont) => match o {
                Ok(o) => Ok((o, cont.ok)),
                Err(e) => Err(e.at::<I>(cont.ok.input.index(), pos, Some(cont.ok.input.pos())).or_merge(cont.ok.err)),
            },
            Err(e) => Err(e),
        }
    }
}

/// an iterator that returns `P1::Output`. It can only be used locally within [`SepWith`].
pub struct ParserSepIterator<'a, 'b, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>, I: Input, C, S, M: Cb> {
    parser: &'b P1,
    sep: &'b P2,
    is_first: bool,
    ret: &'b mut Option<Result<ICont<'a, I, C, S, M>, LazyError<I, M>>>,
}
impl<'a, 'b, I: Input, C, S: Clone, M: Cb, P1: Parser<I, C, S, M>, P2: Parser<I, C, S, M>> Iterator
    for ParserSepIterator<'a, 'b, P1, P2, I, C, S, M>
{
    type Item = P1::Output;
    #[inline]
    fn next(&mut self) -> Option<P1::Output> {
        let ICont { ok, config, drop } = match self.ret.take()? {
            Ok(cont) => cont,
            Err(_) => None?,
        };
        let (input, state) = (ok.input.clone(), ok.state.clone());
        let ok = if self.is_first {
            self.is_first = false;
            ok
        } else {
            let cutted = ok.cutted;
            if cutted {
                drop()
            }
            match run_drop(self.sep.to_ref(), ICont { ok, config, drop }, ()) {
                (Err(e), None) => {
                    *self.ret = Some(Err(e));
                    None?
                },
                (Err(e), Some(())) => {
                    *self.ret = Some(Ok(IOk { input, state, err: Some(e), cutted }.to_cont(config, drop)));
                    return None;
                },
                (Ok((_, ok)), _) => ok,
            }
        };
        let cutted = ok.cutted;
        if cutted {
            drop()
        }
        match run_drop(self.parser.to_ref(), ICont { ok, config, drop }, (input, state)) {
            (Ok((o, ok)), _) => {
                *self.ret = Some(Ok(ok.to_cont(config, drop)));
                Some(o)
            },
            (Err(e), None) => {
                *self.ret = Some(Err(e));
                None
            },
            (Err(e), Some((input, state))) => {
                *self.ret = Some(Ok(IOk { input, state, err: Some(e), cutted }.to_cont(config, drop)));
                None
            },
        }
    }
}
