use tera::{Tera, Context};
use tokio::fs;
use tokio::io::{self, AsyncWriteExt};

use std::fmt;

use crate::core::{prompt, Config};

pub async fn run(config: &Config<'_>) -> Result<(), RunError> {
    let changelog = prompt_run().await.map_err(RunError::Prompt)?;

    create(config, &changelog).await.map_err(RunError::Template)?;

    Ok(())
}

struct Changelog {
    number: u64,
    title:  String,
    points: Option<Vec<String>>,
}

#[derive(Debug)]
pub enum RunError {
    Prompt(io::Error),
    Template(tera::Error),
}

impl fmt::Display for RunError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        use RunError::*;

        match self {
            Prompt(err) => write!(f, "{}", err),
            Template(err) => write!(f, "{}", err),
        }
    }
}

impl std::error::Error for RunError {}

async fn prompt_run() -> Result<Changelog, io::Error> {
    let mut failed = false;

    let number = loop {
        let txt = match failed {
            false => "Ticket Number: ",
            true => "Ticket Number (must be a number): ",
        };

        let raw = prompt(txt).await?;

        match raw.parse::<u64>() {
            Ok(number) => break number,
            Err(_err) => failed = true,
        }
    };

    let title = prompt("Ticket Name: ").await?;

    Ok(Changelog {
        number,
        title,
        points: None,
    })
}

const TEMPLATE: &str = 
r#"- [{{ number }}](https://vizerapp.atlassian.net/browse/VIZ-{{ number }}) - {{ title }}
{%- if points -%}
    {%- for point in points -%}
    - {{ point }}
    {%- endfor -%}
{%- endif %}
"#;

async fn create(config: &Config<'_>, changelog: &Changelog) -> Result<(), tera::Error> {
    let mut tmpl = Tera::default();

    let mut context = Context::new();

    context.insert("number", &changelog.number);
    context.insert("title", &changelog.title);
    context.insert("points", &changelog.points);

    let output = tmpl.render_str(
        TEMPLATE,
        &context,
    )?;

    let path = format!("{}/{}", &config.changelogs_folder, filename(changelog));
    let mut file = fs::File::create(path).await?;
    file.write_all(output.as_bytes()).await?;

    Ok(())
}

fn filename(changelog: &Changelog) -> String {
    let title = changelog.title.to_lowercase();

    let r = regex::Regex::new(r"[^a-zA-Z0-9\s]").unwrap();

    let title = r.replace_all(&title, "$a");

    let spaces = regex::Regex::new(r"\s+").unwrap();

    format!("{}-{}.md", changelog.number, spaces.replace_all(&title, "-"))
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_filename_simple() {
        let changelog = Changelog {
            number: 0,
            title: "This is a thing".to_owned(),
            points: None,
        };

        assert_eq!(filename(&changelog), "0-this-is-a-thing.md");
    }

    #[test]
    fn test_filename_bang() {
        let changelog = Changelog {
            number: 0,
            title: "This is a thing!".to_owned(),
            points: None,
        };

        assert_eq!(filename(&changelog), "0-this-is-a-thing.md");
    }

    #[test]
    fn test_filename_number() {
        let changelog = Changelog {
            number: 0,
            title: "I have 2 of them!".to_owned(),
            points: None,
        };

        assert_eq!(filename(&changelog), "0-i-have-2-of-them.md");
    }

    #[test]
    fn test_filename_double_spaces() {
        let changelog = Changelog {
            number: 0,
            title: "I have 2  of them!".to_owned(),
            points: None,
        };

        assert_eq!(filename(&changelog), "0-i-have-2-of-them.md");
    }
}
