use druid::*;
use druid::widget::*;
use log::*;
use std::time::Instant;
use crate::object;

use crate::state::*;


struct CanvasController {
    drawing: bool,
    pmouse: Option<Point>,
}

impl CanvasController {
    fn new() -> CanvasController {
        CanvasController {
            drawing: false,
            pmouse: None,
        }
    }
}

impl<W: Widget<AppData>> Controller<AppData, W> for CanvasController {
    fn event(&mut self, _child: &mut W, ctx: &mut EventCtx, event: &Event, data: &mut AppData, _env: &Env) {
        match event {
            Event::WindowConnected => ctx.request_paint(),
            Event::MouseDown(_) => {
                if !self.drawing {
                    info!("draw started");
                    self.drawing = true;
                    self.pmouse = None;
                }
            }
            Event::MouseUp(_) => {
                if self.drawing {
                    info!("draw ended");
                    self.pmouse = None;
                    self.drawing = false;
                }
            }
            Event::MouseMove(info) => {
                if self.drawing {
                    let to = info.pos;
                    if let Some(from) = self.pmouse {
                        // we can draw a line
                        dbg!(from, to);
                        data.objects.borrow_mut().push(Box::new(object::Line::new(from.clone(), to.clone())));
                    }

                    self.pmouse = Some(to);
                    ctx.request_paint(); // only in this case do we need to repaint
                }
            }
            _ => ()
        }
    }
}

fn paint_function(ctx: &mut PaintCtx, state: &AppData, env: &Env){
    let start_time = Instant::now();
    info!("redrawing canvas");
    let bounds = ctx.size().to_rect();
    ctx.fill(bounds, &Color::grey8(255));

    for obj in state.objects.borrow().iter() {
        obj.render(ctx, env);
    }

    info!("redrawn {} objects in {:.2?}", state.objects.borrow().len(), start_time.elapsed());
}

pub fn get_canvas_widget() -> impl Widget<AppData> {
    ControllerHost::new(Painter::new(paint_function), CanvasController::new())
}