use std::str::FromStr;
use std::convert::Infallible;
use std::net::SocketAddr;
use hyper::{Body, Request, Response, Server};
use routerify::prelude::*;
use routerify::{Router, RouterService};
use chainseeker_server::*;
use chainseeker_server::bitcoin::*;

const COIN: &str = "integration";

#[derive(Debug, Clone)]
struct MockBitcoinCore {
    //blocks: Vec<Block>,
}

impl Default for MockBitcoinCore {
    fn default() -> Self {
        Self::new()
    }
}

impl MockBitcoinCore {
    pub fn new() -> Self {
        Self {
            blocks: fixtures::regtest_blocks(),
        }
    }
    /// `/chaininfo.json` endpoint.
    async fn chaininfo_handler(req: Request<Body>) -> Result<Response<Body>, Infallible> {
        let mock = req.data::<MockBitcoinCore>().unwrap();
        //Ok(HttpServer::json(bitcoin_rest::ChainInfo {
        //}, false))
        Ok(HttpServer::not_found("Page not found."))
    }
    pub async fn run(&self) {
        let addr = SocketAddr::from((
            "127.0.0.1".parse::<std::net::IpAddr>().expect("Failed to parse HTTP IP address."),
            18443));
        let router = Router::builder()
            .data((*self).clone())
            .get("/chaininfo.json", Self::chaininfo_handler)
            .any(|_req| async {
                Ok(HttpServer::not_found("invalid URL."))
            })
            .err_handler_with_info(|err, _| async move {
                eprintln!("{}", err);
                HttpServer::internal_error(&format!("Something went wrong: {}", err))
            })
            .build()
            .unwrap();
        let service = RouterService::new(router).unwrap();
        let server = Server::bind(&addr).serve(service);
        if let Err(e) = server.await {
            panic!("MockBitcoinCore failed: {}", e);
        }
    }
}

/// Remove `~/.chainseeker/integration` directory
fn cleanup() {
    let path = format!("{}/{}", data_dir(), COIN);
    if std::path::Path::new(&path).exists() {
        std::fs::remove_dir_all(&path).unwrap();
    }
}

#[tokio::test]
async fn integration() {
    cleanup();
    // Load config.
    let mut config = config_example("rbtc");
    config.genesis_block_hash = BlockHash::from_str("0f9188f13cb7b2c71f2a335e3a4fc328bf5beb436012afca590b1a11466e2206").unwrap();
    // Launch MockBitcoinCore.
    {
        tokio::spawn(async move {
            let mock: MockBitcoinCore = Default::default();
            mock.run().await;
        });
    }
    // Launch main.
    {
        let config = config.clone();
        tokio::spawn(async move {
            main(COIN, &config).await
        });
    }
    let client = reqwest::Client::new();
    // Wait until sync finishes.
    loop {
        let res = client.get(format!("http://{}:{}/api/v1/status", config.http_ip, config.http_port)).send().await;
        if let Ok(res) = res {
            let status = res.json::<RestStatus>().await.unwrap();
            println!("Synced blocks: {}", status.blocks);
            if status.blocks >= 102 {
                break;
            }
        }
        tokio::time::sleep(std::time::Duration::from_millis(100)).await;
    }
    cleanup();
}
