/*!
# Chain-LINQ

This crate is an implementation of .NET's LINQ query syntax in rust as a declarative macro. The declarative macro is generated by another declarative macro, specifically the
branching_parser from my other crate, big_mac. 

Available statements closely mirror standard LINQ operations; notably join is missing, as I was unable to find equivalent functionality from iterators.

Generally, each statement maps to an iterator method; here is a list of them, with a description and equivalent method if present:
- from # in #: selects and names variables from a collection. Maps to into_iter().
- select #: ends a linq query and returns an iterator. Maps to map().
- select # into #: partially ends a linq query and puts an iterator into a range variable. Creates a continuation.
- where #: selects only elements that match a particular criteria. Maps to filter().
- let # = #: creates a range element. Does not map to a method. 
- collect #: calls .collect().
- collect # as #: calls .collect with a generic type. Follows select syntax but the as parameter is the destination type.

**REQUIRES [ITERTOOLS](https://docs.rs/itertools/0.10.1/itertools/):**
- orderby #: sorts iterator ascending by a criteria. Maps to unstable_sort_by_key().
- orderby # ascending: sorts iterator ascending by a criteria. Maps to unstable_sort_by_key().
- orderby # descending: sorts iterator descending by a criteria. Maps to unstable_sort_by_key().rev().
- group # by #: groups elements into groups based on some criteria and returns the result. Maps to group_by(). 
- group # by # into #: groups elements into groups based on some criteria, and then creates a continuation.. Maps to group_by().

For more explanation of how LINQ works, check Microsoft's docs [here](https://docs.microsoft.com/en-us/dotnet/csharp/programming-guide/concepts/linq/)

Also useful: [Keyword breakdowns](https://docs.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/query-keywords)

## Examples

```
use chain_linq::linq;

let xs = [(3, 1), (2, 2), (1, 3)];

let x = linq!(
    from (x, y) in xs
    let z = x + y
    select z into zs
    from z in zs
    select z * 2
);
```

```
use chain_linq::linq;

let xss = [vec!(27, 13, 12), vec!(69), vec!(76, 7, 420)];

let x = linq!(
    from xs in xss
    group xs by xs.len() into gs
    from iter in gs
    from x in iter
    collect x + 1 as Vec<i32>
);
```

```
use chain_linq::linq;

let xss = [vec!(27, 13, 12), vec!(69), vec!(76, 7, 420)];

let x = linq!(
    from xs in xss
    from x in xs.into_iter().rev()
    collect x as Vec<i32>
);
```

*/

// Patterns:
// from # in #
// where #
// select #
// select # into #
// orderby #
// orderby # ascending/descending
// group # by #
// group # by # into #
// let # = #

mod macros {
    #[macro_export]
    macro_rules! linq_impl {
        // Archetype: {name of working variable} {cumulative previous expression} {statements to add to closures as prefixes} {pattern to match, remaining tokens}
        // Short names: {var} {prev} {prefix} {___, toks}

        // From clauses
        (from {$var:pat} in {$base:expr}, $($toks:tt)*) => ({
            $crate::linq_impl!{ 
                {$var}
                {$base.into_iter()}
                {}
                {$($toks)*}
            } 
        });

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {from {$new_var:pat} in {$new_base:expr}, $($toks:tt)*}
        ) => {
            $crate::linq_impl!{
                {$new_var}
                {$prev.flat_map(|$var| { $($prefix)* $new_base.into_iter() })}
                {}
                {$($toks)*}
            }
        };

        // Where clause
        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {where {$cond:expr}, $($toks:tt)*}
        ) => {
            $crate::linq_impl!{
                {$var}
                {$prev.filter(|$var| { $($prefix)* $cond })}
                {}
                {$($toks)*}
            }
        };

        // Select clauses
        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {select {$result:expr} into {$new_base:ident}, $($toks:tt)*}
        ) => {
            let $new_base = $prev.map(|$var| { $($prefix)* $result });

            $crate::linq_impl!{ $($toks)* }
        };

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {select {$result:expr},}
        ) => {
            use ::std::iter::*;

            $prev.map(|$var| { $($prefix)* $result })
        };

        // Orderby clauses
        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {orderby {$key:expr}, $($toks:tt)*}
        ) => {
            $crate::linq_impl!{
                {$var}
                {$prev.sorted_unstable_by_key(|$var| { $($prefix)* $key })}
                {}
                {$($toks)*}
            }
        };

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {orderby {$key:expr} ascending, $($toks:tt)*}
        ) => {
            $crate::linq_impl!{
                {$var}
                {$prev.sorted_unstable_by_key(|$var| { $($prefix)* $key })}
                {}
                {$($toks)*}
            }
        };

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {orderby {$key:expr} descending, $($toks:tt)*}
        ) => {
            $crate::linq_impl!{
                {$var}
                {$prev.sorted_unstable_by_key(|$var| { $($prefix)* $key }).rev()}
                {}
                {$($toks)*}
            }
        };

        // Group clauses
        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {group {$grouped:ident} by {$grouper:expr} into {$group:ident}, $($toks:tt)*}
        ) => {
            {
                use ::itertools::*;

                let $group = $prev.group_by(|$grouped| { $($prefix)* $grouper }).into_iter().map(|g| g.1)
                
                $crate::linq_impl!{ $($toks)* }
            }
            
        };

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {group {$grouped:ident} by {$grouper:expr}, $($toks:tt)*}
        ) => {
            {
                use ::std::iter::*;
                use ::itertools::*;
                
                $prev.group_by(|$grouped| { $($prefix)* $grouper })
            }
        };

        // Let clause
        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {let {$new_var:pat} = {$query:expr}, $($toks:tt)*}
        ) => {
            $crate::linq_impl!{
                {$var}
                {$prev}
                {$($prefix)* let $new_var = $query }
                {$($toks)*}
            }
        };

        // Collect clause
        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {collect {$result:expr},}
        ) => {
            use ::std::iter::*;

            $prev.map(|$var| { $($prefix)* $result }).collect()
        };

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {collect {$result:expr} as {$collection:ty},}
        ) => {
            use ::std::iter::*;

            $prev.map(|$var| { $($prefix)* $result }).collect::<$collection>()
        };

        (
            {$var:pat}
            {$prev:expr}
            {$($prefix:stmt)*}
            {}
        ) => {
            compile_error!("Unfinished linq query!");
        };
    }

    big_mac::branching_parser!(
        @unroll
        chain_linq;
        linq
        linq_parser
        linq_filter
        chain_linq::linq_impl;
        {
            from
            {
                #
                {
                    in
                    {
                        # {}
                    }
                }
            }
        }
        {
            where
            {
                # {}
            }
        }
        {
            select 
            {
                #
                {
                    into 
                    {
                        # {}
                    }
                }
                {}
            }
        }
        {
            orderby
            {
                # 
                {}
                {
                    ascending
                }
                {
                    descending
                }
            }
        }
        {
            group
            {
                #
                {
                    by
                    {
                        #
                        {}
                        {
                            into 
                            {
                                # {}
                            }
                        }
                    }
                }
            }
        }
        {
            let
            {
                #
                {
                    =
                    {
                        # {}
                    }
                }
            }
        }
        {
            collect
            {
                #
                {}
                {
                    as
                    {
                        # {}
                    }
                }
            }
        }
    );

    #[allow(unused_imports)]
    pub(crate) use {linq, linq_parser, linq_filter, linq_impl};
}
