//! This module helps implement your own CodeGame WebSocket endpoint.
//!
//! For examples see [the official games](https://github.com/orgs/code-game-project/teams/games/repositories)

use crate::events::EventWrapper;
use crate::server_communications::GameServerCommunications;
use actix::{Actor, Addr, Handler, StreamHandler};
use actix_web_actors::ws;
use std::time::Duration;

// constants to be used in each implementation of `Socket`
/// default websocket heartbeat interval
pub const HEARTBEAT_INTERVAL: Duration = Duration::from_secs(5);
/// default websocket heartbeat timeout
pub const CLIENT_TIMEOUT: Duration = Duration::from_secs(10);
/// constant id to be used when the socket is sending the client it's connected to something
pub const SOCKET_SELF_ID: &str = "self";

/// A generic game socket that can be implemented and extended to fit any game's needs
pub trait Socket<G, S, R>
where
  Self: Actor,
  Self: StreamHandler<Result<ws::Message, ws::ProtocolError>>,
  Self: Handler<EventWrapper<S>>,
  Self: GameServerCommunications<G, R>,
  G: Actor,
{
  /// Creates and returns new instance of `Socket`
  fn new(gs: Addr<G>) -> Self;
  /// Starts the websocket heartbeat
  fn hb(&self, ctx: &mut <Self as Actor>::Context)
  where
    Self: actix::Actor;
  /// Wraps a receivable game event `R` (from self, to self) and emits it
  ///
  /// This can be used for events such as errors that come from the socket and go
  /// to the associated client
  fn self_emit(ctx: &mut <Self as Actor>::Context, event: S)
  where
    Self: actix::Actor;
  /// Serializes and sends a receiveable game event `R` to `ctx`
  fn emit(ctx: &mut <Self as Actor>::Context, wrapped: &EventWrapper<S>)
  where
    Self: actix::Actor;
  /// Handles a receivable game event `R` sent to the `Socket` by the client
  fn handle_receiveable(&mut self, ctx: &mut <Self as Actor>::Context, event: R)
  where
    Self: actix::Actor;
  /// Handles a sendable game event `S` sent to the `Socket` by the game server `G` or another `Socket`
  fn handle_sendable(&mut self, ctx: &mut <Self as Actor>::Context, wrapped: EventWrapper<S>)
  where
    Self: actix::Actor;
}
