//! This module provides a trait that helps `socket::Server` and `server::GameServer` communicate

use crate::events::{EventTarget, EventWrapper, Join};
use actix::{Actor, Addr, Handler};

/// A trait that helps implement communication functions between the `socket::Socket` and the `server::GameServer`
/// 
/// This does is required by the `socket::Socket` trait to keep the code DRY.
pub trait GameServerCommunications<G, S>
where
  G: Actor,
  Self: Actor,
{
  /// Sends an event `S` to `EventTarget` using the `server::GameServer` `G`
  fn send(&self, to: EventTarget, event: S)
  where
    G: Handler<EventWrapper<S>>;
  /// Registers `Self` with the `server::GameServer` `G`
  /// 
  /// This allows other players and the `server::GameServer` `G` send `Self` events
  fn connect(&self, addr: Addr<Self>);
  /// Disconnects from the `server::GameServer` `G`
  fn disconnect(&self);
  /// Joins a game
  fn join(&self, event: Join);
  /// Leaves a game
  fn leave(&self);
}
