//! This module helps implement a `GameServer` for the WebSocket endpoints to communicate via.
//!
//! For examples see [the official games](https://github.com/orgs/code-game-project/teams/games/repositories)

use crate::events::{Connect, Disconnect, EventWrapper, Join, Leave};
use actix::{Actor, Addr, Handler};
use std::collections::{HashMap, HashSet};

/// A type to indicate that this is not just any string but a `GameID`
pub type GameID = String;
/// A type to indicate that this is not just any string but a `SocketID`
pub type SocketID = String;
/// A `HashMap` of `SocketID`s and their corresponding `Addr`s
pub type SocketsMap<C> = HashMap<SocketID, Addr<C>>;
/// A `HashMap` of `GameID`s and the playing `SocketID`s
pub type GamesMap = HashMap<GameID, HashSet<SocketID>>;

/// A constant id to be used as as the `from` property in `events::EventWrapper` when the server is the sender
pub const SERVER_ID: &str = "server";

/// A generic game server that can be implemented and extended to fit any game's needs
///
/// A struct that implements `GameServer` should have a bare minimum of two fields:
/// `sockets: SocketsMap<SomeSocket>` and `games: GamesMap`
pub trait GameServer<C, S>
where
	Self: Actor,
	Self: Handler<EventWrapper<S>>,
	Self: Handler<EventWrapper<Connect<C>>>,
	Self: Handler<EventWrapper<Disconnect>>,
	Self: Handler<EventWrapper<Join>>,
	Self: Handler<EventWrapper<Leave>>,
	C: Actor,
{
	/// Creates a new instance of `GameServer`
	fn new() -> Self;
	/// Inserts an `Addr` and its `id` into the `SocketsMap`
	fn connect(&mut self, id: SocketID, addr: Addr<C>);
	/// Removes an `Addr` from the `SocketsMap`
	fn disconnect(&mut self, id: &SocketID);
	/// Creates a new game by adding a new game key to the `GamesMap`
	fn new_game(&mut self, game_id: GameID);
	/// Ends and removes a game by removing it from the `GamesMap`
	fn end_game(&mut self, game_id: &GameID);
	/// Associates a `SocketID` with a game in the `GamesMap`
	fn join_game(&mut self, id: SocketID, event: Join);
	/// Disassociates a `SocketID` with a game in the `GamesMap`
	fn leave_game(&mut self, id: &SocketID, event: Leave);
	/// Matches the `event.to: EventTarget` and emits the event to the correct destination
	fn emit(&self, event: EventWrapper<S>);
	/// Emits an event to all sockets in a particular game
	fn emit_to_game(&self, game_id: &GameID, event: EventWrapper<S>);
	/// Emits an event to a particular socket
	fn emit_to_socket(&self, id: &SocketID, event: EventWrapper<S>);
}
