//! This module contains standard event structs, the `EventWrapper` struct and
//! the `EventTarget` enum used to communicate between the different `Actor`s.

use crate::server::{GameID, SocketID};
use actix::{Message, Addr};
use serde::{Deserialize, Serialize};

/// A wrapper for an event `E` so that `to` and `from` properties can be added
#[derive(Message, Debug, Clone, Serialize, Deserialize)]
#[rtype(result = "()")]
pub struct EventWrapper<E> {
  /// The id of socket that sent/triggered the event
  pub from: SocketID,
  /// The target of the event / which sockets receive the event
  pub to: EventTarget,
  /// The actual event
  pub event: E,
}

/// Target of an event, meant to be used as `to` property of `EventWrapper`
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum EventTarget {
  /// Specifies the target of the event as the socket that it comes from
  /// 
  /// This can be used when emitting errors or other private events that the other players should __not__ know about
  SocketSelf,
  /// Specifies the target of the event as the `server::GameServer`
  /// 
  /// This can be used when dealing with events that make the `server::GameServer` do something more than just relay the event
  Server,
  /// Specifies the target of the event as another `socket::Socket` connected to the `server::GameServer`
  /// 
  /// This can be used in games where there is a need to privately coordinate
  Socket(SocketID),
  /// Specifies the target of the event as all the `socket::Socket`s in associated with a `GameID`
  /// 
  /// This is the type that will be used most in most games
  Game(GameID),
}

// A few events that should be part of every `Socket`'s events
/// A struct that contains a `reason` field and can be serialized and sent to a client
#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct Error {
  /// The reason the error occurred
  /// 
  /// Reasons are allowed to be technical, since the users are mostly (if not all) developers
  /// and will benefit from error messages that actually say what is wrong, but should also
  /// not be too technical, because many users are not experts at rust or programming in general.
  pub reason: String,
}
/// A struct that contains the address of a `socket::Socket`
/// 
/// This event is used to transmit a `socket::Socket`s `Addr` to the `server::GameServer`
#[derive(Message, Debug, Clone)]
#[rtype(result = "()")]
pub struct Connect<C> where C: actix::Actor {
  /// The `Addr` of an `Actor`
  pub addr: Addr<C>
}
/// A struct with no data
/// 
/// This event is used to tell the `server::GameServer` that the client is disconnecting from the `server::GameServer`
#[derive(Message, Serialize, Deserialize, Debug, Clone)]
#[rtype(result = "()")]
pub struct Disconnect;
/// A struct that contains the `username` of the sender and the `GameID` of the game that they want to join
/// 
/// This event is used by a `socket::Socket` to join a game
#[derive(Message, Serialize, Deserialize, Debug, Clone)]
#[rtype(result = "()")]
pub struct Join {
  /// `GameID` of the game the game that is to be joined
  pub game_id: GameID,
  /// `username` of the `socket::Socket` joining
  pub username: String,
}
/// A struct that contains the `username` of the sender and the `GameID` of the game that they want to leave
/// 
/// This event is used by a `socket::Socket` to leave the current game
#[derive(Message, Serialize, Deserialize, Debug, Clone)]
#[rtype(result = "()")]
pub struct Leave {
  /// `GameID` of the game the `socket::Socket` is part of
  pub game_id: GameID,
  /// `username` of the `socket::Socket` leaving
  pub username: String,
}
/// A struct that contains a `reason`
/// 
/// This event can be sent to any `socket::Socket` to terminate its connection to the entire server.
/// The event along with the `reason` will be sent to the client just before termination so that they
/// know why they were kicked.
#[derive(Message, Serialize, Deserialize, Debug, Clone)]
#[rtype(result = "()")]
pub struct Kick {
  /// The reason for kicking this player
  pub reason: String,
}
