use colored::*;
use serde::Serialize;

use crate::commands::tracker::StatusContext;
use crate::rules::{EvaluationType, path_value, Status};
use crate::rules::path_value::Path;
use crate::rules::values::CmpOperator;
use std::fmt::Debug;
use std::io::Write;
use std::collections::{HashMap, HashSet};
use std::convert::TryInto;
use regex::Regex;
use lazy_static::*;

#[derive(Debug, PartialEq, Serialize)]
pub(super) struct Comparison {
    operator: CmpOperator,
    not_operator_exists: bool,
}

impl From<(CmpOperator, bool)> for Comparison {
    fn from(input: (CmpOperator, bool)) -> Self {
        Comparison {
            operator: input.0,
            not_operator_exists: input.1
        }
    }
}

#[derive(Debug, PartialEq, Serialize)]
pub(super) struct NameInfo<'a> {
    pub(super) rule: &'a str,
    pub(super) path: String,
    pub(super) provided: Option<serde_json::Value>,
    pub(super) expected: Option<serde_json::Value>,
    pub(super) comparison: Option<Comparison>,
    pub(super) message: String,
}

pub(super) trait GenericReporter: Debug {
    fn report(&self,
              writer: &mut dyn Write,
              rules_file_name: &str,
              data_file_name: &str,
              failed: HashMap<String, Vec<NameInfo<'_>>>,
              passed: HashSet<String>,
              skipped:HashSet<String>,
              longest_rule_len: usize) -> crate::rules::Result<()>;
}

#[derive(Debug)]
pub(super) enum StructureType {
    JSON,
    YAML
}

#[derive(Debug)]
pub(super) struct StructuredSummary {
    hierarchy_type: StructureType
}

impl StructuredSummary {
    pub(super) fn new(hierarchy_type: StructureType) -> Self {
        StructuredSummary {
            hierarchy_type
        }
    }
}

#[derive(Debug, Serialize)]
struct DataOutput<'a> {
    data_from: &'a str,
    rules_from: &'a str,
    not_compliant: HashMap<String, Vec<NameInfo<'a>>>,
    not_applicable: HashSet<String>,
    compliant: HashSet<String>,
}

impl GenericReporter for StructuredSummary {
    fn report(&self,
              writer: &mut dyn Write,
              rules_file_name: &str,
              data_file_name: &str,
              failed: HashMap<String, Vec<NameInfo<'_>>>,
              passed: HashSet<String>,
              skipped: HashSet<String>, longest_rule_len: usize) -> crate::rules::Result<()>
    {
        let value = DataOutput {
            rules_from: rules_file_name,
            data_from: data_file_name,
            not_compliant: failed,
            compliant: passed,
            not_applicable: skipped
        };

        match &self.hierarchy_type {
            StructureType::JSON => writeln!(writer, "{}", serde_json::to_string(&value)?),
            StructureType::YAML => writeln!(writer, "{}", serde_yaml::to_string(&value)?),
        };
        Ok(())
    }
}

lazy_static! {
    static ref PATH_FROM_MSG: Regex = Regex::new(r"path\s+=\s+(?P<path>[^ ]+)").ok().unwrap();
}

pub(super) fn extract_name_info<'a>(rule_name: &'a str,
                                    each_failing_clause: &StatusContext) -> crate::rules::Result<NameInfo<'a>> {
    if each_failing_clause.from.is_some() {
        let value = each_failing_clause.from.as_ref().unwrap();
        let (path, from): (String, serde_json::Value) = value.try_into()?;
        Ok(NameInfo {
            rule: rule_name,
            path,
            provided: Some(from),
            expected: match &each_failing_clause.to {
                Some(to) => {
                    let (_, val): (String, serde_json::Value) = to.try_into()?;
                    Some(val)
                },
                None => None,
            },
            comparison: match each_failing_clause.comparator {
                Some(input) => Some(input.into()),
                None => None,
            },
            message: each_failing_clause.msg.as_ref().map_or(
                "".to_string(), |e| if !e.contains("DEFAULT") { e.clone() } else { "".to_string() })
        })
    }
    else {
        //
        // This is crappy, but we are going to extract information from the retrieval error message
        // see path_value.rs for retrieval error messages.
        // TODO merge the query interface to retrieve partial results along with errored one ones and then
        //      change this logic based on the reporting changes. Today we bail out for the first
        //      retrieval error, fast fail semantics
        //

        //
        // No from is how we indicate retrieval errors.
        //
        let (path, message) = each_failing_clause.msg.as_ref().map_or(("".to_string(), "".to_string()), |msg| {
            match PATH_FROM_MSG.captures(msg) {
                Some(cap) => (cap["path"].to_string(), msg.clone()),
                None => ("".to_string(), msg.clone())
            }
        });

        Ok(NameInfo {
            rule: rule_name,
            path,
            provided: None,
            expected: None,
            comparison: None,
            message
        })
    }
}

pub(super) fn colored_string(status: Option<Status>) -> ColoredString {
    let status = match status {
        Some(s) => s,
        None => Status::SKIP,
    };
    match status {
        Status::PASS => "PASS".green(),
        Status::FAIL => "FAIL".red().bold(),
        Status::SKIP => "SKIP".yellow().bold(),
    }
}

pub(super) fn find_all_failing_clauses(context: &StatusContext) -> Vec<&StatusContext> {
    let mut failed = Vec::with_capacity(context.children.len());
    for each in &context.children {
        if each.status.map_or(false, |s| s == Status::FAIL) {
            match each.eval_type {
                EvaluationType::Clause |
                EvaluationType::BlockClause => {
                    failed.push(each);
                    if each.eval_type == EvaluationType::BlockClause {
                        failed.extend(find_all_failing_clauses(each));
                    }
                },

                EvaluationType::Filter |
                EvaluationType::Condition => {
                    continue;
                },

                _ => failed.extend(find_all_failing_clauses(each))
            }
        }
    }
    failed
}

pub(super) fn print_compliant_skipped_info(writer: &mut dyn Write,
                                           passed: &HashSet<String>,
                                           skipped: &HashSet<String>,
                                           rules_file_name: &str,
                                           data_file_name: &str) -> crate::rules::Result<()> {
    if !passed.is_empty() {
        writeln!(writer, "--")?;
    }
    for pass in passed {
        writeln!(writer, "Rule [{}/{}] is compliant for template [{}]", rules_file_name, pass, data_file_name)?;
    }
    if !skipped.is_empty() {
        writeln!(writer, "--")?;
    }
    for skip in skipped {
        writeln!(writer, "Rule [{}/{}] is not applicable for template [{}]", rules_file_name, skip, data_file_name)?;
    }
    Ok(())
}

pub(super) fn print_name_info<R, U, B>(
    writer: &mut dyn Write,
    info: &[NameInfo<'_>],
    longest_rule_len: usize,
    rules_file_name: &str,
    data_file_name: &str,
    retrieval_error: R,
    unary_message: U,
    binary_message: B) -> crate::rules::Result<()>
    where R: Fn(&str, &str, &NameInfo<'_>) -> crate::rules::Result<String>,
          U: Fn(&str, &str, &str, &NameInfo<'_>) -> crate::rules::Result<String>,
          B: Fn(&str, &str, &str, &NameInfo<'_>) -> crate::rules::Result<String>
{
    for each in info {
        let (cmp, not) = match &each.comparison {
            Some(cmp) => (Some(cmp.operator), cmp.not_operator_exists),
            None => (None, false)
        };
        // CFN = Resource [<name>] was not compliant with [<rule-name>] for property [<path>] because provided value [<value>] did not match expected value [<value>]. Error Message [<msg>]
        // General = Violation of [<rule-name>] for property [<path>] because provided value [<value>] did not match expected value [<value>]. Error Message [<msg>]
        // EQUALS failed at property path Properties.Encrypted because provided value [false] did not match with expected value [true].
        match cmp {
            None => {
                // Block Clause retrieval error
                writeln!(writer, "{}", retrieval_error(rules_file_name, data_file_name, each)?)?;
            },

            Some(cmp) => {
                if cmp.is_unary() {
                    writeln!(writer, "{}",
                         unary_message(
                             rules_file_name,
                             data_file_name,
                             match cmp {
                                 CmpOperator::Exists => if !not { "did not exist" } else { "existed" },
                                 CmpOperator::Empty => if !not { "was not empty"} else { "was empty" },
                                 CmpOperator::IsList => if !not { "was not a list " } else { "was list" },
                                 CmpOperator::IsMap => if !not { "was not a struct" } else { "was struct" },
                                 CmpOperator::IsString => if !not { "was not a string " } else { "was string" },
                                 _ => unreachable!()
                             },
                             each)?,
                    )?;

                }
                else {
                    // EQUALS failed at property path Properties.Encrypted because provided value [false] did not match with expected value [true].
                    writeln!(writer, "{}",
                        binary_message(
                            rules_file_name,
                            data_file_name,
                            if not { "did" } else { "did not" },
                            each
                        )?,
                    )?;
                }
            }

        }
    }

    Ok(())
}

