use core::fmt::Debug;
use std::io::{prelude::*, BufReader};
use std::{fs, io, path::Path, path::PathBuf};
pub mod logger;
pub mod json;

pub struct TextFormat {}
impl TextFormat {
    /// Format text with color or certain style.
    pub fn new() -> TextFormat {
        TextFormat {}
    }

    /// Add color to text.
    pub fn colored(&self, text: &str, color: &str) -> String {
        match color {
            "red" => format!("\x1b[31m{}\x1b[0m", text),
            "green" => format!("\x1b[32m{}\x1b[0m", text),
            "yellow" => format!("\x1b[33m{}\x1b[0m", text),
            "blue" => format!("\x1b[34m{}\x1b[0m", text),
            "magenta" => format!("\x1b[35m{}\x1b[0m", text),
            "cyan" => format!("\x1b[36m{}\x1b[0m", text),
            "white" => format!("\x1b[37m{}\x1b[0m", text),
            "light_red" => format!("\x1b[91m{}\x1b[0m", text),
            "light_green" => format!("\x1b[92m{}\x1b[0m", text),
            "light_yellow" => format!("\x1b[93m{}\x1b[0m", text),
            "light_blue" => format!("\x1b[94m{}\x1b[0m", text),
            "light_magenta" => format!("\x1b[95m{}\x1b[0m", text),
            "light_cyan" => format!("\x1b[96m{}\x1b[0m", text),
            "light_white" => format!("\x1b[97m{}\x1b[0m", text),
            _ => text.to_string(),
        }
    }
}

pub struct IO {}
impl IO {
    pub fn say<T: Debug>(e: &T) {
        /* Display data. */
        println!("{:?}", e);
    }

    pub fn pt(text: &str) {
        println!("{}", text);
    }

    pub fn lines_from_file(filename: impl AsRef<Path>) -> Vec<String> {
        let file = std::fs::File::open(filename).expect("Open file failed.");
        let buf = BufReader::new(file);
        buf.lines()
            .map(|l| l.expect("Could not parse line."))
            .collect()
    }
}

pub fn print_type_of<T>(_: &T) {
    println!("{}", std::any::type_name::<T>())
}

pub struct FileSystem {}

impl FileSystem {
    pub fn walk<P>(dir_name: P) -> Vec<String>
    where
        P: AsRef<Path>,
    {
        // Walk over a directory.
        std::fs::read_dir(dir_name)
            .expect("read dir failed")
            .map(|entry| {
                entry
                    .expect("parse extry failed")
                    .path()
                    .to_str()
                    .expect("to str failed")
                    .to_string()
            })
            .collect()
    }

    pub fn read_to_lines<P>(filename: P) -> io::Result<io::Lines<io::BufReader<std::fs::File>>>
    where
        P: AsRef<Path>,
    {
        // Read file contents into vector of string
        let file = std::fs::File::open(filename)?;
        Ok(io::BufReader::new(file).lines())
    }

    pub fn read_to_string<P>(filename: P) -> io::Result<String>
    where
        P: AsRef<Path>,
    {
        // A wrapper of fs::read_to_string.
        fs::read_to_string(filename)
    }

    pub fn read_to_vector<P>(filename: P) -> Vec<String>
    where
        P: AsRef<Path>,
    {
        let fh = std::fs::File::open(filename).expect("failed to open file");
        std::io::BufReader::new(fh)
            .lines()
            .map(|ln| ln.expect("read line failed"))
            .collect()
    }

    // ------- write operations
    pub fn write_iterator_to_file<T>(file_name: &str, stream: T)
    where
        T: IntoIterator<Item = String>,
    {
        let strvec = stream.into_iter().collect::<Vec<String>>();
        std::fs::write(file_name, strvec.join("\n"))
            .expect(&format!("Failed to write contents to file {}", file_name));
    }

    pub fn cp(src: &str, target: &str) {
        fs::copy(src, target).expect(&format!("Failed to COPY file {} to {}", src, target));
    }

    pub fn rm(filename: &str) {
        fs::remove_file(filename).expect(&format!("Failed to REMOVE file {}", filename));
    }

    pub fn exists(filename: &str) -> bool {
        // Does file exist or not.
        Path::new(filename).exists()
    }

    pub fn dir_size(path: impl Into<PathBuf>) -> io::Result<u64> {
        // Get directory size.
        fn rec(mut dir: fs::ReadDir) -> io::Result<u64> {
            dir.try_fold(0, |acc, file| {
                let file = file?;
                let size = match file.metadata()? {
                    data if data.is_dir() => rec(fs::read_dir(file.path())?)?,
                    data => data.len(),
                };
                Ok(acc + size)
            })
        }

        let buf = path.into();
        let f = fs::metadata(&buf).unwrap();
        if f.is_file() {
            return Ok(f.len());
        }
        rec(fs::read_dir(&buf)?)
    }
}
