/*-
 * cdns-rs - a simple sync/async DNS query library
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */
use std::hash::{Hash, Hasher};

use crate::error::CDnsResult;

use crate::cfg_resolv_parser::{ResolveConfig, ResolveConfigLookup};

use super::{QDnsName, QType, common::DnsRequestHeader};


#[derive(Debug)]
pub(crate) struct QDnsReq<'req>
{
    req_name: QDnsName<'req>,
    qtype: QType,
    header: Option<DnsRequestHeader>,
}


impl<'req> Eq for QDnsReq<'req> {}

impl<'req> PartialEq for QDnsReq<'req>
{
    fn eq(&self, other: &QDnsReq) -> bool 
    {
        return self.req_name == other.req_name;
    }
}

/*
impl<'req> Ord for QDnsReq<'req> 
{
    fn cmp(&self, other: &Self) -> Ordering 
    {
        // Sort in reverse order
        return self.header.id.cmp(&other.header.id);
    }
}

impl<'req> PartialOrd for QDnsReq<'req> 
{
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> 
    {
        return Some(self.cmp(other));
    }
}*/

impl<'req> Hash for QDnsReq<'req>
{
    fn hash<H: Hasher>(&self, state: &mut H) 
    {
        self.req_name.hash(state);
    }
}

/*
impl Borrow<u16> for QDnsReq
{
    fn borrow(&self) -> &u16 
    {
        return &self.header.id;
    }
}*/

impl<'req> QDnsReq<'req>
{
    pub(crate)
    fn new(req_name: QDnsName<'req>, qtype: QType, resolvers: &ResolveConfig) -> CDnsResult<Self>
    {
        let lookup = 
            if resolvers.lookup.contains(ResolveConfigLookup::BIND) == true
            {
                Some(DnsRequestHeader::construct_lookup(req_name.clone(), qtype)?)
            }
            else
            {
                None
            };

        return Ok(QDnsReq { req_name: req_name, qtype: qtype, header: lookup });
    }

    pub(crate)
    fn get_header(&self) -> &DnsRequestHeader
    {
        return self.header.as_ref().unwrap();
    }

    pub(crate) 
    fn get_type(&self) -> &QType
    {
        return &self.qtype;
    }

    pub(crate) 
    fn get_req_name(&self) -> &QDnsName<'req>
    {
        return &self.req_name;
    }
}

