/*-
 * cdns-rs - a simple sync/async DNS query library
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

/// This file contains the config file parsers.


use std::path::Path;

use crate::cfg_resolv_parser::{ResolveConfig, OptionFlags};
use crate::{error::*};

use super::caches::CacheOperations;
use super::cfg_parsers::{ConfigParser, read_file};
use super::common::{RESOLV_CFG_PATH, RESOLV_CFG_PATH_P};
use super::log::sync_log_writer;


impl CacheOperations for ResolveConfig
{
    fn is_reload_allowed(&self) -> bool 
    {
        return !self.option_flags.contains(OptionFlags::OPT_NO_RELOAD);
    }
}


impl ConfigParser<ResolveConfig> for ResolveConfig
{
    fn parse_config() -> CDnsResult<Self>
    {
        let file_content = read_file(RESOLV_CFG_PATH)?;
        
        let mut writer = Writer::new();

        let ret = Self::parser_resolv_internal(file_content.as_str(), &mut writer);

        sync_log_writer(writer);

        return ret;
    }

    fn get_file_path() -> &'static Path
    {
        return &RESOLV_CFG_PATH_P;
    }

    fn is_default(&self) -> bool
    {
        return self.nameservers.is_empty();
    }
}

impl ResolveConfig
{
    /// This function should be used when the program which uses this library
    /// requires to override the systems /etc/resolv.conf
    pub 
    fn custom_config(resolv_cfg: &str) -> CDnsResult<Self>
    {
        let mut writer = Writer::new();

        let ret = Self::parser_resolv_internal(resolv_cfg, &mut writer);

        sync_log_writer(writer);

        return ret;
    }
}

