/*-
 * cdns-rs - a simple sync/async DNS query library
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

/*
    Why it is needed?

    Because I have not found any faster way to create a global variable
    which is written only when it is initialized, and later will only be
    read. 

    Usage:
    static TEST: ReadOnlyCell<i32> = unsafe { ReadOnlyCell::new_uninitialized() };
    static TEST2: ReadOnlyCell<i32> = ReadOnlyCell::new(6);
    
    fn main() {
        unsafe {
            TEST.init(5);
        }
    
        println!("{}", *TEST);
        println!("{}", *TEST2);
    }
*/

use std::sync::atomic::{AtomicBool, Ordering};
use std::cell::UnsafeCell;
use std::mem::MaybeUninit;
use std::ops;

pub struct ReadOnlyCell<T> 
{
    title: &'static str,
    inner: UnsafeCell<MaybeUninit<T>>,
    blocked: AtomicBool,
}

unsafe impl<T> Sync for ReadOnlyCell<T> {}

impl<T> ReadOnlyCell<T> 
{
    pub const 
    fn new_not_loceked(init: T, title: &'static str) -> ReadOnlyCell<T> 
    {
        return 
            ReadOnlyCell 
            {
                title: title,
                inner: UnsafeCell::new(MaybeUninit::new(init)),
                blocked: AtomicBool::new(false),
            };
    }

    pub unsafe 
    fn init(&self, init: T) 
    {
        if self.blocked.load(Ordering::SeqCst) == false
        {
            (*self.inner.get()) = MaybeUninit::new(init);
            self.blocked.store(true, Ordering::SeqCst);
        }
        else
        {
            panic!("Assertion trap: attempt to re-initialize the ReadOnlyCell: {}", self.title);
        }
    }

    pub 
    fn is_init(&self) -> bool
    {
        return self.blocked.load(Ordering::SeqCst);
    }
}

impl<T> ops::Deref for ReadOnlyCell<T> 
{
    type Target = T;
    fn deref(&self) -> &T 
    {
        return unsafe 
        { 
            &*(*self.inner.get()).as_ptr()
        };
    }
}

#[test]
#[should_panic]
fn test_panic()
{
    static IS_DEBUG: ReadOnlyCell<bool> = ReadOnlyCell::new_not_loceked(true, "is_debug_flag");

    assert_eq!(*IS_DEBUG, true, "should be true");

    unsafe 
    {
        IS_DEBUG.init(false);
    }

    assert_eq!(*IS_DEBUG, false, "should be false");

    // should panic
    unsafe 
    {
        IS_DEBUG.init(true);
    }
}
