/*-
 * cdns-rs - a simple sync/async DNS query library
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

use std::
{
    ffi::{CStr, CString}, 
    mem::MaybeUninit, 
    net::{IpAddr, Ipv4Addr, Ipv6Addr}
};

use nix::libc;

use crate::{error::*, internal_error, internal_error_map};

/// This file contains code which should be manually ported to other
/// OSes because it is OS specific or there is no support in rust's libc.

/*
pub const IFNAMSIZ: usize =	16;

#[repr(C)]
#[derive(Copy, Clone)]
pub union ifr_ifrn_u
{
    pub ifrn_name: [u8; IFNAMSIZ],
}

#[repr(C)]
#[derive(Copy, Clone)]
pub struct ifmap
{
    pub mem_start: libc::c_long,
    pub mem_end: libc::c_long,
    pub base_addr: libc::c_short,
    pub irq: u8,
    pub dma: u8,
    pub port: u8
    // 3 bytes spare
}

#[repr(C)]
#[derive(Copy, Clone)]
pub union ifsu {
    pub raw_hdlc_proto: *mut libc::c_void,
    pub cisco: *mut libc::c_void,
    pub fr: *mut libc::c_void,
    pub fr_pvc: *mut libc::c_void,
    pub fr_pvc_info: *mut libc::c_void,
    pub sync: *mut libc::c_void,
    pub te1: *mut libc::c_void,
}

#[repr(C)]
#[derive(Copy, Clone)]
pub struct if_settings 
{
	pub type_: libc::c_uint,
	pub size:  libc::c_uint,
	pub ifsu:  ifsu,
}

#[repr(C)]
#[derive(Copy, Clone)]
pub union ifr_ifru_u
{
    pub ifru_addr: libc::sockaddr,
    pub ifru_dstaddr: libc::sockaddr,
    pub ifru_broadaddr: libc::sockaddr,
    pub ifru_netmask: libc::sockaddr,
    pub ifru_hwaddr: libc::sockaddr,
    pub ifru_flags: libc::c_short,
    pub ifru_ivalue: libc::c_int,
    pub ifru_mtu: libc::c_int,
    pub ifru_map: ifmap,
    pub ifru_slave: [i8; IFNAMSIZ],
    pub ifru_newname: [i8; IFNAMSIZ],
    pub ifru_data: *const libc::c_void,
    pub ifru_settings: if_settings
}

#[repr(C)]
#[derive(Copy, Clone)]
pub struct ifreq
{
    pub ifr_name: ifr_ifrn_u,
    pub ifr_ifru: ifr_ifru_u
}*/


/// A network interface info
pub struct IfInfo
{
    /// A pointer to the start of the linked list with IF data
    if_inf: *mut libc::ifaddrs,
}

impl Drop for IfInfo
{
    fn drop(&mut self) 
    {
        if self.if_inf.is_null() == true
        {
            return;
        }

        unsafe
        {
            libc::freeifaddrs(self.if_inf);
        }
    }
}

impl IfInfo
{
    //const NI_MAXHOST: usize = 1025;

    /// Creates an instance with pointer to the linked list with interfaces in system.
    pub unsafe
    fn get_interfaces_info() -> CDnsResult<Self>
    {
        let mut addrs = MaybeUninit::<*mut libc::ifaddrs>::uninit();

        let ifaddrs = libc::getifaddrs(addrs.as_mut_ptr());

        if ifaddrs == -1
        {
            internal_error!(CDnsErrorType::InternalError, "{}", std::io::Error::last_os_error());
        }

        return Ok(
            Self { if_inf: addrs.assume_init() }
        );
    }

    /// Returns the interface IP address if inteface exists.
    pub unsafe
    fn get_ifr_ip(&self, ifr: &str, ip_addr: &IpAddr) -> CDnsResult<Option<IpAddr>>
    {
        if self.if_inf.is_null() == true
        {
            panic!("if_inf is NULL");
        }

        let c_ifr = CString::new(ifr).map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

        let mut tmp = self.if_inf;
        while tmp.is_null() == false
        {
            if (*tmp).ifa_name.is_null() == true
            {
                tmp = (*tmp).ifa_next;
                continue;
            }

            // convert raw char ptr to cstr
            let name = CStr::from_ptr((*tmp).ifa_name);

            // match title
            if name != c_ifr.as_c_str()
            {
                tmp = (*tmp).ifa_next;
                continue;
            }

            let addr = (*tmp).ifa_addr;

            if addr.is_null() == false
            {
                
                let mut host = [0_u8; libc::NI_MAXHOST as usize];

                let fam = (*addr).sa_family;

                if fam == libc::AF_INET as u16 && ip_addr.is_ipv4() == true
                {
                    let s = 
                        libc::getnameinfo(
                            addr,
                            std::mem::size_of::<libc::sockaddr_in>() as u32,
                            host.as_mut_ptr() as *mut i8, 
                            libc::NI_MAXHOST as u32,
                            std::ptr::null_mut(), 
                            0, 
                            libc::NI_NUMERICHOST
                        );

                    if s != 0
                    {
                        internal_error!(CDnsErrorType::InternalError, "{}", std::io::Error::last_os_error());
                    }

                    let c = 
                        CStr::from_ptr(host.as_ptr() as *const i8)
                            .to_str()
                            .map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;
                    
                    let ip4: Ipv4Addr = c.parse().map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;
                    
                    return Ok(Some(IpAddr::from(ip4)));
                }
                else if fam == libc::AF_INET6 as u16 && ip_addr.is_ipv6() == true
                {
                    let s = 
                        libc::getnameinfo(
                            addr,
                            std::mem::size_of::<libc::sockaddr_in6>() as u32,
                            host.as_mut_ptr() as *mut i8, 
                            libc::NI_MAXHOST as u32,
                            std::ptr::null_mut(), 
                            0, 
                            libc::NI_NUMERICHOST
                        );

                    if s != 0
                    {
                        internal_error!(CDnsErrorType::InternalError, "{}", std::io::Error::last_os_error());
                    }

                    let c = 
                        CStr::from_ptr(host.as_ptr() as *const i8)
                            .to_str()
                            .map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;
                    
                    let c2 = 
                        match c.split_once("%")
                        {
                            Some((ip, _)) => ip,
                            None => c
                        };

                    let ip6: Ipv6Addr = c2.parse().map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;
                   
                    return Ok(Some(IpAddr::from(Ipv6Addr::from(ip6))));
                }
            }

            // next
            tmp = (*tmp).ifa_next;
        }

        return Ok(None);
    }
}
