/*-
 * cdns-rs - a simple sync/async DNS query library
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

use std::{sync::Arc};

use super::{network::{SocketTap, NetworkTapType}, common::DnsRequestHeader};


#[derive(Debug)]
pub(crate) struct Tap
{
    /// A socket which should be used to send requests
    tap: Box<NetworkTapType>,

    /// A requests binded to this specific tap
    qdnsr: Vec<DnsRequestHeader>
}

unsafe impl Send for Tap {}
unsafe impl Sync for Tap {}

impl Tap
{
    pub(crate) 
    fn new(tap: Box<NetworkTapType>, req: DnsRequestHeader) -> Self
    {
        return 
            Self
            {
                tap: tap,
                qdnsr: vec![req]
            };
    }

    pub(crate) 
    fn add(&mut self, req: DnsRequestHeader)
    {
        self.qdnsr.push(req);
    }

    pub(crate)
    fn into_inner(self) -> (Box<NetworkTapType>, Vec<DnsRequestHeader>)
    {
        return (self.tap, self.qdnsr);
    }

    /*pub(crate)
    fn inner_requests(self) -> Vec<DnsRequestHeader>
    {
        return self.qdnsr;
    }*/
}

pub(crate) struct AsyncTaps
{
    taps: Vec<Tap>,
}

impl AsyncTaps
{
    pub(crate) 
    fn new_with_capacity(tap_n: usize) -> Self
    {
        return Self{ taps: Vec::with_capacity(tap_n)/*, reqs: HashMap::with_capacity(req_n)*/ };
    }

    pub(crate) 
    fn push(&mut self, tap: Tap)
    {
        self.taps.push(tap);
    }

    pub(crate) 
    fn push_to_last(&mut self, req: DnsRequestHeader)
    {
        let t = self.taps.last_mut().unwrap();

        t.add(req);
    }

    pub(crate) 
    fn len(&self) -> usize
    {
        return self.taps.len();
    }

    pub(crate)
    fn into_iter(self) -> std::vec::IntoIter<Tap>
    {
        return self.taps.into_iter();
    }
}


