/*-
 * cdns-rs - a simple sync/async DNS query library
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

/// This file contains the config file parsers.


use std::path::Path;

use async_trait::async_trait;

use crate::cfg_host_parser::HostConfig;
use crate::{error::*};


use super::caches::CacheOperations;
use super::cfg_parsers::{ConfigParser, read_file};
use super::common::{ HOST_CFG_PATH, HOST_CFG_PATH_P };
use super::log::async_log_writer;

/// An /etc/hosts file parser


impl CacheOperations for HostConfig
{
    fn is_reload_allowed(&self) -> bool 
    {
        return true;
    }
}


#[async_trait]
impl ConfigParser<HostConfig> for HostConfig
{
    async
    fn parse_config() -> CDnsResult<Self>
    {
        let mut writer = Writer::new();

        let file_content = read_file(HOST_CFG_PATH).await?;
        
        let ret = Self::parse_host_file_internal(file_content, &mut writer);

        async_log_writer(writer).await;

        return ret;
    }

    fn get_file_path() -> &'static Path
    {
        return &HOST_CFG_PATH_P;
    }

    fn is_default(&self) -> bool
    {
        return self.is_empty();
    }
}

