/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::collections::VecDeque;
use std::net::IpAddr;
use std::sync::Arc;

use crate::error::*;
use crate::{internal_error};

use super::caches::CACHE;
use super::query::{QDns};
use super::{QuerySetup, QDnsQueriesRes, ResolveConfig};
use super::common::{QType, DnsRdata, DnsSoa};

/// Resolves the A and AAAA query.
///
/// # Arguments
/// 
/// * `fqdn` - [AsRef<str>] a domain name
/// 
/// # Returns
/// 
/// [CDnsResult] - Ok with inner type [Vec] [DnsSoa]. If vector is empty
///     then no results found.
/// 
/// [CDnsResult] - Err with error description 
pub 
fn resolve_fqdn<C>(fqdn: C, custom_resolv: Option<Arc<ResolveConfig>>) -> CDnsResult<Vec<IpAddr>>
where C: AsRef<str>
{
    let resolvers = 
        custom_resolv.map_or(CACHE.clone_resolve_list()?, |f| f);

    // forming request
    let dns = QDns::make_a_aaaa_request(resolvers, fqdn.as_ref(), QuerySetup::default())?;

    // sending request and receiving results
    let res = dns.query();
  
    // extracting data
    let mut iplist: Vec<IpAddr> = vec![];

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            for r in res
            {
                if r.is_ok() == false
                {
                    // skip errors
                    continue;
                }

                for dnsr in r.get_responses()
                {
                    match dnsr.rdata
                    {
                        DnsRdata::A{ ip } => iplist.push(IpAddr::from(ip)),
                        DnsRdata::AAAA{ ip } => iplist.push(IpAddr::from(ip)),
                        _ => continue,
                    }
                }
            }
        },
        QDnsQueriesRes::DnsNotAvailable =>
        {
            internal_error!(CDnsErrorType::DnsNotAvailable, "");
        }
    }

    return Ok(iplist);
}

/// Resolves the MX record by domain name. It returns a list of domains.
/// The list is sorted by the `preference`.
/// 
/// # Arguments
/// 
/// * `fqdn` - [AsRef<str>] a domain name
/// 
/// # Returns
/// 
/// [CDnsResult] - Ok with inner type [Vec] [DnsSoa]. If vector is empty
///     then no results found.
/// 
/// [CDnsResult] - Err with error description 
pub 
fn resolve_mx<C>(fqdn: C, custom_resolv: Option<Arc<ResolveConfig>>) -> CDnsResult<Vec<String>>
where C: AsRef<str>
{
    let resolvers = 
        custom_resolv.map_or(CACHE.clone_resolve_list()?, |f| f);

    let mut dns_req = 
        QDns::make_empty(resolvers, 1, QuerySetup::default());

    dns_req.add_request(QType::MX, fqdn.as_ref());

    // sending request and receiving results
    let res = dns_req.query();
  
    // extracting data
    let mut mxlist: VecDeque<(u16, String)> = VecDeque::with_capacity(5);

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            for r in res
            {
                if r.is_ok() == false
                {
                    // skip errors
                    continue;
                }

                for dnsr in r.get_responses()
                {
                    match dnsr.rdata
                    {
                        DnsRdata::MX{ ref preference, ref exchange } => 
                        {
                            //iterate and search for the suitable place
                            let mut index: usize = 0;

                            for (pref, _) in mxlist.iter()
                            {
                                if *pref >= *preference
                                {
                                    break;
                                }

                                index += 1;
                            }

                            if index == mxlist.len()
                            {
                                // push back
                                mxlist.push_back((*preference, exchange.clone()));
                            }
                            else
                            {
                                mxlist.insert(index, (*preference, exchange.clone()));
                            }
                        }
                        _ => continue,
                    }
                }
            }
        },
        QDnsQueriesRes::DnsNotAvailable =>
        {
            internal_error!(CDnsErrorType::DnsNotAvailable, "");
        }
    }

    return Ok(mxlist.into_iter().map( |(_, ip)| ip).collect());
}

/// Resolves the SOA record
/// 
/// # Arguments
/// 
/// * `fqdn` - [AsRef<str>] a domain name
/// 
/// # Returns
/// 
/// [CDnsResult] - Ok with inner type [Vec] [DnsSoa]. If vector is empty
///     then no results found.
/// 
/// [CDnsResult] - Err with error description 
pub 
fn resolve_soa<C>(fqdn: C, custom_resolv: Option<Arc<ResolveConfig>>) -> CDnsResult<Vec<DnsSoa>>
where C: AsRef<str>
{
    let resolvers = 
        custom_resolv.map_or(CACHE.clone_resolve_list()?, |f| f);

    let mut dns_req = 
        QDns::make_empty(resolvers, 1, QuerySetup::default());

    dns_req.add_request(QType::SOA, fqdn.as_ref());

    // sending request and receiving results
    let res = dns_req.query();

    let mut soa_list: Vec<DnsSoa> = Vec::new();

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            for r in res
            {
                if r.is_ok() == false
                {
                    // skip errors
                    continue;
                }

                for dnsr in r.move_responses()
                {
                    match dnsr.rdata
                    {
                        DnsRdata::SOA{ soa } =>
                        {
                            soa_list.push(soa);
                        },
                        _ => {}
                    }
                }
            }
        },
        QDnsQueriesRes::DnsNotAvailable => {}
    }

    return Ok(soa_list);
}

pub 
fn resolve_reverse<C>(fqdn: C, custom_resolv: Option<Arc<ResolveConfig>>) -> CDnsResult<Vec<String>>
where C: AsRef<str>
{
    let resolvers = 
        custom_resolv.map_or(CACHE.clone_resolve_list()?, |f| f);

    let mut dns_req = 
        QDns::make_empty(resolvers, 1, QuerySetup::default());

    dns_req.add_request(QType::PTR, fqdn.as_ref());

    // sending request and receiving results
    let res = dns_req.query();

    let mut ptr_list: Vec<String> = Vec::new();

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            for r in res
            {
                if r.is_ok() == false
                {
                    // skip errors
                    continue;
                }

                for dnsr in r.move_responses()
                {
                    match dnsr.rdata
                    {
                        DnsRdata::PTR{ fqdn } =>
                        {
                            ptr_list.push(fqdn);
                        },
                        _ => {}
                    }
                }
            }
        },
        QDnsQueriesRes::DnsNotAvailable => {}
    }

    return Ok(ptr_list);
}

#[test]
fn test_mx_resolve()
{
    let mx_doms = resolve_mx("protonmail.com", None);

    assert_eq!(mx_doms.is_err(), false);

    let mx_doms = mx_doms.unwrap();

    let mut index = 0;
    for di in mx_doms
    {
        match index
        {
            0 => assert_eq!(di.as_str(), "mail.protonmail.ch"),
            1 => assert_eq!(di.as_str(), "mailsec.protonmail.ch"),
            _ => panic!("test is required to be modified")
        }

        index += 1;

        println!("{}", di);
    }
    
}


#[test]
fn test_a_aaaa_resolve()
{
    let a_aaaa = resolve_fqdn("protonmail.com", None);

    assert_eq!(a_aaaa.is_ok(), true);

    let a_aaaa = a_aaaa.unwrap();

    for di in a_aaaa
    {
        println!("{}", di);
    }
    
}
