/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::{collections::BTreeSet, os::unix::prelude::AsRawFd, slice::IterMut};

use nix::poll::PollFd;

use crate::error::CDnsResult;

use super::{network::SocketTap, common::DnsRequestHeader};



#[derive(Debug)]
pub(crate) struct Tap
{
    /// A socket which should be used to send requests
    tap: Box<dyn SocketTap>,//Rc<RefCell<Box<dyn SocketTap>>>,

    /// A requests binded to this specific tap
    qdnsr: Vec<DnsRequestHeader>
}

impl Tap
{
    pub(crate) 
    fn new(tap: Box<dyn SocketTap>, req: DnsRequestHeader) -> Self
    {
        return 
            Self
            {
                tap: tap,
                qdnsr: vec![req]
            };
    }

    pub(crate) 
    fn add(&mut self, req: DnsRequestHeader)
    {
        self.qdnsr.push(req);
    }

    pub(crate)
    fn get_tap(&mut self) -> &mut dyn SocketTap
    {
        return self.tap.as_mut();
    }

    pub(crate) 
    fn unwrap_inner(self) -> (Box<dyn SocketTap>, Vec<DnsRequestHeader>)
    {
        return (self.tap, self.qdnsr);
    }

    pub(crate) 
    fn find_request(&self, id: u16) -> Option<&DnsRequestHeader>
    {
        for r in self.qdnsr.iter()
        {
            if r.get_id() == id
            {
                return Some(r);
            }
        }

        return None;
    }
    

    pub(crate)
    fn generate_request_list_id(&self) -> BTreeSet<u16>
    {
        let mut req_map: BTreeSet<u16> = BTreeSet::new();

        for req in self.qdnsr.iter()
        {
            //let pkt = header.to_bytes()?;
            req_map.insert(req.get_id());
        }

        return req_map;
    }

    pub(crate)
    fn generate_packets(&self) -> CDnsResult<Vec<Vec<u8>>>
    {
        let mut out: Vec<Vec<u8>> = Vec::with_capacity(self.qdnsr.len());

        for r in self.qdnsr.iter()
        {
            out.push(r.to_bytes()?);
        }

        return Ok(out);
    }
}

pub(crate) struct PollTaps
{
    taps: Vec<Tap>,
}

impl PollTaps
{
    pub(crate) 
    fn new_with_capacity(tap_n: usize) -> Self
    {
        return Self{ taps: Vec::with_capacity(tap_n)/*, reqs: HashMap::with_capacity(req_n)*/ };
    }

    pub(crate) 
    fn push(&mut self, tap: Tap)
    {
        self.taps.push(tap);
    }

    pub(crate) 
    fn push_to_last(&mut self, req: DnsRequestHeader)
    {
        let t = self.taps.last_mut().unwrap();

        t.add(req);
    }

    pub(crate) 
    fn len(&self) -> usize
    {
        return self.taps.len();
    }

    pub(crate)
    fn get_mut_iter(&mut self) -> IterMut<'_, Tap>
    {
        return self.taps.iter_mut();
    }

    pub(crate)
    fn into_iter(self) -> std::vec::IntoIter<Tap>
    {
        return self.taps.into_iter();
    }
    

    pub(crate) 
    fn find_tap_by_pollfd(&mut self, pfd: &PollFd) -> &mut Tap
    {
        for t in self.taps.iter_mut()
        {
            if t.tap.get_pollfd().as_raw_fd() == pfd.as_raw_fd()
            {
                return t;
            }
        }

        panic!("find_tap_by_pollfd() misuse or error! If used correctly, never fails!");
    }

    pub(crate) 
    fn get_pollfd_list(&self) -> Vec<PollFd>
    {
       return self.taps.iter().map(|t| t.tap.get_pollfd()).collect();
    }

    pub(crate) 
    fn get_reqs_list(&self) -> BTreeSet<u16>
    {
        let mut req_id_tbl: BTreeSet<u16> = BTreeSet::new();

        for t in self.taps.iter()
        {
            req_id_tbl.extend(t.generate_request_list_id());
        }
    
        return req_id_tbl;
    }
}