/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::collections::{LinkedList};
use std::convert::{TryFrom, TryInto};
use std::net::{SocketAddr};
use std::fmt;
use std::time::Duration;
use std::time::Instant;


use crate::error::*;
use crate::{internal_error};

use super::common::*;


/// A query override
#[derive(Debug, Clone)]
pub struct QuerySetup
{
    /// Sets the realtime counter to measure delay
    pub(crate) measure_time: bool, 

    /// Forces to ignore hosts file
    pub(crate) ign_hosts: bool, 

    /// Overrides the timeout duration for reply awaiting.
    pub(crate) timeout: Option<u32>,
}

impl Default for QuerySetup
{
    fn default() -> Self 
    {
        return Self { measure_time: false, ign_hosts: false, timeout: None };
    }
}

impl QuerySetup
{
    /// Turns on/off the time measurment whcih measure how many time went
    /// since query started for each response.
    pub 
    fn set_measure_time(&mut self, flag: bool) -> &mut Self
    {
        self.measure_time = flag;
        
        return self;
    }

    /// Turns on/off the option which when set is force to ignore lookup in
    /// /etc/hosts
    pub 
    fn set_ign_hosts(&mut self, flag: bool) -> &mut Self
    {
        self.ign_hosts = flag;

        return self;
    }

    /// Overrides the timeout. Can not be 0.
    pub 
    fn set_override_timeout(&mut self, timeout: u32) -> &mut Self
    {
        if timeout == 0
        {
            return self;
        }

        self.timeout = Some(timeout);

        return self;
    }

    /// Resets the override timeout.
    pub 
    fn reset_override_timeout(&mut self) -> &mut Self
    {
        self.timeout = None;

        return self;
    }

}

/// A [StatusBits] decoded response status.
#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum QDnsQueryRec
{
    /// Response is Ok and
    Ok,
    /// Server side error
    ServFail,
    /// Query does not exist, but meaningfull when `aa` is true
    NxDomain,
    /// A name server refuses to perform the specified operation
    Refused,
    /// A name server does not support the requested kind of query
    NotImpl,
    /// A fromat error
    FormError,
}

impl QDnsQueryRec
{
    /// Returns true if [QueryMode] is set to continue query other nameservers.  
    /// Returns false if there is no need to query other nameservers i.e 
    ///  [QueryMode::DefaultMode] or the response was from authotherative server.
    pub(crate)
    fn try_next_nameserver(&self, aa: bool) -> bool
    {
        match *self
        {
            Self::NxDomain =>
            {
                if aa == true
                {
                    // the response is from authotherative server
                    return false;
                }

                return true;
            },
            Self::Ok | Self::Refused | Self::ServFail | Self::NotImpl | Self::FormError =>
            {
                return true;
            }
        }
    }
}

impl TryFrom<StatusBits> for QDnsQueryRec
{
    type Error = CDnsError;

    fn try_from(value: StatusBits) -> Result<Self, Self::Error> 
    {
        if value.contains(StatusBits::RESP_NOERROR) == true
        {
            /*let resps: Vec<QueryRec> = 
                ans.response.into_iter().map(|a| a.into()).collect();
            */
            return Ok(QDnsQueryRec::Ok);
        }
        else if value.contains(StatusBits::RESP_FORMERR) == true
        {
            return Ok(QDnsQueryRec::FormError);
        }
        else if value.contains(StatusBits::RESP_NOT_IMPL) == true
        {
            return Ok(QDnsQueryRec::NotImpl);
        }
        else if value.contains(StatusBits::RESP_NXDOMAIN) == true
        {
            return Ok(QDnsQueryRec::NxDomain);
        }
        else if value.contains(StatusBits::RESP_REFUSED) == true
        {
            return Ok(QDnsQueryRec::Refused);
        }
        else if value.contains(StatusBits::RESP_SERVFAIL) == true
        {
            return Ok(QDnsQueryRec::ServFail);
        }
        else
        {
            internal_error!(CDnsErrorType::DnsResponse, "response status bits unknwon result: '{}'", value.bits());
        };
    }
}


impl fmt::Display for QDnsQueryRec
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::Ok => 
            {
                writeln!(f, "OK")?;
            },
            Self::ServFail =>
            {
                writeln!(f, "SERVFAIL")?;
            },
            Self::NxDomain =>
            {
                writeln!(f, "NXDOMAIN")?;
            },
            Self::Refused => 
            {   
                writeln!(f, "REFUSED")?;
            },
            Self::NotImpl => 
            {
                writeln!(f, "NOT IMPLEMENTED")?;
            },
            Self::FormError =>
            {
                writeln!(f, "FORMAT ERROR")?;
            }
        }

        return Ok(());
    }
}

/// A structure which describes the query properties and contains the
/// results.
#[derive(Clone, Debug)]
pub struct QDnsQuery
{
    /// A realtime time elapsed for query
    pub elapsed: Option<Duration>,
    /// Server which performed the response and port number
    pub server: String,
    /// Authoritative Answer
    pub aa: bool,
    /// Authoratives section
    pub authoratives: Vec<DnsResponsePayload>,
    /// Responses
    pub resp: Vec<DnsResponsePayload>,
    /// Status
    pub status: QDnsQueryRec,
}

impl fmt::Display for QDnsQuery
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        write!(f, "Source: {} ", self.server)?;
        if let Some(ref el) = self.elapsed
        {
            write!(f, "{:.2?} ", el)?;
        }

        if self.aa == true
        {
            writeln!(f, "Authoritative answer")?;
        }
        else
        {
            writeln!(f, "Non-Authoritative answer")?;
        }

        writeln!(f, "Authoritatives: {}", self.authoratives.len())?;

        if self.authoratives.len() > 0
        {
            for a in self.authoratives.iter()
            {
                writeln!(f, "{}", a)?;
            }

            writeln!(f, "")?;
        }

        writeln!(f, "Status: {}", self.status)?;

        writeln!(f, "Answers: {}", self.resp.len())?;

        if self.resp.len() > 0
        {
            for r in self.resp.iter()
            {
                writeln!(f, "{}", r)?;
            }

            writeln!(f, "")?;
        }

        return Ok(());
    }
}

impl QDnsQuery
{
    /// Returns true if the response is OK
    pub 
    fn is_ok(&self) -> bool
    {
        return self.status == QDnsQueryRec::Ok;
    }

    pub 
    fn is_authorative(&self) -> bool
    {
        return self.aa;
    }

    pub 
    fn get_elapsed_time(&self) -> Option<&Duration>
    {
        return self.elapsed.as_ref();
    }

    pub 
    fn get_server(&self) -> &String
    {
        return &self.server;
    }

    /// Returns the authorative server data if any
    pub 
    fn get_authoratives(&self) -> &[DnsResponsePayload]
    {
        return self.authoratives.as_slice();
    }

    /// Returns the responses if any
    pub 
    fn get_responses(&self) -> &[DnsResponsePayload]
    {
        return self.resp.as_slice();
    }

    /// Moves the responses from structure
    pub 
    fn move_responses(self) -> Vec<DnsResponsePayload>
    {
        return self.resp;
    }
}

impl QDnsQuery
{
    /// Constructs instance like it is from 'local' source but not from DNS server.
    pub 
    fn from_local(req_pl: Vec<DnsResponsePayload>, now: Option<&Instant>) -> QDnsQuery
    {
        let elapsed = 
            match now
            {
                Some(n) => Some(n.elapsed()),
                None => None
            };

        return 
            Self
            {
                elapsed:        elapsed,
                server:         HOST_CFG_PATH.to_string(),
                aa:             true,
                authoratives:   Vec::new(),
                status:         QDnsQueryRec::Ok,
                resp:           req_pl
            };
    }

    /// Constructs an instance from the remote response.
    pub 
    fn from_response(server: &SocketAddr, ans: DnsRequestAnswer, now: Option<&Instant>) -> CDnsResult<Self>
    {
        return Ok(
            Self
            {
                elapsed:        now.map_or(None, |n| Some(n.elapsed())),
                server:         server.to_string(),
                aa:             ans.header.status.contains(StatusBits::AUTH_ANSWER),
                authoratives:   ans.authoratives,
                status:         ans.header.status.try_into()?,
                resp:           ans.response,
            }
        );
    }
}

/// The result enum. 
#[derive(Debug)]
pub enum QDnsQueriesRes
{
    /// Received results
    DnsOk
    {
        res: Vec<QDnsQuery>,
    },
    /// All nameservers did not respond 
    DnsNotAvailable,
}

impl QDnsQueriesRes
{ 
    pub(crate)
    fn extend(&mut self, other: Self)
    {
        if other.is_results() == false
        {
            return;
        }

        match *self
        {
            Self::DnsOk{ ref mut res } =>
            {
                res.extend(other.into_inner().unwrap());
            },
            Self::DnsNotAvailable =>
            {
                *self = Self::DnsOk{ res: other.into_inner().unwrap() };
            }
        }
    }

    /// Unwraps the results without panics.
    pub 
    fn into_inner(self) -> Option<Vec<QDnsQuery>>
    {
        match self
        {
            Self::DnsNotAvailable => None,
            Self::DnsOk{ res } => return Some(res),
        }
    }

    /// Checks if the received result contains any results.
    /// It does not check if results were successfull. It just
    /// answers on question if any DNS server responded or record 
    /// in local databases was found.
    pub 
    fn is_results(&self) -> bool
    {
        match *self
        {
            Self::DnsNotAvailable => return false,
            _ => return true,
        }
    }

    /// This function tells how many responses it contains.
    /// If Self is [QDnsQueriesRes::DnsNotAvailable] then 0 will be returned.
    pub 
    fn len(&self) -> usize
    {
        if let Self::DnsOk{res} = self
        {
            return res.len();
        }
        else
        {
            return 0;
        }
    }

    /// This function returns the results if any.
    pub 
    fn list_results(&self) -> Option<std::slice::Iter<'_, QDnsQuery>>
    {
        if let Self::DnsOk{res} = self
        {
            return Some(res.iter());
        }
        else
        {
            return None;
        }
    }
}

impl From<QDnsQuery> for QDnsQueriesRes
{
    fn from(query: QDnsQuery) -> Self 
    {
        return QDnsQueriesRes::DnsOk{ res: vec![query] };
    }
}

impl From<LinkedList<QDnsQuery>> for QDnsQueriesRes
{
    fn from(responses: LinkedList<QDnsQuery>) -> Self 
    {
        if responses.len() > 0
        {
            return QDnsQueriesRes::DnsOk{ res: responses.into_iter().map(|h| h).collect() };
        }
        else
        {
            return QDnsQueriesRes::DnsNotAvailable;
        }
    }
}

impl fmt::Display for QDnsQueriesRes
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::DnsOk{ ref res } =>
            {
                for r in res.iter()
                {
                    write!(f, "{}", r)?;
                }
            },
            Self::DnsNotAvailable => write!(f, "No DNS server available")?
        }

        return Ok(());
    }
}


