/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

//! cdns-rs
//! 
//! An implementation of client side DNS query library - a resolver, a set of 
//! routines that provide access to the Internet Domain Name System (DNS). 
//! 
//! This library provides both:
//! sync and async crates to query remote DNS servers and local hosts file.
//! 
//! This code will be considered stable only when the major part of verion number
//! will reach v 1.x.x
//! 
//! # Limitations
//! 
//! - This library reads /etc/hosts and partially understands /etc/resolv.conf i.e
//! only 'nameserver' option is parsed at the moment.
//! - This library does not support TCP query at the moment.
//! - The nsswitch.conf is also ignored as this crate at the moment  is not pretending 
//!     to be a glibc/libc replacement.
//! - Does not support DNS-over-TLS at the moment
//! - Does not support DNSSEC at the moment
//! - /etc/resolve.conf is limited to 'nameserver'
//! 
//! # Supports
//! 
//! Supported GNU/Linux and *BSD (incl OSX, but not guaranteed)
//! 
//! # Layout
//! 
//! Dirs: 
//! - a_sync - contains all async code
//! - sync - contains all sync code
//! Files:
//! */*:
//! - portable.rs - contains the code which requres porting to other OS. 
//!     This code usually incompatiable with different UNIX-type OSes.
//! - error.rs - an error wrapper and mapper
//! - tokenizer.rs - contains simple tokenizer for parsing config files.
//! - read_only_cell.rs - an implementation of the once written cell for 
//!     initialization purposes.
//! - cfg_host_parser.rs - a common code for the /etc/host which is used by 
//!     both sync/async
//! - cfg_resolv_parser.rs - a common code for the /etc/resolv.conf which is
//!     used by sync/async
//! - query.rs - a common code of query
//! - query_private.rs - internal items for query construction
//! 
//! */sync/*:
//! - caches.rs - contains a cached in memory config files like /etc/hosts.
//!     If file was modified, the cache controller attempts to reload file during
//!     library call from main program. Sync realization of common file.
//! - cfg_parsers.rs - contains the confgiration files parsers.
//! - common.rs - this file contains all code related to DNS protocol i.e structures,
//!     parsing.
//! - network.rs - a code which deals with networking part, i.e msg exchange with DNS
//!     server.
//! - mutex.rs - a local implementaion of mutex for dafu usage in multithreading env.
//!     Derived from rust's mutex implementation.
//! - poison.rs - a mutex poison detection derived from rust's code.
//! - query.rs - a file which contains main logic
//! - query_polltaps.rs - a pairing of socket with request depending on config
//! - query_private.rs - a sync realization of the common file
//! - request.rs - a pre-defined requests
//! - log.rs - a log facility
//! 
//! Features: 
//! - feature = "use_async" for asynchronious code (use cdns_rs::a_sync::{...};)
//! - feature = "use_sync" for synchronious code (use cdns_rs::sync::{...};)
//! - feature = "no_error_output" does not output error
//! - feature = "custom_error_output" outputs error to defined output
//! `no_error_output` and `custom_error_output` can not be defined simultaniously.
//! 
//! If both `no_error_output` and `custom_error_output` are not defined then the errors
//! are output to stderr.
//!
//! All features can be used simultaniously.
//! # Usage
//! 
//! i.e 
//! cdns-rs = {version = "0.2", default-features = false, features = ["use_sync", "no_error_output"]}
//! 
//! By default, both `use_async`, `use_sync`, `no_error_output` features are enabled.
//! 
//! Example:
//! ```Rust
//! 
//! extern crate cdns_rs;
//! 
//! use cdns_rs::{QType, QDnsQueriesRes, QDnsQuery, QuerySetup, DnsRdata};
//! use cdns_rs::sync::QDns;
//! use cdns_rs::sync::caches::CACHE;
//! 
//! fn main() {
//! 
//!     println!("Hello, world!");
//! 
//!     let resolvers = CACHE.clone_resolve_list().unwrap();
//! 
//!     let mut q = QDns::make_empty(resolvers, 1, QuerySetup::default());
//!     q.add_request(QType::PTR, "127.0.0.1");
//! 
//!     let res = q.query().into_inner();
//! 
//!     if let Some(dq) = res
//!     {
//!         for d in dq
//!         {
//!             let resp = d.move_responses();
//! 
//!             for r in resp
//!             {
//!                 match r.rdata
//!                 {
//!                     DnsRdata::PTR{fqdn} =>
//!                     {
//!                         println!("{}", fqdn);
//!                     },
//!                     _ => {}
//!                 }
//!             }
//!         }
//!     }
//!     
//! ```


extern crate rand;
#[macro_use] extern crate lazy_static;
//extern crate libc;
#[macro_use] extern crate bitflags;
extern crate nix;

//extern crate indexmap;

#[cfg(feature = "use_sync")]
extern crate byteorder;
#[cfg(feature = "use_sync")]
extern crate crossbeam_utils;
#[cfg(feature = "use_sync")]
pub mod sync;


#[cfg(feature = "use_async")]
extern crate async_recursion;
#[cfg(feature = "use_async")]
extern crate tokio;
#[cfg(feature = "use_async")]
extern crate tokio_byteorder;
#[cfg(feature = "use_async")]
extern crate async_trait;
#[cfg(feature = "use_async")]
pub mod a_sync;

pub mod cfg_host_parser;
pub mod cfg_resolv_parser;

mod query_private;

pub mod query;
pub mod common;
mod portable;
#[macro_use] pub mod error;
mod tokenizer;
mod read_only_cell;

pub use lazy_static::*;
pub use error::*;
pub use common::{QType, DnsResponsePayload, DnsRdata, DnsSoa};
pub use query::{QDnsQueriesRes, QDnsQuery, QuerySetup, QDnsQueryRec};
