/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::{str, fmt::{self, Write}};

pub enum Writer
{
    None,
    Wr(Vec<u8>)
}

unsafe impl Send for Writer {}
impl Write for Writer
{
    fn write_str(&mut self, s: &str) -> fmt::Result 
    {
        match *self
        {
            Self::None => return Ok(()),
            Self::Wr( ref mut buf ) =>
            {
                buf.extend(s.as_bytes());
            }
        }
        
        return Ok(());
    }
}

impl Writer
{
    pub 
    fn new() -> Self
    {
        #[cfg(not(feature = "no_error_output"))]
        return Self::Wr( Vec::with_capacity(256) );

        #[cfg(feature = "no_error_output")]
        return Self::None;
    }

    pub 
    fn is_some(&self) -> bool
    {
        match *self
        {
            Self::None => return false,
            _ => return true
        }
    }

    pub unsafe
    fn get_str(&self) -> &str
    {
        match *self
        {
            Self::Wr( ref buf ) =>
            {
                return str::from_utf8_unchecked( buf.as_slice() );
            },
            Self::None => return "",
        }
    }
}


#[cfg(not(feature = "no_error_output"))]
#[macro_export]
macro_rules! writer_error {
    ($dst:expr, $($arg:tt)*) => (
        { let _ = std::fmt::Write::write_fmt($dst, format_args!($($arg)*)); }
    )
}

#[cfg(feature = "no_error_output")]
#[macro_export]
macro_rules! writer_error {
    ($dst:expr, $($arg:tt)*) => ({})
}


pub(crate)
fn map_read_err(e: std::io::Error) -> CDnsError
{
    return CDnsError::new(CDnsErrorType::IoError, format!("{}", e));
}

#[derive(Clone, PartialEq, Eq)]
pub struct CDnsError
{
    pub err_code: CDnsErrorType,
    pub message: String,
}

impl CDnsError
{
    pub fn new(err_code: CDnsErrorType, msg: String) -> Self
    {
        return CDnsError{err_code: err_code, message: msg};
    }
}

impl fmt::Display for CDnsError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "cdns: [{}], {}", self.err_code, self.message)
    }
}
impl fmt::Debug for CDnsError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "cdns: [{}], {}", self.err_code, self.message)
    }
}

#[derive(Clone, PartialEq, Eq)]
pub enum CDnsErrorType
{
    /// Error related to DNS response parsing
    DnsResponse,
    /// Received response with unknown ID in header
    RespIdMismatch,
    /// Internal error (assertions...)
    InternalError,
    /// Socket, File error
    IoError,
    /// Response was truncated
    MessageTruncated,
    /// Timeout event
    RequestTimeout,
    /// Error in configuraion file (format)
    ConfigError,
    /// Nameservers unreachable
    DnsNotAvailable,
}

impl fmt::Display for CDnsErrorType 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        match *self 
        {
            Self::DnsResponse => write!(f, "DNS response"),
            Self::RespIdMismatch => write!(f, "Response ID mismatch"),
            Self::InternalError => write!(f, "Internal Error"),
            Self::IoError => write!(f, "IO Error"),
            Self::MessageTruncated => write!(f, "Message was truncated"),
            Self::RequestTimeout => write!(f, "Request receive timout"),
            Self::ConfigError => write!(f, "Config file error"),
            Self::DnsNotAvailable => write!(f, "DNS not available"),
        }
    }
}

pub type CDnsResult<T> = Result<T, CDnsError>;

#[macro_export]
macro_rules! internal_error 
{
    ($src:expr,$($arg:tt)*) => (
        return std::result::Result::Err($crate::CDnsError::new($src, format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! internal_error_map
{
    ($src:expr,$($arg:tt)*) => (
        $crate::CDnsError::new($src, format!($($arg)*))
    )
}