/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/// This file contains the config file parsers.


use std::path::Path;

use async_trait::async_trait;


use crate::cfg_resolv_parser::{ResolveConfig, OptionFlags, ResolveConfEntry, RoundRobinIterator, ROUND_ROBIN_CNT};
use crate::{error::*};

use super::caches::CacheOperations;
use super::cfg_parsers::{ConfigParser, read_file};
use super::common::{RESOLV_CFG_PATH, RESOLV_CFG_PATH_P};
use super::log::async_log_writer;


impl CacheOperations for ResolveConfig
{
    fn is_reload_allowed(&self) -> bool 
    {
        return !self.option_flags.contains(OptionFlags::OPT_NO_RELOAD);
    }
}


#[async_trait]
impl ConfigParser<ResolveConfig> for ResolveConfig
{
    async 
    fn parse_config() -> CDnsResult<Self>
    {
        let file_content = read_file(RESOLV_CFG_PATH).await?;
        
        let mut writer = Writer::new();

        let ret = Self::parser_resolv_internal(file_content.as_str(), &mut writer);

        async_log_writer(writer).await;

        return ret;
    }

    fn get_file_path() -> &'static Path
    {
        return &RESOLV_CFG_PATH_P;
    }

    fn is_default(&self) -> bool
    {
        return self.nameservers.is_empty();
    }
}

impl ResolveConfig
{
    /// This function should be used when the program which uses this library
    /// requires to override the systems /etc/resolv.conf
    pub async 
    fn async_custom_config(resolv_cfg: &str) -> CDnsResult<Self>
    {
        let mut writer = Writer::new();
        
        let ret = Self::parser_resolv_internal(resolv_cfg, &mut writer);

        async_log_writer(writer).await;

        return ret;
    }
}


