/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/// This file contains the config file parsers.

use std::path::Path;

use async_trait::async_trait;

use tokio::io::{AsyncReadExt};
use tokio::fs::File;
use tokio::sync::Mutex;

use crate::{internal_error_map, error::*};


#[async_trait]
pub trait ConfigParser<T>
{
    async fn parse_config() -> CDnsResult<T>;

    fn get_file_path() -> &'static Path;

    fn is_default(&self) -> bool;
}

pub(super) async 
fn read_file(path: &str) -> CDnsResult<String>
{
    let mut file = 
        File::open(path).await
            .map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

    let mut file_content: String = String::new();

    file.read_to_string(&mut file_content).await
        .map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

    return Ok(file_content);
}



