/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::collections::{LinkedList, BTreeSet, HashSet};
use std::convert::{TryFrom};
use std::net::{IpAddr};
use std::sync::Arc;
use std::time::Duration;
use std::time::Instant;

use nix::poll::{PollFd, PollFlags};

use crate::cfg_resolv_parser::{ResolveConfig, ResolveConfigFamily, ResolveConfEntry};
use crate::error::*;
use crate::query::{QuerySetup, QDnsQueriesRes, QDnsQuery, QDnsQueryRec};
use crate::{write_error, internal_error, internal_error_map};
use crate::query_private::QDnsReq;

use super::common::*;
use super::caches::CACHE;
use super::network::{new_udp, new_tcp, SocketTap};
use super::query_polltaps::{PollTaps, Tap};

/// A main instance which contains all common logic.
pub struct QDns<'req>
{
    /// An instance of the parser /etc/resolv.conf or custom config
    resolvers: Arc<ResolveConfig>,
    /// A pre-ordered list of the requests, if more than one
    ordered_req_list: Vec<QDnsReq<'req>>,
    /// Override options
    opts: QuerySetup,
}

impl<'req> QDns<'req>
{
    /// Initializes new empty storage for requests.
    /// 
    /// In some cases it is good idea to combine different requests, because by default
    /// all requests are performed in parallel. But in some cases it is bad idea.
    /// 
    /// # Arguments
    /// 
    /// * `resolvers` - an [Arc] [ResolveConfig] which contains configuration i.e nameservers
    /// 
    /// * `planned_reqs_len` - how many requests are planned
    /// 
    /// * `opts` - [QuerySetup] additional options or overrides. Use default() for default
    ///     values.
    /// 
    /// # Returns
    /// 
    /// Never panics. Returns Self.
    pub 
    fn make_empty(resolvers: Arc<ResolveConfig>, planned_reqs_len: usize, opts: QuerySetup) -> Self
    {
        return 
            Self
            {
                resolvers: resolvers,
                ordered_req_list: Vec::with_capacity(planned_reqs_len),
                opts: opts,
            };
    }

    /// Adds new request to previously created empty storage for request with [QDns::make_empty].
    /// 
    /// # Arguemnts
    /// 
    /// * `qtype` - a [QType] type of the request
    /// 
    /// * `req_name` - a [Into] [QDnsName] which is target. i.e 'localhost' or 'domain.tld'
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] - Ok with nothing in inner type
    /// 
    /// * [CDnsResult] - Err with error description
    pub 
    fn add_request<R>(&mut self, qtype: QType, req_name: R)
    where R: Into<QDnsName<'req>>
    {
        let qr = QDnsReq::new(req_name.into(), qtype);

        self.ordered_req_list.push(qr);

        return;
    }

    /// This is helper which makes for you an A, AAAA query. The order of A and AAAA and
    /// which from both are allowed is defined in the [ResolveConfig].
    /// 
    /// Use this function directly. Do not use [QDns::make_empty]
    /// 
    /// # Arguments
    /// 
    /// * `resolvers` - an [Arc] [ResolveConfig] which contains configuration i.e nameservers
    /// 
    /// * `req_name` - a [Into] [QDnsName] which is target i.e 'localhost' or 'domain.tld'
    /// 
    /// * `opts` - [QuerySetup] additional options or overrides. Use default() for default
    ///     values.
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] - Ok with Self as inner type
    /// 
    /// * [CDnsResult] - Err with error description
    pub 
    fn make_a_aaaa_request<R>(resolvers: Arc<ResolveConfig>, req_name: R, opts: QuerySetup) -> CDnsResult<Self>
    where R: Into<QDnsName<'req>>
    {
        // store the A and AAAA depending on order
        let reqs: Vec<QDnsReq<'req>> = 
            match resolvers.family
            {
                ResolveConfigFamily::INET4_INET6 => 
                {
                    let req_n: QDnsName = req_name.into();

                    vec![
                        QDnsReq::new(req_n.clone(), QType::A),
                        QDnsReq::new(req_n, QType::AAAA),
                    ]
                },
                ResolveConfigFamily::INET6_INET4 => 
                {
                    let req_n: QDnsName = req_name.into();

                    vec![
                        QDnsReq::new(req_n.clone(), QType::AAAA),
                        QDnsReq::new(req_n, QType::A),
                    ]
                },
                ResolveConfigFamily::INET6 => 
                {
                    vec![
                        QDnsReq::new(req_name.into(), QType::AAAA),
                    ]
                },
                ResolveConfigFamily::INET4 => 
                {
                    vec![
                        QDnsReq::new(req_name.into(), QType::A),
                    ]
                }
                _ =>
                {
                    // set default
                    let req_n: QDnsName = req_name.into();

                    vec![
                        QDnsReq::new(req_n.clone(), QType::A),
                        QDnsReq::new(req_n, QType::AAAA),
                    ]
                }
            };

        
        
        return Ok(
            Self
            {
                resolvers: resolvers,
                ordered_req_list: reqs,
                opts: opts,
            }
        );
    }

    /// Runs the created query/ies
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] with [QDnsQueriesRes] which may contain results
    /// 
    /// * [CDnsResult] with error
    /// 
    /// Should not panic. MT-safe.
    pub 
    fn query(mut self) -> QDnsQueriesRes
    {
        // check if we need to measure time
        let now = 
            if self.opts.measure_time == true
            {
                Some(Instant::now())
            }
            else
            {
                None
            };

        let mut qres: QDnsQueriesRes = QDnsQueriesRes::DnsNotAvailable;

        // determine where to look firstly i.e file -> bind, bind -> file
        // or bind only, or file only
        if self.resolvers.lookup.is_file_first()
        {
            match self.lookup_file(now.as_ref())
            {
                Ok(r) =>
                    qres.extend(r),
                Err(e) =>
                    write_error!("{}", e),
            }

            // if something left unresolved, try ask internet
            if self.ordered_req_list.is_empty() == false && self.resolvers.lookup.is_bind() == true
            {
                match self.process_request(now.as_ref())
                {
                    Ok(r) => 
                        qres.extend(r),
                    Err(e) =>
                        write_error!("{}", e),
                }
            }
        }
        else
        {
            match self.process_request(now.as_ref())
            {
                Ok(r) => 
                    qres.extend(r),
                Err(e) =>
                    write_error!("{}", e),
            }


            if self.ordered_req_list.is_empty() == false && self.resolvers.lookup.is_file() == true
            {
                match self.lookup_file(now.as_ref())
                {
                    Ok(r) => 
                        qres.extend(r),
                    Err(e) =>
                        write_error!("{}", e),
                }
            }
        }

        return qres;
    }

    /// Returns timeout
    fn get_timeout(&self) -> Duration
    {
        if let Some(timeout) = self.opts.timeout
        {
            return Duration::from_secs(timeout as u64);
        }
        else
        {
            return Duration::from_secs(self.resolvers.timeout as u64);
        }
    }

    /// Searches in /etc/hosts
    fn lookup_file(&mut self, now: Option<&Instant>) -> CDnsResult<QDnsQueriesRes>
    {
        // check if the it is overriden
        if self.opts.ign_hosts == false
        {
            let hlist = CACHE.clone_host_list()?;

            let mut dnsquries: LinkedList<QDnsQuery> = LinkedList::new();

            self.ordered_req_list.retain(|req| 
                {
                    match *req.get_type()
                    {
                        QType::A | QType::AAAA => 
                        {
                            let req_name = String::from(req.get_req_name());
    
                            if let Some(res) = hlist.search_by_fqdn(req.get_type(), req_name.as_str())
                            {
                                // create storage for response
                                let drp = 
                                    match DnsResponsePayload::new_local(*req.get_type(), res)
                                    {
                                        Ok(r) => r,
                                        Err(e) =>
                                        {
                                            write_error!("{}", e);

                                            return true;
                                        }
                                    };

                                // store to list
                                dnsquries.push_back(
                                    QDnsQuery::from_local(drp, now)
                                );

                                return false;
                            }
                            else
                            {
                                return true;
                            }
                        },
                        QType::PTR => 
                        {
                            let ip: IpAddr = 
                                match IpAddr::try_from(req.get_req_name())
                                {
                                    Ok(r) => r,
                                    Err(e) =>
                                    {
                                        // just skip
                                        write_error!("{}", e);

                                        return true;
                                    }
                                };
    
                            if let Some(r) = hlist.search_by_ip(&ip)
                            {   
                                // create storage for response
                                let drp = 
                                    match DnsResponsePayload::new_local(QType::PTR, r)
                                    {
                                        Ok(r) => r,
                                        Err(e) =>
                                        {
                                            write_error!("{}", e);

                                            return true;
                                        }
                                    };

                                // store to list
                                dnsquries.push_back(
                                    QDnsQuery::from_local(drp, now)
                                ); 

                                return false;
                            }
                            else
                            {
                                return true;
                            }
                        },
                        _ => 
                        {
                            // just skip, don't throw error
                            return true;
                        }
                    }
                }
            );

            /*for req in self.ordered_req_list.iter()
            {   
            }*/

            return Ok( QDnsQueriesRes::from(dnsquries) );
        }
        else
        {
            return Ok(QDnsQueriesRes::DnsNotAvailable);
        }
    }

    /// Creates socket based on config and flag. If `force_tcp` is set then Tcp tap will
    /// be created.
    fn create_socket(&self, force_tcp: bool, resolver: &ResolveConfEntry) -> CDnsResult<Box<dyn SocketTap>>
    {
        // create socket, if `force_tcp` is set then the attempt to switch from UDP to TCP
        if self.resolvers.option_flags.is_force_tcp() == true || force_tcp == true
        {
            return new_tcp(resolver.get_resolver_ip(), 53, resolver.get_adapter_ip());
        }
        else
        {
            return new_udp(resolver.get_resolver_ip(), 53, resolver.get_adapter_ip());
        };
    }

    /// Creates the sockets based on the config i.e if socket reopen is needed.
    fn create_sockets(
        &self, 
        resolver: &ResolveConfEntry, 
        requery_list: Option<LinkedList<DnsRequestHeader>>,
        force_tcp: bool,
    ) -> CDnsResult<PollTaps>
    {
        let mut taps: PollTaps = PollTaps::new_with_capacity(self.ordered_req_list.len());
        

        let force_tcp: bool = 
            force_tcp == true || self.resolvers.option_flags.is_force_tcp() == true;

        if let Some(requery) = requery_list
        {
            let mut ids: HashSet<u16> = HashSet::with_capacity(requery.len());

            

            for mut req in requery
            {
                if self.resolvers.option_flags.is_reopen_socket() == true || taps.len() == 0
                { 
                    // create socket and store
                    let tap = 
                        self.create_socket(force_tcp, resolver)?;

                    // make sure the ID is uniq
                    loop
                    {
                        req.regenerate_id();

                        if ids.insert(req.get_id()) == true
                        {
                            break;
                        }
                    }

                    taps.push(Tap::new(tap, req));
                }
                else
                {
                    taps.push_to_last(req);
                }
            }

            
        }
        else
        {
            let mut ids: HashSet<u16> = HashSet::with_capacity(self.ordered_req_list.len());

            // form the list of reqests
            for req in self.ordered_req_list.iter()
            {
                let mut drh_req = DnsRequestHeader::from_qdns_req(req, self.resolvers.as_ref())?;

                // make sure the ID is uniq
                loop
                {
                    if ids.insert(drh_req.get_id()) == true
                    {
                        break;
                    }

                    drh_req.regenerate_id();
                }

                if self.resolvers.option_flags.is_reopen_socket() == true || taps.len() == 0
                { 
                    // create socket and store
                    let tap = 
                        self.create_socket(force_tcp, resolver)?;

                    taps.push(Tap::new(tap, drh_req));
                }
                else
                {
                    taps.push_to_last(drh_req);
                }
            } // for

           
        }

        return Ok(taps);
    }

    /// Returns the [QDnsQueryRec::Ok] only if no other type did not occure
    fn get_result(responses: &LinkedList<QDnsQuery>) -> QDnsQueryRec
    {
        let mut resp = QDnsQueryRec::Ok;

        for r in responses.iter()
        {
            if r.is_ok() == false
            {
                resp = r.status;

                return resp;
            }
        }

        return resp;
    }

    /// Accesses the first record in responses and gets the Authorative flag
    fn get_authorative(responses: &LinkedList<QDnsQuery>) -> bool
    {
        return responses.front().map_or(false, |r| r.aa);
    }

    /// Quering nameservers
    fn process_request(&mut self, now: Option<&Instant>) -> CDnsResult<QDnsQueriesRes>
    {
        let mut qresponses: LinkedList<QDnsQuery> = LinkedList::new();

       // let mut resolved: HashSet<&'req QDnsReq> = HashSet::with_capacity(self.ordered_req_list.len());

        for resolver in self.resolvers.get_resolvers_iter()?
        {
            let qresp = 
                if self.resolvers.option_flags.is_no_parallel() == true
                {
                    match self.no_parallel(now, resolver, None, false)
                    {
                        Ok(qresp) => qresp,
                        Err(e) =>
                        {
                            write_error!("{}", e);
                            continue;
                        }
                    }
                }
                else
                {
                    match self.parallel(now, resolver, None, false)
                    {
                        Ok(qresp) => qresp,
                        Err(e) =>
                        {
                            write_error!("{}", e);
                            continue;
                        }
                    }
                };

            // check if results is ok
            let dnsres = Self::get_result(&qresp);
            let aa = Self::get_authorative(&qresp);

            qresponses.extend(qresp);

            if dnsres.try_next_nameserver(aa) == false
            {
                break;
            }
            
        }

        return Ok(QDnsQueriesRes::from(qresponses));
    }

    /// Runs sequentually
    fn no_parallel(
        &self, 
        now: Option<&Instant>, 
        resolver: &ResolveConfEntry,
        requery: Option<LinkedList<DnsRequestHeader>>,
        force_tcp: bool
    ) -> CDnsResult<LinkedList<QDnsQuery>>
    {
        // -- this is not last nameserver, continue --
        // a linked list of responses
        let mut responses: LinkedList<QDnsQuery> = LinkedList::new();

        // a storage fot truncated request
        let mut truncated_list: LinkedList<DnsRequestHeader> = LinkedList::new();

        // form the list of reqests binded to taps
        let taps_list = 
            self.create_sockets(resolver, requery, force_tcp)?;

        for ptap in taps_list.into_iter()
        {
            let (mut tap, qdnsr) = ptap.unwrap_inner();

            // open connection, with timeout because working in non parallel mode
            tap.connect(Some(self.get_timeout()))?;

            for req in qdnsr
            {
                // convert to byte packet
                let pkt = req.to_bytes()?;

                // send
                tap.send(pkt.as_slice())?;

                // read response from socket
                let ans = self.read_response(tap.as_mut())?;

                // verify the request with response
                match ans.verify(&req)
                {
                    Ok(_) => {},
                    Err(ref e) 
                        if e.err_code == CDnsErrorType::MessageTruncated =>
                    {
                        // message was received truncated, then add to list
                        // to resend query via TCP if not TCP
                        write_error!("{}", e);

                        truncated_list.push_back(req);
                    },
                    Err(e) => 
                        return Err(e),
                };

                // verified
                let resp = QDnsQuery::from_response(tap.get_remote_addr(), ans, now)?;

                responses.push_back(resp);
            } // for 

        } // for 

        if truncated_list.is_empty() == false
        {
            // befor leaving we need to finish resolving what was truncated
            if force_tcp == true || self.resolvers.option_flags.is_force_tcp() == true
            {
                // throw error
                internal_error!(CDnsErrorType::MessageTruncated, "Message is truncated even using TCP. Give up.");
            }

            let res = 
                self.no_parallel(now, resolver, Some(truncated_list), true)?;

            responses.extend(res);
        }
        
        return Ok(responses);
    }

    /// Runs in parallel
    fn parallel(
        &self, 
        now: Option<&Instant>, 
        resolver: &ResolveConfEntry, 
        requery: Option<LinkedList<DnsRequestHeader>>,
        force_tcp: bool
    ) -> CDnsResult<LinkedList<QDnsQuery>>
    {
        // -- this is not last nameserver, continue --
        // a linked list of responses
        let mut responses: LinkedList<QDnsQuery> = LinkedList::new();

        // form the list of reqests binded to taps
        let mut taps_list = self.create_sockets(resolver, requery, force_tcp)?;

        // now the `taps` contains a pairs of socket - request/s which is needed
        // to satisfy the REOPEN if remote nameserver does not support multiple requests
        // from same port

        // send everyting and wait until all responses come back
        for ptap in taps_list.get_mut_iter()
        {
            // open connection, no timeout as the pollfd woring in sock async mode
            ptap.get_tap().connect(None)?;

            let pkts = ptap.generate_packets()?;

            // send everything which is in the reqs_map
            for pkt in pkts
            {
                ptap.get_tap().send(pkt.as_slice())?;
            }
        } // for

        // a storage fot truncated request
        let mut truncated_list: LinkedList<DnsRequestHeader> = LinkedList::new();

        // poll, receive responses
        let poll_data = 
            self.run_poll(taps_list, &mut truncated_list, now)?;

        // store responses to list
        responses.extend(poll_data);

        if truncated_list.is_empty() == false
        {
            // befor leaving we need to finish resolving what was truncated
            if force_tcp == true || self.resolvers.option_flags.is_force_tcp() == true
            {
                // throw error
                internal_error!(CDnsErrorType::MessageTruncated, "Message is truncated even using TCP. Give up.");
            }

            let res = 
                self.parallel(now, resolver, Some(truncated_list), true)?;

            responses.extend(res);
        }
        
        return Ok(responses);
    }

    /// Reads response from tap
    fn read_response(&self, socktap: &mut (dyn SocketTap)) -> CDnsResult<DnsRequestAnswer>
    {
        let mut rcvbuf = vec![0_u8; 1024];

        // receive message
        socktap.recv(rcvbuf.as_mut_slice())?;

        // parsing response to structure
        let ans = DnsRequestAnswer::try_from(rcvbuf.as_slice())?;

        return Ok(ans);
    }

    /// Runs the poll(2) for parallel requests
    fn run_poll(
        &self, 
        mut taps_list: PollTaps,
        trunc_list: &mut LinkedList<DnsRequestHeader>,
        now: Option<&Instant>
    ) ->CDnsResult<LinkedList<QDnsQuery>>
    {
        // a collection of responses
        let mut responses: LinkedList<QDnsQuery> = LinkedList::new();

        // calculate timeout
        let mut timeout: nix::libc::c_int = (self.resolvers.timeout * 1000) as i32;

        let mut poll_list: Vec<PollFd> = taps_list.get_pollfd_list();
        let mut reqs_await_list: BTreeSet<u16> = taps_list.get_reqs_list();

        if poll_list.len() == 0
        {
            internal_error!(CDnsErrorType::InternalError, "Poll list is empty!");
        }

        loop
        {
            // if no requests leftunserved, then exit run_poll()
            if reqs_await_list.is_empty() == true
            {
                // leave loop because all requests were received
                break;
            }

            // debug
            if timeout <= 0
            {
                write_error!("Somehow the timeout is 0, setting to 5000ms");
                timeout = 5000;
            }

            // poll 
            let poll_res = 
                nix::poll::poll(poll_list.as_mut_slice(), timeout)
                    .map_err(|e| internal_error_map!(CDnsErrorType::IoError, "{}", e))?;

            if poll_res == 0
            {
                // timeout
                internal_error!(CDnsErrorType::RequestTimeout, "timeout poll");
            }
            else
            {
                for pfd in poll_list.iter()
                {
                    let revents = pfd.revents();

                    if revents.is_none() == true
                    {
                        panic!("Crate NIX returned None due to unknown data returned by kernel!\
                            Try update nix crate! Probably it is outdated!");
                    }

                    let revents = revents.unwrap();

                    // there is something to read
                    if revents.contains(PollFlags::POLLIN) == true
                    {
                        // get the tap (socket) by the FD
                        let ptap = taps_list.find_tap_by_pollfd(pfd);

                        // read response from socket
                        let ans = self.read_response(ptap.get_tap())?;

                        let ans_id = ans.header.id;

                        // check if received request is regestered for the found tap
                        // if ID is invalid (request does not exist) discard all results
                        let req = 
                            ptap.find_request(ans_id)
                                .ok_or_else(|| 
                                    internal_error_map!(CDnsErrorType::RespIdMismatch, "id: '{}' not found", ans_id)
                                )?;

                        // verify the request with response
                        match ans.verify(&req)
                        {
                            Ok(_) => {},
                            Err(ref e) 
                                if e.err_code == CDnsErrorType::MessageTruncated =>
                            {
                                // message was received truncated, then add to list
                                // to resend query via TCP if not TCP
                                write_error!("{}", e);

                                trunc_list.push_back(req.clone());
                            },
                            Err(e) => 
                                return Err(e),
                        }

                        drop(req);

                        // verified
                        let resp = QDnsQuery::from_response(ptap.get_tap().get_remote_addr(), ans, now)?;

                        responses.push_back(resp);

                        // remove the req from list
                        if reqs_await_list.remove(&ans_id) == false
                        {
                            internal_error!(
                                CDnsErrorType::InternalError, 
                                "request ID: '{}' for some reason can not be removed", 
                                ans_id
                            );
                        }
                    }
                } // for
            }

        } // loop

        return Ok(responses);
    }
}


#[test]
fn test_ip2pkt()
{
    use std::time::Instant;
    use std::net::{IpAddr, Ipv4Addr};
    
    let test = IpAddr::V4(Ipv4Addr::new(8, 8, 8, 8));

    let now = Instant::now();

    let res = ip2pkt(&test);

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_ok(), true, "err: {}", res.err().unwrap());

    let res = res.unwrap();
    let ctrl = b"\x01\x38\x01\x38\x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\x61\x00";

    assert_eq!(res.as_slice(), ctrl);
}


#[test]
fn test_byte2hexchar()
{
    assert_eq!(byte2hexchar(1), 0x31);
    assert_eq!(byte2hexchar(9), 0x39);
    assert_eq!(byte2hexchar(10), 'a' as u8);
    assert_eq!(byte2hexchar(15), 'f' as u8);
}


#[test]
fn reverse_lookup_test()
{
    use std::time::Instant;
    
    let ipp: IpAddr = "8.8.8.8".parse().unwrap();
    let test = QDnsName::from(&ipp);

    let resolvers = CACHE.clone_resolve_list().unwrap();
    let mut query_setup = QuerySetup::default();
    query_setup.set_measure_time(true);

    let now = Instant::now();

    let mut dns_req = 
        QDns::make_empty(resolvers, 1, query_setup);

    dns_req.add_request(QType::PTR, test);
    let res = dns_req.query();

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_results(), true, "{}", res);

    println!("{}", res);

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            let rec = &res[0];

            //assert_eq!(rec.server.as_str(), "/etc/hosts");
            assert_eq!(rec.status, QDnsQueryRec::Ok);

            assert_eq!(rec.resp.len(), 1);
            assert_eq!(rec.resp[0].rdata, DnsRdata::PTR{ fqdn: "dns.google".to_string() });
            
        },
        _ => assert_eq!(true, false, "expected DnsResultSingle"),
    }
}

#[test]
fn reverse_lookup_hosts_test()
{
    use std::time::Instant;

    let ipp: IpAddr = "127.0.0.1".parse().unwrap();
    let test = QDnsName::from(&ipp);
    
    let now = Instant::now();

    let mut query_setup = QuerySetup::default();
    query_setup.set_measure_time(true);

    let resolvers = CACHE.clone_resolve_list().unwrap();

    let mut dns_req = 
        QDns::make_empty(resolvers, 1, query_setup);

    dns_req.add_request(QType::PTR, test);

    let res = dns_req.query();
    
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_results(), true);

    println!("{}", res);

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            let rec = &res[0];

            assert_eq!(rec.server.as_str(), "/etc/hosts");
            assert_eq!(rec.status, QDnsQueryRec::Ok);

            assert_eq!(rec.resp.len(), 1);
            assert_eq!(rec.resp[0].rdata, DnsRdata::PTR{ fqdn: "localhost".to_string() });
            
        },
        _ => assert_eq!(true, false, "expected DnsResultSingle"),
    }
}


#[test]
fn reverse_lookup_a()
{
    use std::time::Instant;

    let test = QDnsName::from("dns.google");

    let mut query_setup = QuerySetup::default();
    query_setup.set_measure_time(true);


    let resolvers = CACHE.clone_resolve_list().unwrap();
    let res = 
        QDns::make_a_aaaa_request(resolvers, test, query_setup).unwrap();
    
    
    let now = Instant::now();
    let res = res.query();
    

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

  /*  assert_eq!(res.is_ok(), true, "{}", res.err().unwrap());
    
    let res = res.unwrap();*/
    println!("{}", res);
}

