/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

pub mod cdns_custom_output
{
    use std::{str, fmt::{self, Write}};

    use crate::{sync::mutex::Mutex, read_only_cell::ReadOnlyCell};

    struct DummyOutputAdapter{}

    unsafe impl Send for DummyOutputAdapter {}

    impl Write for DummyOutputAdapter
    {
        fn write_str(&mut self, _s: &str) -> fmt::Result 
        {
            return Ok(());
        }
    }

    impl DummyOutputAdapter
    {
        fn new() -> Self
        {
            return Self{};
        }
    }

    static IS_INITIALIZED: ReadOnlyCell<bool> = 
        ReadOnlyCell::new_not_loceked(true, "is_custom_output_init");


    lazy_static!{
        /// An instance which implements the io::Write + Send
        static ref WRITABLE_BUFFER: Mutex<Box<dyn Write + Send>> = Mutex::new(Box::new(DummyOutputAdapter::new()));
        static ref LAST_ERROR: Mutex<Option<String>> = Mutex::new(None);
    }

    /// Initializes the output to the custom buffer.
    /// This function does not check if you set the output twice and panics!
    /// 
    /// # Arguments
    /// 
    /// * `w` - a dynamic instance in [Box] which implements [Write] and [Send]
    /// 
    /// # Panics
    /// 
    /// If the mutex was poisoned, may panic. Also will panic, if it will be 
    /// attempted to reinitialize the instance. If this is a problem, use 
    /// [initialize_safe]
    pub 
    fn initialize(w: Box<dyn Write + Send>)
    {
        // occure the lock
        let mut lock = WRITABLE_BUFFER.lock().unwrap();

        // set new instance
        *lock = w;

        // lock the status
        unsafe { IS_INITIALIZED.init(true) };

        return;
    }

    /// Initializes the output to the custom buffer.
    /// This function does not check if you set the output twice.
    /// 
    /// # Arguments
    /// 
    /// * `w` - a dynamic instance in [Box] which implements [Write] and [Send]
    /// 
    /// # Returns
    /// 
    /// * [bool] true if mutex is not poisoned
    /// 
    /// * [bool] false if mutex was poisoned
    /// 
    /// # Panics
    /// 
    /// Does not panic!
    pub 
    fn initialize_safe(w: Box<dyn Write + Send>) -> bool
    {
        //occure lock
        match WRITABLE_BUFFER.lock()
        {
            Ok(mut r) => 
            {
                // check if instance is initialized
                if IS_INITIALIZED.is_init() == false
                {
                    // set instance
                    *r = w;

                    // lock
                    unsafe { IS_INITIALIZED.init(true) };

                    return true;
                }

                return false;
            },
            Err(_e) => return false,
        }
    }

    /// Returns the status. Once initialized, can not be deinitialized.
    /// 
    /// # Returns
    /// 
    /// * [bool] true, if initialized
    /// 
    /// * [bool] false, if not initialized
    /// 
    /// # Panics
    /// 
    /// May panic is mutex is poisoned!
    pub 
    fn is_initialized() -> bool
    {
        return IS_INITIALIZED.is_init();
    }

    /// Returns the status of the mutex answering the quiestion if
    /// the mutex which guards the access is poisoned.
    /// 
    /// # Returns
    /// 
    /// * [bool] true, if poisoned
    /// 
    /// * [bool] false, if not poisoned
    /// 
    /// # Panics
    /// 
    /// Never panics!
    pub 
    fn is_poisoned() -> bool
    {
        return WRITABLE_BUFFER.is_poisoned();
    }

    /// Moves the last error from storage.
    pub 
    fn last_error() -> Option<String>
    {
        match LAST_ERROR.lock()
        {
            Ok(mut r) => return (*r).take(),
            Err(_e) => return None,
        }
    }

    #[inline]
    pub 
    fn cdns_custom_output_write(s: &str)
    {
        if IS_INITIALIZED.is_init() == false
        {
            // just ignore
            return;
        }

        match WRITABLE_BUFFER.lock()
        {
            Ok(mut r) =>
            {
                let err = (*r).write_str(s);

                if let Err(e) = err
                {
                    match LAST_ERROR.lock()
                    {
                        Ok(mut r) => *r = Some(e.to_string()),
                        Err(_e) => {}
                    }
                }
            }
            Err(_e) => {}
        }

        return;
    }
}

use crate::error::Writer;

/*
n = force_none
c = force_custom
d = debug_assertions
t = test

V - set
X - not allowed
O - does not care, ignored
     n c d t
none V X O O
std  X X O O
cus  X V O O
*/

#[cfg(
    all(
        feature = "no_error_output",
        feature = "custom_error_output",
    )
)]
compile_error!("It is not allowed to use features 
    'no_error_output' and 'custom_error_output' simultaniously");


// ------ OUTPUT to stdout and stderr ------ FEAUTE: no + any(debug_assertions, test)

/// A macro for printing error to stderror all error messages generated by
/// the library.
#[cfg(
    all(
        not(feature = "no_error_output"),
        not(feature = "custom_error_output"),
        any( debug_assertions, test)
    )
)]
#[macro_export]
macro_rules! write_error 
{
    ($($arg:tt)*) => (
        eprintln!($($arg)*)
    )
}

#[cfg(
    all(
        not(feature = "no_error_output"),
        not(feature = "custom_error_output"),
        any( debug_assertions, test)
    )
)]
pub 
fn sync_log_writer(w: Writer)
{
    eprintln!("{}", s);
}


// ------ OUTPUT to custom ----- FEATURE: custom_error_output

#[cfg(
    all(
        not(feature = "no_error_output"),
        feature = "custom_error_output",
        any(debug_assertions, test)
    )
)]
pub use cdns_custom_output::*;


/// A macro for printing to progrmas buffer all error messages generated by
/// the library.
#[cfg(
    all(
        not(feature = "no_error_output"),
        feature = "custom_error_output",
        any(debug_assertions, test)
    )
)]
#[macro_export]
macro_rules! write_error 
{
    ($($arg:tt)*) => (
        $crate::sync::log::cdns_custom_output::cdns_custom_output_write(format!($($arg)*).as_str())
    )
}

#[cfg(
    all(
        not(feature = "no_error_output"),
        feature = "custom_error_output",
        any(debug_assertions, test)
    )
)]
pub 
fn sync_log_writer(w: Writer)
{
    if w.is_some() == true
    {
        cdns_custom_output_write(unsafe { w.get_str() });
    }
}


// ----- OUTPUT to null ----- FEATURE: no_error_output

/*#[cfg(
    all(
        not(feature = "custom_error_output"),
        feature = "no_error_output",
    )
)]
#[inline]
pub 
fn empty_func() { return; }*/

/// A macro for printing to nowhere all error messages generated by
/// the library.
#[cfg(
    all(
        not(feature = "custom_error_output"),
        feature = "no_error_output",
    )
)]
#[macro_export]
macro_rules! write_error 
{
    ($($arg:tt)*) => (
        {}
    )
}

#[cfg(
    all(
        not(feature = "custom_error_output"),
        feature = "no_error_output",
    )
)]
pub 
fn sync_log_writer(_w: Writer)
{
    return;
}
