/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/*
                                1  1  1  1  1  1
  0  1  2  3  4  5  6  7  8  9  0  1  2  3  4  5
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | ID |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|QR|   Opcode  |AA|TC|RD|RA|   Z    |   RCODE   |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | QDCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | ANCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | NSCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | ARCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+

                                1  1  1  1  1  1
  0  1  2  3  4  5  6  7  8  9  0  1  2  3  4  5
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|                                               |
/                       QNAME                   /
/                                               /
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|                       QTYPE                   |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|                       QCLASS                  |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+

*/

/// This file contains the common structures and constants of DNS protocol and 
/// other parts of the code. Also it contains a parser from/to network packet.

use std::{io::{Cursor, Read, Write}, mem::size_of, net::{Ipv4Addr, Ipv6Addr}};
use byteorder::{BigEndian, ReadBytesExt, WriteBytesExt};

use crate::error::*;
use crate::{internal_error, internal_error_map};


pub use crate::common::*;

/// Verifies the sequence of labels coding and copies data from pkt to
/// buffer.
/// 
/// # Arguments
/// 
/// * `pkt` - a reference to the [Cursor]
/// 
/// # Returns 
/// 
/// * [CDnsResult] with data copied to new [Vec<u8>]
pub 
fn pkt2name(pkt: &mut Cursor<&[u8]>) -> CDnsResult<Vec<u8>>
{
    let qname_s = pkt.position() as usize;

    loop
    {
        let cnt = pkt.read_u8().map_err(map_read_err)?;

        if cnt == 0
        {
            break;
        }
        else
        {
            for _ in 0..cnt
            {
                let c = pkt.read_u8().map_err(map_read_err)?;
                if c == 0
                {
                    internal_error!(
                        CDnsErrorType::DnsResponse, 
                        "incorrectly encoded QNAME in response, found '0' at offset: '{}'",
                        pkt.position()
                    );
                }
            }
        }
    }

    // temporary store the position of the cursor
    let cur_pos = pkt.position() as usize;

    if (cur_pos - qname_s) <= 1
    {
        internal_error!(
            CDnsErrorType::DnsResponse, 
            "read name is too short"
        );
    }

    /*pkt.set_position(qname_s);
    
    // creating buffer for QNAME
    let mut req: Vec<u8> = vec![0_u8; (cur_pos - qname_s) as usize];
    pkt.read_exact(&mut req).map_err(map_read_err)?;

    // restore position
    pkt.set_position(cur_pos);*/
    

    return Ok(pkt.get_ref()[qname_s..cur_pos].to_vec());
}


/// This function converts a QNAME (encoded domains) to the String FQDN.
pub 
fn name2str(pkt: &mut Cursor<&[u8]>, mut opt_rdlen: Option<u16>) -> CDnsResult<String>
{
    let cur_in_pos = pkt.position();
    //let mut p_comp: u8 = 0;
    let mut comp: u8;
    let mut output: Vec<String> = Vec::with_capacity(6);

    loop
    {
        if let Some(rdlen) = opt_rdlen
        {
            if pkt.position() - cur_in_pos >= rdlen as u64
            {
                return Ok(output.join("."));
            }
        }

        // reading word from pkt to detect the type of message
       // p_comp = comp;
        comp = pkt.read_u8().map_err(map_read_err)?;

        /*if p_comp > 0 && comp > 0
        {
            output.push('.');
        }*/

        let msb = comp & 0xC0;

        if msb == 0xC0
        {
            if opt_rdlen.is_none() == true
            {
                opt_rdlen = Some(2);
            }

            let offset1: u16 = ((comp & !0xC0) as u16) << 8;// | pkt.read_u8().map_err(map_read_err)? as u16;
            let offset2: u16 = pkt.read_u8().map_err(map_read_err)? as u16;

            let offset = offset1 | offset2;

            //println!("debug: 1: {} 2: {} offset: {}", offset1, offset2, offset);

            // C0 0...
            if offset as usize >= pkt.get_ref().len()
            {
                internal_error!(
                    CDnsErrorType::DnsResponse, 
                    "incoreclty formated packet: offset: '{}' > len: '{}'", 
                    offset, pkt.get_ref().len()
                );
            }

            // save current cursor position
            let cur_pos = pkt.position();
            pkt.set_position(offset as u64);

            // converting the name to FQDN
            output.push(name2str(pkt, None)?);
            //output.push_str(cache.get(&offset).ok_or_else(|| internal_error_map!(CDnsErrorType::InternalError, "cache mismatch"))?);

            // restoring cursor positon
            pkt.set_position(cur_pos);
        }
        else if msb == 0x00 
        {
            if comp == 0
            {
                if let Some(rdlen) = opt_rdlen
                {
                    let dif = pkt.position() - cur_in_pos;

                    if rdlen as u64 != dif
                    {
                        internal_error!(CDnsErrorType::DnsResponse, "incorrect rdlen: '{}', exp: '{}'", rdlen, dif);
                    }
                }

                return Ok(output.join("."));
            }
            else
            {
                let mut tmp: String = String::with_capacity(comp as usize);

                for _ in 0..comp
                {
                    let c = pkt.read_u8().map_err(map_read_err)?;
                    if c == 0
                    {
                        internal_error!(CDnsErrorType::DnsResponse, "protocol violation, incorrectly encoded QNAME in response");
                    }

                    tmp.push(c as char);
                }

                output.push(tmp);
            }
        }
        else
        {
            internal_error!(CDnsErrorType::DnsResponse, "incorrect compression: {:x}", msb);
        }

    } // loop
}


impl DnsRdata
{
    /// Reads the RData from pkt and converts it into QType container
    /// 
    /// # Arguments
    /// 
    /// * `pkt` - a [Cursor] mutable reference to raw DNS reply data
    /// 
    /// * `rlen` - a length of RData section
    /// 
    /// * `qtype` - a [QType] reference to the response answer Qtype
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] with Self or error
    fn convert(pkt: &mut Cursor<&[u8]>, rlen: u16, qtype: &QType) -> CDnsResult<Self>
    {
        match &qtype
        {
            QType::A => 
            {
                // read 4 bytes (octets)
                if rlen != 4
                {
                    internal_error!(CDnsErrorType::DnsResponse, "ipv4 len expected: '4', got: '{}'", rlen);
                }

                let ip = pkt.read_u32::<BigEndian>().map_err(map_read_err)?;
                return Ok(Self::A{ ip: Ipv4Addr::from(ip) });
            },
            QType::AAAA => 
            {
                // read rlen bytes (octets)
                if rlen != 16
                {
                    internal_error!(CDnsErrorType::DnsResponse, "ipv6 len expected: '16', got: '{}'", rlen);
                }

                let ip = pkt.read_u128::<BigEndian>().map_err(map_read_err)?;
                return Ok(Self::AAAA{ ip: Ipv6Addr::from(ip) });
            },
            QType::MX =>
            {
                return Ok(
                    Self::MX
                    { 
                        preference: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                        exchange: name2str(pkt, Some(rlen - 2))?,
                    }
                );
            },
            QType::CNAME =>
            {
                return Ok(
                    Self::CNAME{ fqdn: name2str(pkt, Some(rlen))? }
                );
            },
            QType::PTR =>
            {
                return Ok(
                    Self::PTR{ fqdn: name2str(pkt, Some(rlen))? }
                );
            },
            QType::NS =>
            {
                return Ok(
                    Self::NS{ fqdn: name2str(pkt, Some(rlen))? }
                );
            },
            QType::SOA =>
            {
                return Ok(
                    Self::SOA
                    {   
                        soa:
                            DnsSoa
                            {
                                /// primary name server
                                pnm: name2str(pkt, None)?, 
                                /// responsible authority mailbox
                                ram: name2str(pkt, None)?,
                                /// serial number
                                serial: pkt.read_u32::<BigEndian>().map_err(map_read_err)?,
                                /// refresh interval
                                interv_refr: pkt.read_u32::<BigEndian>().map_err(map_read_err)?,
                                /// retry interval
                                interv_retry: pkt.read_u32::<BigEndian>().map_err(map_read_err)?,
                                /// expire limit
                                expire_limit: pkt.read_u32::<BigEndian>().map_err(map_read_err)?,
                                /// minimum TTL
                                min_ttl: pkt.read_u32::<BigEndian>().map_err(map_read_err)?,
                            }
                    }
                );
            },
            _ =>
            {
                let ipos = pkt.position();
                let npos: u64 = pkt.position() + rlen as u64;
                let buf_size = npos - ipos;
             
                let mut data: Vec<u8> = vec![0_u8; buf_size as usize];//pkt.get_ref()[s..s+rlen as usize].to_vec();
                pkt.read_exact(&mut data).map_err(map_read_err)?;
             
                pkt.set_position(npos);
                //   pkt.set_position((s+rlen as usize) as u64);
                /*QType::MD | QType::MF | QType::SOA | QType::MB | QType::MG | 
            QType::MR | QType::NULL | QType::WKS | QType::HINFO | QType::TXT |
            QType::AFSDB | QType::KEY | QType::CERT | QType::DS | QType::RRSIG |
            QType::NSEC | QType::DNSKEY | QType::NSEC3 | QType::NSEC3PARAM | 
            QType::CDS | QType::CDNSKEY | QType::OPENPGPKEY  */

                return Ok(Self::UNKNOWN{ data: data });
            }
        }
    }
}


impl DnsResponsePayload
{
    fn new(pkt: &mut Cursor<&[u8]>) -> CDnsResult<Self>
    {
        let name = name2str(pkt, None)?;
        let qtype = QType::u16_to_type(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?;
        let qclass = QClass::u16_to_class(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?;
        let ttl = pkt.read_i32::<BigEndian>().map_err(map_read_err)?;
        let rdlength = pkt.read_u16::<BigEndian>().map_err(map_read_err)?;
        let rdata = DnsRdata::convert(pkt, rdlength, &qtype)?;

        return Ok(
            DnsResponsePayload
            {
                name: name,
                dtype: qtype,
                class: qclass,
                ttl: ttl,
                rdlength: rdlength,
                rdata: rdata,
            }
        );
    }
}


impl TryFrom<&[u8]> for DnsRequestAnswer
{
    type Error = CDnsError;

    fn try_from(value: &[u8]) -> Result<Self, Self::Error> 
    {
        return DnsRequestAnswer::parse(value);
    }
}

impl DnsRequestAnswer
{
    /// Parses the raw data from received packet.
    /// 
    /// # Arguments
    /// 
    /// * `ans` - the received data
    fn parse(ans: &[u8]) -> CDnsResult<Self>
    {
        // creating cursor
        let mut pkt = Cursor::new(ans);

        // parsing header
        let id = pkt.read_u16::<BigEndian>().map_err(map_read_err)?;

        let status = pkt.read_u16::<BigEndian>().map_err(map_read_err)?;

        let header = 
            DnsHeader
            {
                id: id,
                status: StatusBits::from_bits(status).ok_or_else(|| 
                    internal_error_map!(CDnsErrorType::DnsResponse, "unknown status bits: '{}'", status)
                )?,
                qdcount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                ancount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                nscount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                arcount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
            };

        // todo!("what about truncated messages");

        // check if there is no more than 1 answer
        if header.status.contains(StatusBits::QR_RESP) == false
        {
            internal_error!(
                CDnsErrorType::DnsResponse, 
                "incorret QR flag in STATUS register of response, expected 1 got 0"
            );
        }
       /*else if header.ancount > 1
        {
            internal_error!(
                CDnsErrorType::DnsRespose, 
                "too many responses = '{}', multiple responses are not supported", 
                header.ancount
            );
        }*/
        /*else if header.ancount == 0 && header.status.contains(StatusBits::RESP_NOERROR) == true
        {
            internal_error!(CDnsErrorType::DnsRespose, "answer count: 0 while response contains no error");
        }*/
        
        // reading request
        //let cur_offset = pkt.position();
    
        let request = 
            DnsRequestPayload
            {
                qname: pkt2name(&mut pkt)?,
                qtype: QType::u16_to_qtype(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?,
                qclass: QClass::u16_to_qclass(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?,
            };

        let mut an_list: Vec<DnsResponsePayload> = Vec::with_capacity(header.ancount as usize);
        let mut rr_list: Vec<DnsResponsePayload> = Vec::with_capacity(header.arcount as usize);
        let mut ns_list: Vec<DnsResponsePayload> = Vec::with_capacity(header.nscount as usize);

        for _ in 0..header.ancount
        {
            an_list.push(DnsResponsePayload::new(&mut pkt)?);
        }
        
        // reading authoritative section if any
        for _ in 0..header.nscount
        {
            ns_list.push(DnsResponsePayload::new(&mut pkt)?);
        }

        // reading additional if any
        for _ in 0..header.arcount
        {
            rr_list.push(DnsResponsePayload::new(&mut pkt)?);
        }

        return Ok(
            DnsRequestAnswer
            {
                header: header,
                request: request,
                response: an_list,
                additional: rr_list,
                authoratives: ns_list,
            }
        );
    }
}

impl DnsRequestHeader
{
    pub 
    fn to_bytes(&self) -> CDnsResult<Vec<u8>>
    {
        let pkt_size: usize = size_of::<DnsHeader>() + self.payload.qname.len() + 4; 

        let mut pkt = Cursor::new(vec![0_u8; pkt_size]);

        pkt.write_u16::<BigEndian>(self.header.id).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.status.bits()).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.qdcount).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.ancount).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.nscount).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.arcount).map_err(map_read_err)?;

        //for p in self.payload.iter()
        //{
            pkt.write(self.payload.qname.as_slice()).map_err(map_read_err)?;
            pkt.write_u16::<BigEndian>(self.payload.qtype.into()).map_err(map_read_err)?;
            pkt.write_u16::<BigEndian>(self.payload.qclass.into()).map_err(map_read_err)?;
        //}

        return Ok(pkt.into_inner());
    }
}


#[test]
fn test_pkt2name()
{
    use std::time::Instant;

    fn prepare(d: &[u8]) -> Cursor<&[u8]>
    {
       // let vd = d.to_vec();

        return Cursor::new(d);
    }

    let t1 = b"\x03\x64\x6e\x73\x06\x67\x6f\x6f\x67\x6c\x65\x00";

    let mut cur1 = prepare(t1);

    let now = Instant::now();
    let r1 = pkt2name(&mut cur1);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r1.is_ok(), true);
    assert_eq!(r1.as_ref().unwrap(), t1);

    let t2 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74\x00";

    let mut cur2 = prepare(t2);
    let now = Instant::now();
    let r2 = pkt2name(&mut cur2);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r2.is_ok(), true);
    assert_eq!(r2.as_ref().unwrap(), t2);

    let t3 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r3 = pkt2name(&mut prepare(t3));

    assert_eq!(r3.is_ok(), false);

    let t4 = 
    b"\x10\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r4 = pkt2name(&mut prepare(t4));

    assert_eq!(r4.is_ok(), false);
}

#[test]
fn test_qname2str()
{
    use std::time::Instant;

    fn prepare(d: &[u8]) -> Cursor<&[u8]>
    {
        //let vd = d.to_vec();

        return Cursor::new(d);
    }

    let t1 = b"\x03\x64\x6e\x73\x06\x67\x6f\x6f\x67\x6c\x65\x00";

    let mut cur1 = prepare(t1);
    let now = Instant::now();
    let r1 = name2str(&mut cur1, None);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r1.is_ok(), true);
    assert_eq!(r1.as_ref().unwrap(), "dns.google");

    let t2 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74\x00";

    let mut cur2 = prepare(t2);

    let now = Instant::now();
    let r2 = name2str(&mut cur2, None);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r2.is_ok(), true);
    assert_eq!(r2.as_ref().unwrap(), "mad07s09-in-x0e.1e100.net");

    let t3 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r3 = name2str(&mut prepare(t3), None);

    assert_eq!(r3.is_ok(), false);

    let t4 = 
    b"\x10\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r4 = name2str(&mut prepare(t4), None);

    assert_eq!(r4.is_ok(), false);
}

#[test]
fn test_response_parser()
{
    use std::time::Instant;

    let pkt = 
    b"\x15\xc8\x81\x80\x00\x01\x00\x01\x00\x00\x00\x02\x01\x38\x01\x38\
    \x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\
    \x61\x00\x00\x0c\x00\x01\xc0\x0c\x00\x0c\x00\x01\x00\x01\x35\xf0\
    \x00\x0c\x03\x64\x6e\x73\x06\x67\x6f\x6f\x67\x6c\x65\x00\xc0\x32\
    \x00\x01\x00\x01\x00\x00\x00\xe8\x00\x04\x08\x08\x08\x08\xc0\x32\
    \x00\x01\x00\x01\x00\x00\x00\xe8\x00\x04\x08\x08\x04\x04";

    let dummy = 
        DnsRequestHeader
        {
            header: DnsHeader {id: 0x15c8, ..Default::default()},
            payload: DnsRequestPayload
                {
                    qname: b"\x01\x38\x01\x38\x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\x61\x00".to_vec(),
                    qtype: QType::PTR,
                    qclass: QClass::IN, 
                }
        };

    let now = Instant::now();
    let ans = DnsRequestAnswer::try_from(pkt.as_slice());
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(ans.is_ok(), true, "{}", ans.err().unwrap());
    
    let ans = ans.unwrap();

    let verif = ans.verify(&dummy);
    assert_eq!(verif.is_ok(), true, "{}", verif.err().unwrap());

    assert_eq!(ans.header.id, 0x15c8);
    assert_eq!(ans.header.ancount, 1);
    assert_eq!(ans.header.arcount, 2);
    assert_eq!(ans.header.nscount, 0);
    assert_eq!(ans.header.qdcount, 1);

    assert_eq!(ans.response[0].rdata, DnsRdata::PTR{ fqdn: "dns.google".to_string() });
    assert_eq!(ans.response[0].name, "8.8.8.8.in-addr.arpa".to_string());
    assert_eq!(ans.response[0].dtype, QType::PTR);
    assert_eq!(ans.response[0].class, QClass::IN);
    assert_eq!(ans.response[0].ttl, 79344);
    assert_eq!(ans.response[0].rdlength, 12);

    assert_eq!(ans.additional[0].rdata, DnsRdata::A{ ip: "8.8.8.8".parse().unwrap() });
    assert_eq!(ans.additional[0].name, "dns.google".to_string());
    assert_eq!(ans.additional[0].ttl, 232);
    assert_eq!(ans.additional[0].dtype, QType::A);
    assert_eq!(ans.additional[0].class, QClass::IN);
    assert_eq!(ans.additional[1].rdata, DnsRdata::A{ ip: "8.8.4.4".parse().unwrap() });
    assert_eq!(ans.additional[1].name, "dns.google".to_string());
    assert_eq!(ans.additional[1].ttl, 232);
    assert_eq!(ans.additional[1].dtype, QType::A);
    assert_eq!(ans.additional[1].class, QClass::IN);
}

#[test]
fn test_response2()
{
    use std::time::Instant;

    let pkt = 
    b"\x4b\x38\x81\x80\x00\x01\x00\x05\x00\x00\x00\x00\x05\x67\x6d\x61\
    \x69\x6c\x03\x63\x6f\x6d\x00\x00\x0f\x00\x01\xc0\x0c\x00\x0f\x00\
    \x01\x00\x00\x0b\x55\x00\x1b\x00\x05\x0d\x67\x6d\x61\x69\x6c\x2d\
    \x73\x6d\x74\x70\x2d\x69\x6e\x01\x6c\x06\x67\x6f\x6f\x67\x6c\x65\
    \xc0\x12\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0b\x55\x00\x09\x00\x0a\
    \x04\x61\x6c\x74\x31\xc0\x29\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0b\
    \x55\x00\x09\x00\x1e\x04\x61\x6c\x74\x33\xc0\x29\xc0\x0c\x00\x0f\
    \x00\x01\x00\x00\x0b\x55\x00\x09\x00\x14\x04\x61\x6c\x74\x32\xc0\
    \x29\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0b\x55\x00\x09\x00\x28\x04\
    \x61\x6c\x74\x34\xc0\x29";

    let dummy = 
        DnsRequestHeader
        {
            header: DnsHeader {id: 0x4b38, ..Default::default()},
            payload: DnsRequestPayload
                {
                    qname: b"\x05\x67\x6d\x61\x69\x6c\x03\x63\x6f\x6d\x00".to_vec(),
                    qtype: QType::MX,
                    qclass: QClass::IN, 
                }
        };

    let now = Instant::now();
    let ans = DnsRequestAnswer::try_from(pkt.as_slice());
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(ans.is_ok(), true, "{}", ans.err().unwrap());
    let ans = ans.unwrap();

    let verif = ans.verify(&dummy);
    assert_eq!(verif.is_ok(), true, "{}", verif.err().unwrap());

    assert_eq!(ans.header.id, 0x4b38);
    assert_eq!(ans.header.ancount, 5);
    assert_eq!(ans.header.arcount, 0);
    assert_eq!(ans.header.nscount, 0);
    assert_eq!(ans.header.qdcount, 1);

    assert_eq!(ans.response[0].rdata, DnsRdata::MX{ preference: 5, exchange: "gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[0].name, "gmail.com".to_string());
    assert_eq!(ans.response[0].dtype, QType::MX);
    assert_eq!(ans.response[0].class, QClass::IN);
    assert_eq!(ans.response[0].ttl, 2901);
    assert_eq!(ans.response[0].rdlength, 27);

    assert_eq!(ans.response[1].rdata, DnsRdata::MX{ preference: 10, exchange: "alt1.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[1].name, "gmail.com".to_string());
    assert_eq!(ans.response[1].dtype, QType::MX);
    assert_eq!(ans.response[1].class, QClass::IN);
    assert_eq!(ans.response[1].ttl, 2901);
    assert_eq!(ans.response[1].rdlength, 9);

    assert_eq!(ans.response[2].rdata, DnsRdata::MX{ preference: 30, exchange: "alt3.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[2].name, "gmail.com".to_string());
    assert_eq!(ans.response[2].dtype, QType::MX);
    assert_eq!(ans.response[2].class, QClass::IN);
    assert_eq!(ans.response[2].ttl, 2901);
    assert_eq!(ans.response[2].rdlength, 9);

    assert_eq!(ans.response[3].rdata, DnsRdata::MX{ preference: 20, exchange: "alt2.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[3].name, "gmail.com".to_string());
    assert_eq!(ans.response[3].dtype, QType::MX);
    assert_eq!(ans.response[3].class, QClass::IN);
    assert_eq!(ans.response[3].ttl, 2901);
    assert_eq!(ans.response[3].rdlength, 9);

    assert_eq!(ans.response[4].rdata, DnsRdata::MX{ preference: 40, exchange: "alt4.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[4].name, "gmail.com".to_string());
    assert_eq!(ans.response[4].dtype, QType::MX);
    assert_eq!(ans.response[4].class, QClass::IN);
    assert_eq!(ans.response[4].ttl, 2901);
    assert_eq!(ans.response[4].rdlength, 9);
}


#[test]
fn test_response3()
{
    use std::time::Instant;

    let pkt = 
b"\xd0\x79\x81\x80\x00\x01\x00\x17\x00\x00\x00\x00\x06\x72\
\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00\x00\xff\x00\x01\xc0\x0c\x00\
\x06\x00\x01\x00\x00\x0e\x10\x00\x30\x04\x6e\x73\x32\x31\x07\x63\
\x6c\x6f\x75\x64\x6e\x73\x03\x6e\x65\x74\x00\x07\x73\x75\x70\x70\
\x6f\x72\x74\xc0\x2c\x78\x77\xe2\xf1\x00\x00\x1c\x20\x00\x00\x07\
\x08\x00\x12\x75\x00\x00\x00\x0e\x10\xc0\x0c\x00\x2e\x00\x01\x00\
\x00\x0e\x10\x00\x5d\x00\x06\x0d\x02\x00\x00\x0e\x10\x61\xf2\xaf\
\x39\x61\xcb\x22\x39\x28\xeb\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\
\x6b\x00\x2f\x79\xed\x73\xd0\x75\xc8\xa9\xa2\x2a\x08\x3c\x78\xb9\
\xfb\x43\xc5\x8e\x62\x4c\xbc\x36\xeb\x22\x96\x45\x59\x36\x1f\x69\
\x3a\x7a\x5e\x67\xd0\x54\x1e\xf0\xf2\x11\x1a\x72\x00\x56\x89\x26\
\x79\xf4\x06\x1a\x0c\x59\x41\x32\x60\x68\x75\x05\x3d\x90\xed\x1e\
\x0f\xfc\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x02\xc0\x27\
\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x08\x05\x70\x6e\x73\
\x32\x33\xc0\x2c\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x07\
\x04\x6e\x73\x32\x33\xc0\x2c\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\
\x10\x00\x08\x05\x70\x6e\x73\x32\x32\xc0\x2c\xc0\x0c\x00\x02\x00\
\x01\x00\x00\x0e\x10\x00\x08\x05\x70\x6e\x73\x32\x34\xc0\x2c\xc0\
\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x07\x04\x6e\x73\x32\x34\
\xc0\x2c\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x08\x05\x70\
\x6e\x73\x32\x31\xc0\x2c\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\
\x00\x07\x04\x6e\x73\x32\x32\xc0\x2c\xc0\x0c\x00\x2e\x00\x01\x00\
\x00\x0e\x10\x00\x5d\x00\x02\x0d\x02\x00\x00\x0e\x10\x61\xf2\xaf\
\x39\x61\xcb\x22\x39\x28\xeb\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\
\x6b\x00\xd1\xc5\x6c\xf1\xfb\xd0\x75\xf1\x38\x20\x28\x80\x4c\xe0\
\x59\xa5\xa8\xab\x84\x79\xd8\x37\x48\xa7\xa5\x3f\x08\x9b\x4c\xca\
\x40\x2b\xcb\x2c\xda\xcc\xc2\x18\xad\x07\x9e\xf8\x4e\x17\x8d\xb1\
\x2b\x2d\xa2\xa6\x17\xdb\x55\x30\xbc\xa2\xb9\xa0\x01\x71\x01\xe5\
\xdc\x4f\xc0\x0c\x00\x01\x00\x01\x00\x00\x0e\x10\x00\x04\xd9\x14\
\x70\xd0\xc0\x0c\x00\x2e\x00\x01\x00\x00\x0e\x10\x00\x5d\x00\x01\
\x0d\x02\x00\x00\x0e\x10\x61\xf2\xaf\x39\x61\xcb\x22\x39\x28\xeb\
\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00\xf8\x57\x68\xf0\xad\
\x9e\xfb\x3a\x0f\x66\xbd\xcc\x48\xe7\x29\x0a\xf4\xd8\xf6\xbe\xbc\
\x04\x76\x02\x27\x64\xf2\xc9\x42\x6d\x75\x54\x83\x0a\x11\xda\x0a\
\x02\x6b\x8c\xf1\x65\xc4\x21\x44\xea\x89\x09\x01\xc8\xa1\xe2\x11\
\x8f\xed\x67\x39\x69\x33\xdd\x97\x22\x1a\xd3\xc0\x0c\x00\x0f\x00\
\x01\x00\x00\x0e\x10\x00\x11\x00\x0a\x04\x6d\x61\x69\x6c\x04\x6e\
\x69\x78\x64\x03\x6f\x72\x67\x00\xc0\x0c\x00\x2e\x00\x01\x00\x00\
\x0e\x10\x00\x5d\x00\x0f\x0d\x02\x00\x00\x0e\x10\x61\xf2\xaf\x39\
\x61\xcb\x22\x39\x28\xeb\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\
\x00\xd4\xdd\x07\xd9\xb6\xb2\xba\x57\xa9\x1d\x3b\xaa\x6c\x55\xc4\
\x3d\x73\x79\xea\x74\xfe\xd7\x23\x0c\xb4\xab\x8f\x4b\x1f\xd9\x8a\
\xb2\x4a\x5c\xad\x3e\x8e\x4a\x85\xbb\xbd\x75\xf2\x47\x2c\xa8\x89\
\x21\x75\x89\xb1\x12\xc4\xd2\xf7\x40\x06\x52\x57\x83\x8a\xaa\x7b\
\x75\xc0\x0c\x00\x10\x00\x01\x00\x00\x0e\x10\x00\x34\x33\x76\x3d\
\x73\x70\x66\x31\x20\x2b\x6d\x78\x20\x2b\x61\x3a\x6d\x61\x69\x6c\
\x2e\x6e\x69\x78\x64\x2e\x6f\x72\x67\x20\x69\x70\x34\x3a\x32\x31\
\x37\x2e\x32\x30\x2e\x31\x31\x32\x2e\x32\x30\x38\x20\x2d\x61\x6c\
\x6c\xc0\x0c\x00\x2e\x00\x01\x00\x00\x0e\x10\x00\x5d\x00\x10\x0d\
\x02\x00\x00\x0e\x10\x61\xf2\xaf\x39\x61\xcb\x22\x39\x28\xeb\x06\
\x72\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00\xd1\x6c\xd1\xf4\x3b\xe0\
\x44\xba\xfe\xe9\xdb\x82\xbd\x89\x5f\xa1\x07\x72\xdd\x47\xad\x4e\
\x91\xd5\xc3\xfe\x3e\x39\x74\xdb\x50\x50\x19\x6c\x3f\x6c\xb7\xa8\
\x01\x03\x6a\xf5\xa7\xf3\x9b\xf7\x76\xd4\xff\xa3\xd5\x43\xfc\xec\
\xa9\x89\x24\xf8\xd2\xb6\x76\xd4\x20\xbc\xc0\x0c\x00\x30\x00\x01\
\x00\x00\x0e\x10\x00\x44\x01\x01\x03\x0d\xf3\x87\xe2\x7c\x2b\x82\
\x40\x72\x7c\xfd\xc9\x2b\xe8\x22\xd6\xa9\x40\xc0\xab\x03\x25\x7d\
\x92\xae\xf3\x17\x71\x82\x67\xc6\xcd\xb6\x4b\x11\x62\xc6\xfa\x06\
\xec\x4c\x9f\xd9\xe6\xaf\x5c\x3d\xe4\x32\xde\x11\x1b\x09\x13\xe3\
\xd0\xba\x66\xd1\xbc\x32\xdb\x13\xd7\x1d\xc0\x0c\x00\x30\x00\x01\
\x00\x00\x0e\x10\x00\x44\x01\x00\x03\x0d\xd4\x43\xde\x96\xe5\xea\
\x0a\xf9\x5d\x4f\x72\x88\x9c\xd9\x9c\xf7\xa6\x3f\x12\xd7\xf3\xea\
\x8a\x6b\x44\x4c\x79\x23\x81\x94\x43\xa3\xbd\x9e\xb8\xde\xfe\x8c\
\xe6\x21\xe3\x8a\x71\xba\x05\xd2\x0f\x98\x5b\xfc\x7e\x72\x8c\xe9\
\x9a\xc0\x49\x00\xca\xd5\x62\x93\x7f\x03\xc0\x0c\x00\x2e\x00\x01\
\x00\x00\x0e\x10\x00\x5d\x00\x30\x0d\x02\x00\x00\x0e\x10\x61\xf2\
\xaf\x39\x61\xcb\x22\x39\x28\xeb\x06\x72\x65\x6c\x6b\x6f\x6d\x02\
\x73\x6b\x00\xc0\x93\x23\x1d\xcb\x1e\x79\xfe\x7c\x40\x3e\xd4\x33\
\x5f\xed\x69\x8e\x7d\x75\xff\x73\x6b\x24\x71\x8f\x50\xf8\xe0\x49\
\xce\x5f\x62\x0c\x8c\xb3\x06\x8f\x26\xea\x20\xa0\xe3\x71\xe0\xa1\
\x8b\xe0\x4a\x2f\x1d\x4b\x79\x2c\x52\x6b\xa4\x43\xb5\x70\x27\x01\
\xb0\x63\x47\xc0\x0c\x00\x2e\x00\x01\x00\x00\x0e\x10\x00\x5d\x00\
\x30\x0d\x02\x00\x00\x0e\x10\x61\xf2\xaf\x39\x61\xcb\x22\x39\x97\
\x18\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00\x41\x87\x75\x1d\
\x30\x44\xd1\x94\x40\xd4\xe6\x40\x98\x62\x94\x53\xad\x53\xe2\xed\
\xc0\xc0\xb7\xa3\x20\x15\xae\x59\xbb\x97\x45\xfb\x0e\xbf\x70\xf3\
\xb1\x24\x79\xe8\x85\x6c\x2a\x66\x10\xb6\x75\x99\x7b\x77\x78\x65\
\xa6\x67\x8d\x59\xa6\x14\xf7\xe6\x77\xab\x53\x9c\xc0\x0c\x00\x33\
\x00\x01\x00\x00\x02\x58\x00\x0d\x01\x00\x00\x0a\x08\x90\xc7\xf1\
\x74\x0b\x0c\xfb\x34\xc0\x0c\x00\x2e\x00\x01\x00\x00\x02\x58\x00\
\x5d\x00\x33\x0d\x02\x00\x00\x00\x00\x61\xf2\xaf\x39\x61\xcb\x22\
\x39\x28\xeb\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00\xc4\x4d\
\x00\x48\x9c\x86\x49\xac\x8d\x03\x28\x23\xac\xec\xf5\x5b\xb6\xe5\
\x2f\xf6\xae\xaa\x01\x5a\x66\x52\xf7\x43\xc3\xb1\xe5\xef\xe5\xbf\
\x5f\x71\x5d\xa1\x57\x64\x66\x5e\xa1\x6f\x96\xa8\xcd\x48\x85\xe4\
\x20\xe2\xfb\xb0\xc1\x00\x47\x72\xc8\x72\x98\xc7\x41\xd9";
    
    let dummy = 
        DnsRequestHeader
        {
            header: DnsHeader {id: 0xd079, ..Default::default()},
            payload: DnsRequestPayload
                {
                    qname: b"\x06\x72\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00".to_vec(),
                    qtype: QType::ALL,
                    qclass: QClass::IN, 
                }
        };

    let now = Instant::now();
    let ans = DnsRequestAnswer::try_from(pkt.as_slice());
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(ans.is_ok(), true, "{}", ans.err().unwrap());
    let ans = ans.unwrap();

    let verif = ans.verify(&dummy);
    assert_eq!(verif.is_ok(), true, "{}", verif.err().unwrap());

    assert_eq!(ans.header.id, 0xd079);
    assert_eq!(ans.header.ancount, 23);
    assert_eq!(ans.header.arcount, 0);
    assert_eq!(ans.header.nscount, 0);
    assert_eq!(ans.header.qdcount, 1);

    let ansord = &ans.response[0];
    assert_eq!(
        ansord.rdata, 
        DnsRdata::SOA
        { 
            soa:
                DnsSoa
                {
                    pnm: "ns21.cloudns.net".to_string(), 
                    ram: "support.cloudns.net".to_string(), 
                    serial: 2021122801,  
                    interv_refr: 7200,
                    interv_retry: 1800,
                    expire_limit: 1209600,
                    min_ttl: 3600
                }
        }
    );
    assert_eq!(ansord.name, "relkom.sk".to_string());
    assert_eq!(ansord.dtype, QType::SOA);
    assert_eq!(ansord.class, QClass::IN);
    assert_eq!(ansord.ttl, 3600);
    assert_eq!(ansord.rdlength, 48);

    let a1 = 
b"\x00\x06\x0d\x02\
\x00\x00\x0e\x10\x61\xf2\xaf\x39\x61\xcb\x22\x39\x28\xeb\x06\x72\
\x65\x6c\x6b\x6f\x6d\x02\x73\x6b\x00\x2f\x79\xed\x73\xd0\x75\xc8\
\xa9\xa2\x2a\x08\x3c\x78\xb9\xfb\x43\xc5\x8e\x62\x4c\xbc\x36\xeb\
\x22\x96\x45\x59\x36\x1f\x69\x3a\x7a\x5e\x67\xd0\x54\x1e\xf0\xf2\
\x11\x1a\x72\x00\x56\x89\x26\x79\xf4\x06\x1a\x0c\x59\x41\x32\x60\
\x68\x75\x05\x3d\x90\xed\x1e\x0f\xfc";

    let ansord = &ans.response[1];
    assert_eq!(ansord.rdata, DnsRdata::UNKNOWN{ data: a1.to_vec() });
    assert_eq!(ansord.name, "relkom.sk".to_string());
    assert_eq!(ansord.dtype, QType::RRSIG);
    assert_eq!(ansord.class, QClass::IN);
    assert_eq!(ansord.ttl, 3600);
    assert_eq!(ansord.rdlength, 93);

    let ansord = &ans.response[2];
    assert_eq!(ansord.rdata, DnsRdata::NS{ fqdn: "ns21.cloudns.net".to_string() });
    assert_eq!(ansord.name, "relkom.sk".to_string());
    assert_eq!(ansord.dtype, QType::NS);
    assert_eq!(ansord.class, QClass::IN);
    assert_eq!(ansord.ttl, 3600);
    assert_eq!(ansord.rdlength, 2);
}


#[test]
fn test_request()
{
    use std::time::Instant;
    use std::net::IpAddr;

    let ipp = IpAddr::V4(Ipv4Addr::new(8, 8, 8, 8));
    let test = QDnsName::from(&ipp);


    let now = Instant::now();

    let req = DnsRequestHeader::construct_lookup(test, QType::PTR);

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(req.is_ok(), true, "{}", req.err().unwrap());

    let pkt = req.unwrap().to_bytes();
    assert_eq!(pkt.is_ok(), true);
    let pkt = pkt.unwrap();

    let ctrl = 
    b"\x15\xc8\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x01\x38\x01\x38\
    \x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\
    \x61\x00\x00\x0c\x00\x01";

    assert_eq!(&pkt[2..], &ctrl[2..]);
}

#[test]
fn test_request_1()
{
    use std::time::Instant;
    use std::net::IpAddr;

    let ipp = IpAddr::V4("100.150.111.80".parse().unwrap());
    let test = QDnsName::from(&ipp);

    let now = Instant::now();

    let req = DnsRequestHeader::construct_lookup(test, QType::PTR);

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(req.is_ok(), true, "{}", req.err().unwrap());

    let pkt = req.unwrap().to_bytes();
    assert_eq!(pkt.is_ok(), true);
    let pkt = pkt.unwrap();

    let ctrl = 
    b"\x74\xa1\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x02\x38\x30\x03\
    \x31\x31\x31\x03\x31\x35\x30\x03\x31\x30\x30\x07\x69\x6e\x2d\x61\
    \x64\x64\x72\x04\x61\x72\x70\x61\x00\x00\x0c\x00\x01";
    
    assert_eq!(&pkt[2..], &ctrl[2..]);
}

#[test]
fn test_request6()
{
    use std::time::Instant;
    use std::net::IpAddr;
    
    let ipp =  IpAddr::V6("2a00:1450:4003:802::200e".parse().unwrap());
    let test = QDnsName::from(&ipp);

    let now = Instant::now();

    let req = DnsRequestHeader::construct_lookup(test, QType::PTR);

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(req.is_ok(), true, "{}", req.err().unwrap());

    let pkt = req.unwrap().to_bytes();
    assert_eq!(pkt.is_ok(), true);
    let pkt = pkt.unwrap();

    let ctrl = 
    b"\xee\xec\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x01\x65\x01\x30\
    \x01\x30\x01\x32\x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\
    \x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\x01\x32\x01\x30\
    \x01\x38\x01\x30\x01\x33\x01\x30\x01\x30\x01\x34\x01\x30\x01\x35\
    \x01\x34\x01\x31\x01\x30\x01\x30\x01\x61\x01\x32\x03\x69\x70\x36\
    \x04\x61\x72\x70\x61\x00\x00\x0c\x00\x01";

    assert_eq!(&pkt[2..], &ctrl[2..]);
}

