/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/// This file contains the config file parsers.


use std::path::Path;

use crate::cfg_resolv_parser::{ResolveConfig, OptionFlags, ResolveConfEntry};
use crate::{internal_error_map, error::*};

use super::caches::CacheOperations;
use super::cfg_parsers::{ConfigParser, read_file, ROUND_ROBIN_CNT};
use super::common::{RESOLV_CFG_PATH, RESOLV_CFG_PATH_P};
use super::log::sync_log_writer;


pub struct RoundRobinIterator<'iterator>
{
    end_cycle: usize,
    current_index: usize,
    slice: &'iterator [ResolveConfEntry]
}

impl<'iterator> RoundRobinIterator<'iterator>
{
    pub 
    fn new(start_index: usize, slice: &'iterator [ResolveConfEntry]) -> RoundRobinIterator<'iterator>
    {
        if start_index == slice.len()
        {
            panic!("Assertion trap: start_index == slice.len i.e {} == {}", start_index, slice.len());
        }
        
        return 
            RoundRobinIterator
            {
                end_cycle: start_index + slice.len(),
                current_index: start_index,
                slice: slice
            };
    }
} 

impl<'iterator> Iterator for RoundRobinIterator<'iterator>
{
    type Item = &'iterator ResolveConfEntry;

    fn next(&mut self) -> Option<Self::Item> 
    {
        let mut curr = self.current_index;
        
        self.current_index += 1;

        if /*self.current_index*/curr == self.end_cycle
        {
            return None;
        }

        if curr >= self.slice.len()
        {
            curr -= self.slice.len();
        }

        return self.slice.get(curr);
    }
}


impl CacheOperations for ResolveConfig
{
    fn is_reload_allowed(&self) -> bool 
    {
        return !self.option_flags.contains(OptionFlags::OPT_NO_RELOAD);
    }
}


impl ConfigParser<ResolveConfig> for ResolveConfig
{
    fn parse_config() -> CDnsResult<Self>
    {
        let file_content = read_file(RESOLV_CFG_PATH)?;
        
        let mut writer = Writer::new();

        let ret = Self::parser_resolv_internal(file_content.as_str(), &mut writer);

        sync_log_writer(writer);

        return ret;
    }

    fn get_file_path() -> &'static Path
    {
        return &RESOLV_CFG_PATH_P;
    }

    fn is_default(&self) -> bool
    {
        return self.nameservers.is_empty();
    }
}

impl ResolveConfig
{
    /// This function should be used when the program which uses this library
    /// requires to override the systems /etc/resolv.conf
    pub 
    fn custom_config(resolv_cfg: &str) -> CDnsResult<Self>
    {
        let mut writer = Writer::new();

        let ret = Self::parser_resolv_internal(resolv_cfg, &mut writer);

        sync_log_writer(writer);

        return ret;
    }

    pub 
    fn get_resolvers_iter(&self) -> CDnsResult<RoundRobinIterator> 
    {
        if self.option_flags.is_rotate() == true
        {
            let mut l_cnt = 
                ROUND_ROBIN_CNT.lock().map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

            let start_index = *l_cnt;

            *l_cnt += 1;
            if *l_cnt == self.nameservers.len()
            {
                *l_cnt = 0;
            }

            drop(l_cnt);

            return Ok(RoundRobinIterator::new(start_index, self.nameservers.as_slice()));
        }
        else
        {
            return Ok(RoundRobinIterator::new(0, self.nameservers.as_slice()));
        };
    }
}

