/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021-2022  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::{sync::Arc};

use super::{network::{SocketTap, NetworkTapType}, common::DnsRequestHeader};


#[derive(Debug)]
pub(crate) struct Tap
{
    /// A socket which should be used to send requests
    tap: Box<NetworkTapType>,

    /// A requests binded to this specific tap
    qdnsr: Vec<DnsRequestHeader>
}

unsafe impl Send for Tap {}
unsafe impl Sync for Tap {}

impl Tap
{
    pub(crate) 
    fn new(tap: Box<NetworkTapType>, req: DnsRequestHeader) -> Self
    {
        return 
            Self
            {
                tap: tap,
                qdnsr: vec![req]
            };
    }

    pub(crate) 
    fn add(&mut self, req: DnsRequestHeader)
    {
        self.qdnsr.push(req);
    }

    pub(crate)
    fn into_inner(self) -> (Box<NetworkTapType>, Vec<DnsRequestHeader>)
    {
        return (self.tap, self.qdnsr);
    }

    /*pub(crate)
    fn inner_requests(self) -> Vec<DnsRequestHeader>
    {
        return self.qdnsr;
    }*/
}

pub(crate) struct AsyncTaps
{
    taps: Vec<Tap>,
}

impl AsyncTaps
{
    pub(crate) 
    fn new_with_capacity(tap_n: usize) -> Self
    {
        return Self{ taps: Vec::with_capacity(tap_n)/*, reqs: HashMap::with_capacity(req_n)*/ };
    }

    pub(crate) 
    fn push(&mut self, tap: Tap)
    {
        self.taps.push(tap);
    }

    pub(crate) 
    fn push_to_last(&mut self, req: DnsRequestHeader)
    {
        let t = self.taps.last_mut().unwrap();

        t.add(req);
    }

    pub(crate) 
    fn len(&self) -> usize
    {
        return self.taps.len();
    }

    pub(crate)
    fn into_iter(self) -> std::vec::IntoIter<Tap>
    {
        return self.taps.into_iter();
    }
}


