/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/*
                                1  1  1  1  1  1
  0  1  2  3  4  5  6  7  8  9  0  1  2  3  4  5
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | ID |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|QR|   Opcode  |AA|TC|RD|RA|   Z    |   RCODE   |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | QDCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | ANCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | NSCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
                    | ARCOUNT |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+

                                1  1  1  1  1  1
  0  1  2  3  4  5  6  7  8  9  0  1  2  3  4  5
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|                                               |
/                       QNAME                   /
/                                               /
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|                       QTYPE                   |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
|                       QCLASS                  |
+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+

*/

/// This file contains the common structures and constants of DNS protocol and 
/// other parts of the code. Also it contains a parser from/to network packet.

use std::{fmt, io::{Cursor, Read, Write}, mem::size_of, net::{IpAddr, Ipv4Addr, Ipv6Addr}};
use byteorder::{BigEndian, ReadBytesExt, WriteBytesExt};

use crate::error::*;
use crate::{internal_error, internal_error_map};

use super::cfg_parsers::HostnameEntry;

/// A location on the disk where resolv.conf is located
pub const RESOLV_CFG_PATH: &'static str = "/etc/resolv.conf";
/// A hosts file location on disk
pub const HOST_CFG_PATH: &'static str = "/etc/hosts";
/// A nsswitch location
pub const NSSWITCH_CFG_PATH: &'static str = "/etc/nsswitch.conf";

const IN_ADDR_ARPA: &[u8] = b"\x07in-addr\x04arpa\x00";
const IN_ADDR6_ARPA: &[u8] = b"\x03ip6\x04arpa\x00";

/// Converts a part of the octec of IP to printable hex i.e valid range is from
/// 0 up to 15 will be converted to 0x30..0x39 and 'a'..'f'.
/// 
/// # Arguments
/// 
/// * `b` - a part of the octec
/// 
/// # Returns
/// 
/// * u8 an ASCII representation
/// 
/// # Throws
/// 
/// Panic when input b >= 16
pub 
fn byte2hexchar(b: u8) -> u8
{
    match b
    {
        0..=9 => return '0' as u8 + b,
        10..=15 => return 'a' as u8 + (b - 10),
        _ => panic!("out of hex range!")
    }
}

/// Converts [IpAddr] to sequence of bytes coded specially for the payload
/// of package.
/// 
/// # Arguments
/// 
/// * `ip` - the argument with IP address
/// 
/// # Returns
/// 
/// * [CDnsResult] with vector of coded data
pub 
fn ip2pkt(ip: &IpAddr) -> CDnsResult<Vec<u8>>
{    
    match *ip
    {
        IpAddr::V4(ref ipv4) =>
            return ipv4_pkt(ipv4),
        IpAddr::V6(ref ipv6) =>
            return ipv6_pkt(ipv6)
    };
}

const MAX_NAME_LEN: usize = 63;

pub 
fn ipv4_pkt(ip: &Ipv4Addr) -> CDnsResult<Vec<u8>>
{    

    // pre allocate space
    let mut out: Vec<u8> = Vec::with_capacity(16 + IN_ADDR_ARPA.len());
    
    let mut octets = ip.octets();
    octets.reverse();

    for oct in octets
    {
        let str_oct = oct.to_string();

        if str_oct.len() > 3
        {
            internal_error!(
                CDnsErrorType::InternalError, 
                "domain component too long, len: '{}' oct: '{}'", 
                str_oct.len(), str_oct
            );
        }

        let ln: u8 = str_oct.len() as u8;

        out.push(ln);
        out.extend(str_oct.as_bytes());
    }

    out.extend(IN_ADDR_ARPA);

    return Ok(out);
}

pub 
fn ipv6_pkt(ip: &Ipv6Addr) -> CDnsResult<Vec<u8>>
{    
    // pre allocate space
    let mut out: Vec<u8> = Vec::with_capacity(32 + IN_ADDR6_ARPA.len());
    
    let mut octets = ip.octets();
    octets.reverse();

    for oct in octets
    {
        let h_oct = byte2hexchar((oct & 0xF0) >> 4);
        //format!("{:x}", (oct & 0xF0) >> 4);
        let l_oct = byte2hexchar(oct & 0x0F);
        //format!("{:x}", oct & 0x0F);

        //let ln: u8 = str_oct.len() as u8;

        out.push(1);
        out.push(l_oct);
        //out.extend(l_oct.as_bytes());
        out.push(1);
        out.push(h_oct);
    }

    out.extend(IN_ADDR6_ARPA);

    return Ok(out);
}

pub 
fn name2pkt(name: &str) -> CDnsResult<Vec<u8>>
{    
    // pre allocate space
    let mut out: Vec<u8> = Vec::with_capacity(name.len() + 2);
    
    for n in name.split(".")
    {
        if n.len() >= MAX_NAME_LEN
        {
            internal_error!(CDnsErrorType::InternalError, "name too long: '{}' in: '{}'", n.len(), name);
        }

        out.push((n.len() & 0xFF) as u8);
        out.extend(n.as_bytes());
    }

    out.push(0);

    return Ok(out);
}


/// Verifies the sequence of labels coding and copies data from pkt to
/// buffer.
/// 
/// # Arguments
/// 
/// * `pkt` - a reference to the [Cursor]
/// 
/// # Returns 
/// 
/// * [CDnsResult] with data copied to new [Vec<u8>]
pub 
fn pkt2name(pkt: &mut Cursor<&[u8]>) -> CDnsResult<Vec<u8>>
{
    let qname_s = pkt.position() as usize;

    loop
    {
        let cnt = pkt.read_u8().map_err(map_read_err)?;

        if cnt == 0
        {
            break;
        }
        else
        {
            for _ in 0..cnt
            {
                let c = pkt.read_u8().map_err(map_read_err)?;
                if c == 0
                {
                    internal_error!(
                        CDnsErrorType::DnsResponse, 
                        "incorrectly encoded QNAME in response, found '0' at offset: '{}'",
                        pkt.position()
                    );
                }
            }
        }
    }

    // temporary store the position of the cursor
    let cur_pos = pkt.position() as usize;

    if (cur_pos - qname_s) <= 1
    {
        internal_error!(
            CDnsErrorType::DnsResponse, 
            "read name is too short"
        );
    }

    /*pkt.set_position(qname_s);
    
    // creating buffer for QNAME
    let mut req: Vec<u8> = vec![0_u8; (cur_pos - qname_s) as usize];
    pkt.read_exact(&mut req).map_err(map_read_err)?;

    // restore position
    pkt.set_position(cur_pos);*/
    

    return Ok(pkt.get_ref()[qname_s..cur_pos].to_vec());
}


/// This function converts a QNAME (encoded domains) to the String FQDN.
pub 
fn name2str(pkt: &mut Cursor<&[u8]>, mut opt_rdlen: Option<u16>) -> CDnsResult<String>
{
    let cur_in_pos = pkt.position();
    //let mut p_comp: u8 = 0;
    let mut comp: u8;
    let mut output: Vec<String> = Vec::with_capacity(6);

    loop
    {
        if let Some(rdlen) = opt_rdlen
        {
            if pkt.position() - cur_in_pos >= rdlen as u64
            {
                return Ok(output.join("."));
            }
        }

        // reading word from pkt to detect the type of message
       // p_comp = comp;
        comp = pkt.read_u8().map_err(map_read_err)?;

        /*if p_comp > 0 && comp > 0
        {
            output.push('.');
        }*/

        let msb = comp & 0xC0;

        if msb == 0xC0
        {
            if opt_rdlen.is_none() == true
            {
                opt_rdlen = Some(2);
            }

            let offset1: u16 = ((comp & !0xC0) as u16) << 8;// | pkt.read_u8().map_err(map_read_err)? as u16;
            let offset2: u16 = pkt.read_u8().map_err(map_read_err)? as u16;

            let offset = offset1 | offset2;

            //println!("debug: 1: {} 2: {} offset: {}", offset1, offset2, offset);

            // C0 0...
            if offset as usize >= pkt.get_ref().len()
            {
                internal_error!(
                    CDnsErrorType::DnsResponse, 
                    "incoreclty formated packet: offset: '{}' > len: '{}'", 
                    offset, pkt.get_ref().len()
                );
            }

            // save current cursor position
            let cur_pos = pkt.position();
            pkt.set_position(offset as u64);

            // converting the name to FQDN
            output.push(name2str(pkt, None)?);
            //output.push_str(cache.get(&offset).ok_or_else(|| internal_error_map!(CDnsErrorType::InternalError, "cache mismatch"))?);

            // restoring cursor positon
            pkt.set_position(cur_pos);
        }
        else if msb == 0x00 
        {
            if comp == 0
            {
                if let Some(rdlen) = opt_rdlen
                {
                    let dif = pkt.position() - cur_in_pos;

                    if rdlen as u64 != dif
                    {
                        internal_error!(CDnsErrorType::DnsResponse, "incorrect rdlen: '{}', exp: '{}'", rdlen, dif);
                    }
                }

                return Ok(output.join("."));
            }
            else
            {
                let mut tmp: String = String::with_capacity(comp as usize);

                for _ in 0..comp
                {
                    let c = pkt.read_u8().map_err(map_read_err)?;
                    if c == 0
                    {
                        internal_error!(CDnsErrorType::DnsResponse, "protocol violation, incorrectly encoded QNAME in response");
                    }

                    tmp.push(c as char);
                }

                output.push(tmp);
            }
        }
        else
        {
            internal_error!(CDnsErrorType::DnsResponse, "incorrect compression: {:x}", msb);
        }

    } // loop
}

#[derive(Clone, Debug, Eq, PartialEq)]
pub enum DnsRdata
{
    None,
    A{ ip: Ipv4Addr },
    NS{ fqdn: String },
    MD{ data: Vec<u8> },
    MF{ data: Vec<u8> },
    CNAME{ fqdn: String },
    SOA{ data: Vec<u8> },
    MB{ data: Vec<u8> },
    MG{ data: Vec<u8> },
    MR{ data: Vec<u8> },
    NULL{ data: Vec<u8> },
    WKS{ data: Vec<u8> },
    PTR{ fqdn: String },
    HINFO{ data: Vec<u8> },
    MX{ preference: u16, exchange: String },
    TXT{ data: Vec<u8> },
    AFSDB{ data: Vec<u8> },
    KEY{ data: Vec<u8> },
    AAAA{ ip: Ipv6Addr },
    CERT{ data: Vec<u8> },
    DS{ data: Vec<u8> },
    RRSIG{ data: Vec<u8> },
    NSEC{ data: Vec<u8> },
    DNSKEY{ data: Vec<u8> },
    NSEC3{ data: Vec<u8> },
    NSEC3PARAM{ data: Vec<u8> },
    CDS{ data: Vec<u8> },
    CDNSKEY{ data: Vec<u8> },
    OPENPGPKEY{ data: Vec<u8> },    
    UNKNOWN{ data: Vec<u8> },
}

impl fmt::Display for DnsRdata
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::None => 
                write!(f, "No record"),
            Self::A{ ref ip } => 
                write!(f, "{}", ip),
            Self::NS{ ref fqdn } => 
                write!(f, "{}", fqdn),
            Self::AAAA{ ref ip} => 
                write!(f, "{}", ip),
            Self::MX{ preference, ref exchange } =>
                write!(f, "{} {}", preference, exchange),
            Self::CNAME{ ref fqdn } => 
                write!(f, "{}", fqdn),
            Self::PTR{ ref fqdn } => 
                write!(f, "{}", fqdn),
            Self::UNKNOWN{ .. } => 
                write!(f, "UNKNOWN"),
            _ => write!(f, "RAW DATA"),
        }
    }
}

impl Default for DnsRdata
{
    fn default() -> Self 
    {
        return Self::None;   
    }
}

impl DnsRdata
{
    /// Reads the RData from pkt and converts it into QType container
    /// 
    /// # Arguments
    /// 
    /// * `pkt` - a [Cursor] mutable reference to raw DNS reply data
    /// 
    /// * `rlen` - a length of RData section
    /// 
    /// * `qtype` - a [QType] reference to the response answer Qtype
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] with Self or error
    fn convert(pkt: &mut Cursor<&[u8]>, rlen: u16, qtype: &QType) -> CDnsResult<Self>
    {
        match &qtype
        {
            QType::A => 
            {
                // read 4 bytes (octets)
                if rlen != 4
                {
                    internal_error!(CDnsErrorType::DnsResponse, "ipv4 len expected: '4', got: '{}'", rlen);
                }

                let ip = pkt.read_u32::<BigEndian>().map_err(map_read_err)?;
                return Ok(Self::A{ ip: Ipv4Addr::from(ip) });
            },
            QType::AAAA => 
            {
                // read rlen bytes (octets)
                if rlen != 16
                {
                    internal_error!(CDnsErrorType::DnsResponse, "ipv6 len expected: '16', got: '{}'", rlen);
                }

                let ip = pkt.read_u128::<BigEndian>().map_err(map_read_err)?;
                return Ok(Self::AAAA{ ip: Ipv6Addr::from(ip) });
            },
            QType::MX =>
            {
                return Ok(
                    Self::MX
                    { 
                        preference: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                        exchange: name2str(pkt, Some(rlen - 2))?,
                    }
                );
            },
            QType::CNAME =>
            {
                return Ok(
                    Self::CNAME{ fqdn: name2str(pkt, Some(rlen))? }
                );
            },
            QType::PTR =>
            {
                return Ok(
                    Self::PTR{ fqdn: name2str(pkt, Some(rlen))? }
                );
            },
            QType::NS =>
            {
                return Ok(
                    Self::NS{ fqdn: name2str(pkt, Some(rlen))? }
                );
            },
            _ =>
            {
                let s = pkt.position() as usize;
                let data = pkt.get_ref()[s..s+rlen as usize].to_vec();

                /*QType::MD | QType::MF | QType::SOA | QType::MB | QType::MG | 
            QType::MR | QType::NULL | QType::WKS | QType::HINFO | QType::TXT |
            QType::AFSDB | QType::KEY | QType::CERT | QType::DS | QType::RRSIG |
            QType::NSEC | QType::DNSKEY | QType::NSEC3 | QType::NSEC3PARAM | 
            QType::CDS | QType::CDNSKEY | QType::OPENPGPKEY  */

                return Ok(Self::UNKNOWN{ data: data });
            }
        }
    }

    pub 
    fn is_some(&self) -> bool
    {
        return *self != Self::None;
    }
}

/// A request payload with (request) lines.
#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct DnsRequestPayload
{
    /// a domain name represented as a sequence of labels
    pub qname: Vec<u8>,

    /// a two octet code which specifies the type of the query.
    pub qtype: QType,
    
    /// a two octet code that specifies the class of the query.
    pub qclass: QClass,
}

impl DnsRequestPayload
{
    pub 
    fn new(qname: Vec<u8>, qtype: QType, qclass: QClass) -> Self
    {
        return DnsRequestPayload{ qname: qname, qtype: qtype.into(), qclass: qclass.into() };
    }
}

/// A response section
#[derive(Clone, Debug, Default, Eq, PartialEq)]
pub struct DnsResponsePayload
{
    // A domain name that was queried, in the same format as the QNAME in the question
    pub name: String,
    /// Two octets containing one of th type codes.
    pub dtype: QType,
    /// Two octets which specify the class of the data in the RDATA field
    pub class: QClass,
    /// specifies the time interval that the resource record may be cached before the source
    /// of the information should again be consulted
    pub ttl: i32,
    /// specifies the length in octets of the RDATA field
    pub rdlength: u16,
    /// a variable length string of octets that describes the resource
    pub rdata: DnsRdata,
}

impl DnsResponsePayload
{
    fn new(pkt: &mut Cursor<&[u8]> ) -> CDnsResult<Self>
    {
        let name = name2str(pkt, None)?;
        let qtype = QType::u16_to_type(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?;
        let qclass = QClass::u16_to_class(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?;
        let ttl = pkt.read_i32::<BigEndian>().map_err(map_read_err)?;
        let rdlength = pkt.read_u16::<BigEndian>().map_err(map_read_err)?;
        let rdata = DnsRdata::convert(pkt, rdlength, &qtype)?;

        return Ok(
            DnsResponsePayload
            {
                name: name,
                dtype: qtype,
                class: qclass,
                ttl: ttl,
                rdlength: rdlength,
                rdata: rdata,
            }
        );
    }

    /// Creates a record from [HostnameEntry], but only for types like
    /// A, AAAA, PTR
    pub(crate)
    fn new_local(dtype: QType, data: &HostnameEntry) -> CDnsResult<Vec<Self>>
    {
        match dtype
        {
            QType::A => 
            {
                let mut out: Vec<Self> = Vec::with_capacity(1);

                let ipv4: Ipv4Addr = 
                    if let IpAddr::V4(ipv4) = data.get_ip() 
                    {
                        ipv4.clone()
                    }
                    else
                    {
                        internal_error!(CDnsErrorType::InternalError, "wrong data type");
                    };

                out.push(
                    DnsResponsePayload
                    {
                        name: [data.get_hostnames()[0].as_str(), ".local"].concat(),
                        dtype: dtype,
                        class: QClass::IN,
                        ttl: i32::MAX,
                        rdlength: 0,
                        rdata: DnsRdata::A{ ip: ipv4 },
                    }
                );

                return Ok(out);
            },
            QType::AAAA => 
            {
                let mut out: Vec<Self> = Vec::with_capacity(1);

                let ipv6: Ipv6Addr = 
                    if let IpAddr::V6(ipv6) = data.get_ip() 
                    {
                        ipv6.clone()
                    }
                    else
                    {
                        internal_error!(CDnsErrorType::InternalError, "wrong data type");
                    };

                out.push(
                    DnsResponsePayload
                    {
                        name: [data.get_hostnames()[0].as_str(), ".local"].concat(),
                        dtype: dtype,
                        class: QClass::IN,
                        ttl: i32::MAX,
                        rdlength: 0,
                        rdata: DnsRdata::AAAA{ ip: ipv6 },
                    }
                );

                return Ok(out);
            },
            QType::PTR => 
            {
                let mut out: Vec<Self> = Vec::with_capacity(data.get_hostnames().len());

                for h in data.get_hostnames_iter()
                {
                    out.push(
                        DnsResponsePayload
                        {
                            name: [data.get_ip().to_string().as_str(), ".local"].concat(),
                            dtype: dtype,
                            class: QClass::IN,
                            ttl: i32::MAX,
                            rdlength: 0,
                            rdata: DnsRdata::PTR{ fqdn: h.clone() },
                        }
                    );
                }
            
                return Ok(out);
            },
            _ => 
            {
                internal_error!(CDnsErrorType::InternalError, "new_local can not be used for types except A, AAAA, PTR");
            }
        }
        
    }
}

/*impl Default for DnsResponsePayload
{
    fn default() -> Self 
    {
        return Self::Empty;    
    }
}*/

impl fmt::Display for DnsResponsePayload
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        write!(f, "{} {} {} {} {}", 
            self.name, self.dtype, self.class, self.ttl, self.rdata)  
    }
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct DnsRequestAnswer
{
    /// request header
    pub header: DnsHeader,
    /// Query content
    pub request: DnsRequestPayload,
    /// Resonse section
    pub response: Vec<DnsResponsePayload>,
    /// Additional RR
    pub additional: Vec<DnsResponsePayload>,
    /// Authortative nameservers
    pub authoratives: Vec<DnsResponsePayload>,
}

impl DnsRequestAnswer
{
    /// Parses the raw data from received packet. Also it validates that ID is
    /// and other fields are correct
    /// 
    /// # Arguments
    /// 
    /// * `ans` - the received data
    /// 
    /// * `req` - a request which was sent
    pub 
    fn parse(ans: &[u8], req: &DnsRequestHeader) -> CDnsResult<Self>
    {
        // creating cursor
        let mut pkt = Cursor::new(ans);

        // parsing header
        let id = pkt.read_u16::<BigEndian>().map_err(map_read_err)?;

        if id != req.header.id
        {
            internal_error!(
                CDnsErrorType::DnsResponse, 
                "request and response ID did not match: '{}' != '{}'", 
                req.header.id, id
            );
        }

        let status = pkt.read_u16::<BigEndian>().map_err(map_read_err)?;

        let header = 
            DnsHeader
            {
                id: id,
                status: StatusBits::from_bits(status).ok_or_else(|| 
                    internal_error_map!(CDnsErrorType::DnsResponse, "unknown status bits: '{}'", status)
                )?,
                qdcount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                ancount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                nscount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
                arcount: pkt.read_u16::<BigEndian>().map_err(map_read_err)?,
            };

        // todo!("what about truncated messages");

        // check if there is no more than 1 answer
        if header.status.contains(StatusBits::QR_RESP) == false
        {
            internal_error!(
                CDnsErrorType::DnsResponse, 
                "incorret QR flag in STATUS register of response, expected 1 got 0"
            );
        }
       /*else if header.ancount > 1
        {
            internal_error!(
                CDnsErrorType::DnsRespose, 
                "too many responses = '{}', multiple responses are not supported", 
                header.ancount
            );
        }*/
        /*else if header.ancount == 0 && header.status.contains(StatusBits::RESP_NOERROR) == true
        {
            internal_error!(CDnsErrorType::DnsRespose, "answer count: 0 while response contains no error");
        }*/
        else if header.status.contains(StatusBits::TRUN_CATION) == true
        {
            internal_error!(CDnsErrorType::MessageTruncated, "DNS response was truncated, aborting processing");
        }

        // reading request
        //let cur_offset = pkt.position();
    
        let request = 
            DnsRequestPayload
            {
                qname: pkt2name(&mut pkt)?,
                qtype: QType::u16_to_qtype(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?,
                qclass: QClass::u16_to_qclass(pkt.read_u16::<BigEndian>().map_err(map_read_err)?)?,
            };

        // check if request matches with what we sent previously
        if request != req.payload
        {
            internal_error!(CDnsErrorType::DnsResponse, "received request section is different from sent");
        }
        else if req.payload.qtype != QType::ALL
        {
            // check qtype matches requested. (is it correct?)
            if req.payload.qtype != request.qtype
            {
                internal_error!(
                    CDnsErrorType::DnsResponse, 
                    "requested QTYPE differ received TYPE: '{}' != '{}'", 
                    req.payload.qtype, 
                    request.qtype
                );
            }
        }

        let mut an_list: Vec<DnsResponsePayload> = Vec::with_capacity(header.ancount as usize);
        let mut rr_list: Vec<DnsResponsePayload> = Vec::with_capacity(header.arcount as usize);
        let mut ns_list: Vec<DnsResponsePayload> = Vec::with_capacity(header.nscount as usize);

        for _ in 0..header.ancount
        {
            an_list.push(DnsResponsePayload::new(&mut pkt)?);
        }
        
        // reading authoritative section if any
        for _ in 0..header.nscount
        {
            ns_list.push(DnsResponsePayload::new(&mut pkt)?);
        }

        // reading additional if any
        for _ in 0..header.arcount
        {
            rr_list.push(DnsResponsePayload::new(&mut pkt)?);
        }

        return Ok(
            DnsRequestAnswer
            {
                header: header,
                request: request,
                response: an_list,
                additional: rr_list,
                authoratives: ns_list,
            }
        );
    }
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct DnsHeader
{
    /// A 16 bit identifier assigned by the program that
    /// generates any kind of query.
    pub id: u16,

    /// Query control flags
    pub status: StatusBits,

    /// an unsigned 16 bit integer specifying the number of
    /// entries in the question section.
    pub qdcount: u16,

    /// an unsigned 16 bit integer specifying the number of
    /// resource records in the answer section.
    pub ancount: u16,

    /// an unsigned 16 bit integer specifying the number of name
    /// server resource records in the authority records
    /// section.
    pub nscount: u16,

    /// an unsigned 16 bit integer specifying the number of
    /// resource records in the additional records section.
    pub arcount: u16,
}


#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct DnsRequestHeader
{
    /// request header
    pub header: DnsHeader,
    /// Query content
    pub payload: DnsRequestPayload,
}

impl DnsRequestHeader
{
    pub 
    fn to_bytes(&self) -> CDnsResult<Vec<u8>>
    {
        let pkt_size: usize = size_of::<DnsHeader>() + self.payload.qname.len() + 4; 

        let mut pkt = Cursor::new(vec![0_u8; pkt_size]);

        pkt.write_u16::<BigEndian>(self.header.id).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.status.bits()).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.qdcount).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.ancount).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.nscount).map_err(map_read_err)?;
        pkt.write_u16::<BigEndian>(self.header.arcount).map_err(map_read_err)?;

        //for p in self.payload.iter()
        //{
            pkt.write(self.payload.qname.as_slice()).map_err(map_read_err)?;
            pkt.write_u16::<BigEndian>(self.payload.qtype.into()).map_err(map_read_err)?;
            pkt.write_u16::<BigEndian>(self.payload.qclass.into()).map_err(map_read_err)?;
        //}

        return Ok(pkt.into_inner());
    }
}

/// A two octet code which specifies the type of the query.
/// TYPE fields are used in resource records.  Note that these types are a
/// subset of QTYPEs.
/// QTYPE fields appear in the question part of a query.  QTYPES are a
/// superset of TYPEs, hence all TYPEs are valid QTYPEs.
#[repr(u16)]
#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum QType
{
    /// 1 a host address
    A = 1,
    /// 2 an authoritative name server
    NS = 2,
    /// 3 a mail destination (Obsolete - use MX)
    MD = 3,
    /// 4 a mail forwarder (Obsolete - use MX)
    MF = 4,
    /// 5 the canonical name for an alias
    CNAME = 5,
    /// 6 marks the start of a zone of authority
    SOA = 6,
    /// 7 a mailbox domain name (EXPERIMENTAL)
    MB = 7,
    /// 8 a mail group member (EXPERIMENTAL)
    MG = 8,
    /// 9 a mail rename domain name (EXPERIMENTAL)
    MR = 9,
    /// 10 a null RR (EXPERIMENTAL)
    NULL = 10,
    /// 11 a well known service description
    WKS = 11,
    /// 12 a domain name pointer
    PTR = 12,
    /// 13 host information
    HINFO = 13,
    /// 14 mailbox or mail list information
    MINFO = 14,
    /// 15 mail exchange
    MX = 15,
    /// 16 text strings
    TXT = 16,
    /// 18 AFS database record 
    AFSDB = 18,
    /// 25 Key record
    KEY = 25,
    /// 28 IPv6 address record
    AAAA = 28,
    /// 37 Certificate records
    CERT = 37,
    /// 43 Delegation signer
    DS = 43,
    /// 46 DNSSEC signature
    RRSIG = 46,
    /// 47 Next Secure record
    NSEC = 47,
    /// DNS Key record
    DNSKEY = 48,
    /// 50 Next Secure record version 3
    NSEC3 = 50,
    /// 51 NSEC3 parameters
    NSEC3PARAM = 51,
    /// 59 Child DS 
    CDS = 59,
    /// 60 Child copy of DNSKEY record, for transfer to parent
    CDNSKEY = 60,
    /// OpenPGP public key record 
    OPENPGPKEY = 61,
    // ---- QTYPE ----
    /// 252 A request for a transfer of an entire zone
    AXFR = 252,
    /// 253 A request for mailbox-related records (MB, MG or MR)
    MAILB = 253,
    /// 254 A request for mail agent RRs (Obsolete - see MX)
    MAILA = 254,
    /// 257 Certification Authority Authorization 
    CAA = 257,
    /// *
    ALL = 255,
    /// 32769 DNSSEC Lookaside Validation record
    DLV = 32769,
}

impl Default for QType
{
    fn default() -> Self 
    {
        return Self::A;    
    }
}

impl Into<u16> for QType
{
    fn into(self) -> u16 
    {
        return self as u16;
    }
}

impl fmt::Display for QType
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result
    {
        match *self
        {
            Self::A          => write!(f, "A"),
            Self::NS         => write!(f, "NS"),
            Self::MD         => write!(f, "MD"),
            Self::MF         => write!(f, "MF"),
            Self::CNAME      => write!(f, "CNAME"),
            Self::SOA        => write!(f, "SOA"),
            Self::MB         => write!(f, "MB"),
            Self::MG         => write!(f, "MG"),
            Self::MR         => write!(f, "MR"),
            Self::NULL       => write!(f, "NULL"),
            Self::WKS        => write!(f, "WKS"),
            Self::PTR        => write!(f, "PTR"),
            Self::HINFO      => write!(f, "HINFO"),
            Self::MINFO      => write!(f, "MINFO"),
            Self::MX         => write!(f, "MX"),
            Self::TXT        => write!(f, "TXT"),
            Self::AFSDB      => write!(f, "AFSDB"),
            Self::KEY        => write!(f, "KEY"),
            Self::AAAA       => write!(f, "AAAA"),
            Self::CERT       => write!(f, "CERT"),
            Self::DS         => write!(f, "DS"),
            Self::RRSIG      => write!(f, "RRSIG"),
            Self::NSEC       => write!(f, "NSEC"),
            Self::DNSKEY     => write!(f, "DNSKEY"),
            Self::NSEC3      => write!(f, "NSEC"),
            Self::NSEC3PARAM => write!(f, "NSEC3PARAM"),
            Self::CDS        => write!(f, "CDS"),
            Self::CDNSKEY    => write!(f, "CDNSKEY"),
            Self::OPENPGPKEY => write!(f, "OPENPGPKEY"),
            Self::AXFR       => write!(f, "AXFR"),
            Self::MAILB      => write!(f, "MAILB"),
            Self::MAILA      => write!(f, "MAILA"),
            Self::CAA        => write!(f, "CAA"),
            Self::ALL        => write!(f, "ALL"),
            Self::DLV        => write!(f, "DLV"),
        }
    }
}


impl QType
{
    pub 
    fn u16_to_qtype(value: u16) -> CDnsResult<QType>
    {
        match value
        {
            x if x == Self::AXFR as u16    => return Ok(Self::AXFR),
            x if x == Self::MAILB as u16   => return Ok(Self::MAILB),
            x if x == Self::MAILA as u16   => return Ok(Self::MAILA),
            x if x == Self::ALL as u16     => return Ok(Self::ALL),
            x if x == Self::DLV as u16     => return Ok(Self::DLV),
            _ => return Self::u16_to_type(value),
        }
    }

    pub 
    fn u16_to_type(value: u16) -> CDnsResult<QType>
    {

        match value
        {
            x if x == Self::A as u16            => return Ok(Self::A),
            x if x == Self::NS as u16           => return Ok(Self::NS),
            x if x == Self::MD as u16           => return Ok(Self::MD),
            x if x == Self::MF as u16           => return Ok(Self::MF),
            x if x == Self::CNAME as u16        => return Ok(Self::CNAME),
            x if x == Self::SOA as u16          => return Ok(Self::SOA),
            x if x == Self::MB as u16           => return Ok(Self::MB),
            x if x == Self::MG as u16           => return Ok(Self::MG),
            x if x == Self::MR as u16           => return Ok(Self::MR),
            x if x == Self::NULL as u16         => return Ok(Self::NULL),
            x if x == Self::WKS as u16          => return Ok(Self::WKS),
            x if x == Self::PTR as u16          => return Ok(Self::PTR),
            x if x == Self::HINFO as u16        => return Ok(Self::HINFO),
            x if x == Self::MINFO as u16        => return Ok(Self::MINFO),
            x if x == Self::MX as u16           => return Ok(Self::MX),
            x if x == Self::TXT as u16          => return Ok(Self::TXT),
            x if x == Self::AFSDB as u16        => return Ok(Self::AFSDB),
            x if x == Self::KEY as u16          => return Ok(Self::KEY),
            x if x == Self::AAAA as u16         => return Ok(Self::AAAA),
            x if x == Self::CERT as u16         => return Ok(Self::CERT),
            x if x == Self::DS as u16           => return Ok(Self::DS),
            x if x == Self::RRSIG as u16        => return Ok(Self::RRSIG),
            x if x == Self::NSEC as u16         => return Ok(Self::NSEC),
            x if x == Self::DNSKEY as u16       => return Ok(Self::DNSKEY),
            x if x == Self::NSEC3 as u16        => return Ok(Self::NSEC3),
            x if x == Self::NSEC3PARAM as u16   => return Ok(Self::NSEC3PARAM),
            x if x == Self::CDS as u16          => return Ok(Self::CDS),
            x if x == Self::CDNSKEY as u16      => return Ok(Self::CDNSKEY),
            x if x == Self::OPENPGPKEY as u16   => return Ok(Self::OPENPGPKEY),
            _ => internal_error!(CDnsErrorType::DnsResponse, "unknown request record type: '{}'", value),
        }
    }
}

/// A two octet code that specifies the class of the query.
/// 
/// 
#[repr(u16)]
#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum QClass
{
    /// the Internet 1
    IN = 1,
    /// the CSNET class (Obsolete) 2
    CS = 2,
    /// the CHAOS class 3
    CH = 3,
    /// Hesiod [Dyer 87] 4
    HS = 4,
    
    //  --- QClass
    /// any class 255
    ALL = 255,
}

impl fmt::Display for QClass
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::IN  => write!(f, "IN"),
            Self::CS  => write!(f, "CS"),
            Self::CH  => write!(f, "CH"),
            Self::HS  => write!(f, "HS"),
            Self::ALL => write!(f, "ALL"),
        }
    }
}

impl Default for QClass
{
    fn default() -> Self 
    {
        return Self::IN;    
    }
}

impl Into<u16> for QClass
{
    fn into(self) -> u16 
    {
        return self as u16;
    }
}

impl QClass
{
    /// Converts u16 to enum [QClass].
    /// 
    /// QCLASS fields appear in the question part of a query.  QCLASS are a
    /// superset of CLASSes, hence all CLASSes are valid QCLASSes.
    pub 
    fn u16_to_qclass(value: u16) -> CDnsResult<QClass>
    {
        match value
        {
            x if x == QClass::ALL as u16 => return Ok(QClass::ALL),
            _ => Self::u16_to_class(value),
        }
    }

    /// Converts u16 to enum [QClass], but for the response part of a query.
    pub 
    fn u16_to_class(value: u16) -> CDnsResult<QClass>
    {
        match value
        {
            x if x == QClass::IN as u16 => return Ok(QClass::IN),
            x if x == QClass::CS as u16 => return Ok(QClass::CS),
            x if x == QClass::CH as u16 => return Ok(QClass::CH),
            x if x == QClass::HS as u16 => return Ok(QClass::HS),
            _ => internal_error!(CDnsErrorType::DnsResponse, "unknown QCLASS type: '{}'", value),
        }
    }
}

// (req.status & !StatusBits::OPCODE_STANDARD) | ...
// () 

bitflags! {     
    /// Flags  which control the status of th DNS query    
    #[derive(Default)] 
    pub struct StatusBits: u16  
    {     
        // Pkt type Request/Result R/W    
        /// A request/response bit, when 1 - response
        const QR_RESP           = 0x8000;

        // Request Type (Write only)
        /// a standard query (QUERY) USE inverse to reset
        const OPCODE_STANDARD   = 0x87FF;
        /// an inverse query (IQUERY)
        const OPCODE_IQUERY     = 0x0040;
        /// a server status request (STATUS)
        const OPCODE_STATUS     = 0x0020;
        // reserved for future use 3-15
        // 0x4000 .. 0x0800

        /// AA  Authoritative Answer (Read only)
        /// Authoritative Answer - this bit is valid in responses,
        /// and specifies that the responding name server is an
        /// authority for the domain name in question section.
        const AUTH_ANSWER       = 0x0400;

        /// TrunCation (Read only)
        /// TrunCation - specifies that this message was truncated
        /// due to length greater than that permitted on the transmission channel.
        const TRUN_CATION       = 0x0200;

        /// RD Recursion desired (write only)
        /// Recursion Desired - this bit may be set in a query and
        /// is copied into the response.  If RD is set, it directs
        /// the name server to pursue the query recursively.
        /// Recursive query support is optional.
        const RECURSION_DESIRED = 0x0100;

        /// RA Recursion available (read only)
        /// Recursion Available - this be is set or cleared in a
        /// response, and denotes whether recursive query support is
        /// available in the name server.
        const RECURSION_AVAIL   = 0x0080;

        /// Z Zero reserver
        /// Reserved for future use.  Must be zero in all queries
        /// and responses.
        const RSERVER0          = 0x0040; 

        const ANSWER_AUTHN      = 0x0020; 
        const NON_AUTH_DATA     = 0x0010; 

        // RCODE response code 

        /// No error condition
        const RESP_NOERROR      = 0x0000; 
        /// Format error - The name server was unable to interpret the query.
        const RESP_FORMERR      = 0x0001; 
        /// Server failure - The name server was unable to process this query due to a
        /// problem with the name server.
        const RESP_SERVFAIL     = 0x0002; 
        /// Name Error - Meaningful only for responses from an authoritative name
        /// server, this code signifies that the domain name referenced in the query does
        /// not exist.
        const RESP_NXDOMAIN     = 0x0003;
        /// Not Implemented - The name server does not support the requested kind of query.
        const RESP_NOT_IMPL     = 0x0004; 
        /// Refused - The name server refuses to perform the specified operation for
        /// policy reasons.
        const RESP_REFUSED      = 0x0005; 
        // ..= 0x000F
    }
}


#[test]
fn test_pkt2name()
{
    use std::time::Instant;

    fn prepare(d: &[u8]) -> Cursor<&[u8]>
    {
       // let vd = d.to_vec();

        return Cursor::new(d);
    }

    let t1 = b"\x03\x64\x6e\x73\x06\x67\x6f\x6f\x67\x6c\x65\x00";

    let mut cur1 = prepare(t1);

    let now = Instant::now();
    let r1 = pkt2name(&mut cur1);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r1.is_ok(), true);
    assert_eq!(r1.as_ref().unwrap(), t1);

    let t2 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74\x00";

    let mut cur2 = prepare(t2);
    let now = Instant::now();
    let r2 = pkt2name(&mut cur2);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r2.is_ok(), true);
    assert_eq!(r2.as_ref().unwrap(), t2);

    let t3 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r3 = pkt2name(&mut prepare(t3));

    assert_eq!(r3.is_ok(), false);

    let t4 = 
    b"\x10\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r4 = pkt2name(&mut prepare(t4));

    assert_eq!(r4.is_ok(), false);
}

#[test]
fn test_qname2str()
{
    use std::time::Instant;

    fn prepare(d: &[u8]) -> Cursor<&[u8]>
    {
        //let vd = d.to_vec();

        return Cursor::new(d);
    }

    let t1 = b"\x03\x64\x6e\x73\x06\x67\x6f\x6f\x67\x6c\x65\x00";

    let mut cur1 = prepare(t1);
    let now = Instant::now();
    let r1 = name2str(&mut cur1, None);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r1.is_ok(), true);
    assert_eq!(r1.as_ref().unwrap(), "dns.google");

    let t2 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74\x00";

    let mut cur2 = prepare(t2);

    let now = Instant::now();
    let r2 = name2str(&mut cur2, None);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(r2.is_ok(), true);
    assert_eq!(r2.as_ref().unwrap(), "mad07s09-in-x0e.1e100.net");

    let t3 = 
    b"\x0f\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r3 = name2str(&mut prepare(t3), None);

    assert_eq!(r3.is_ok(), false);

    let t4 = 
    b"\x10\x6d\x61\x64\x30\x37\x73\x30\x39\x2d\x69\x6e\x2d\x78\x30\x65\x05\x31\x65\x31\
    \x30\x30\x03\x6e\x65\x74";

    let r4 = name2str(&mut prepare(t4), None);

    assert_eq!(r4.is_ok(), false);
}

#[test]
fn test_response_parser()
{
    use std::time::Instant;

    let pkt = 
    b"\x15\xc8\x81\x80\x00\x01\x00\x01\x00\x00\x00\x02\x01\x38\x01\x38\
    \x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\
    \x61\x00\x00\x0c\x00\x01\xc0\x0c\x00\x0c\x00\x01\x00\x01\x35\xf0\
    \x00\x0c\x03\x64\x6e\x73\x06\x67\x6f\x6f\x67\x6c\x65\x00\xc0\x32\
    \x00\x01\x00\x01\x00\x00\x00\xe8\x00\x04\x08\x08\x08\x08\xc0\x32\
    \x00\x01\x00\x01\x00\x00\x00\xe8\x00\x04\x08\x08\x04\x04";

    let dummy = 
        DnsRequestHeader
        {
            header: DnsHeader {id: 0x15c8, ..Default::default()},
            payload: DnsRequestPayload
                {
                    qname: b"\x01\x38\x01\x38\x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\x61\x00".to_vec(),
                    qtype: QType::PTR,
                    qclass: QClass::IN, 
                }
        };

    let now = Instant::now();
    let ans = DnsRequestAnswer::parse(pkt, &dummy);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(ans.is_ok(), true, "{}", ans.err().unwrap());
    
    let ans = ans.unwrap();

    assert_eq!(ans.header.id, 0x15c8);
    assert_eq!(ans.header.ancount, 1);
    assert_eq!(ans.header.arcount, 2);
    assert_eq!(ans.header.nscount, 0);
    assert_eq!(ans.header.qdcount, 1);

    assert_eq!(ans.response[0].rdata, DnsRdata::PTR{ fqdn: "dns.google".to_string() });
    assert_eq!(ans.response[0].name, "8.8.8.8.in-addr.arpa".to_string());
    assert_eq!(ans.response[0].dtype, QType::PTR);
    assert_eq!(ans.response[0].class, QClass::IN);
    assert_eq!(ans.response[0].ttl, 79344);
    assert_eq!(ans.response[0].rdlength, 12);

    assert_eq!(ans.additional[0].rdata, DnsRdata::A{ ip: "8.8.8.8".parse().unwrap() });
    assert_eq!(ans.additional[0].name, "dns.google".to_string());
    assert_eq!(ans.additional[0].ttl, 232);
    assert_eq!(ans.additional[0].dtype, QType::A);
    assert_eq!(ans.additional[0].class, QClass::IN);
    assert_eq!(ans.additional[1].rdata, DnsRdata::A{ ip: "8.8.4.4".parse().unwrap() });
    assert_eq!(ans.additional[1].name, "dns.google".to_string());
    assert_eq!(ans.additional[1].ttl, 232);
    assert_eq!(ans.additional[1].dtype, QType::A);
    assert_eq!(ans.additional[1].class, QClass::IN);
}

#[test]
fn test_response2()
{
    use std::time::Instant;

    let pkt = 
    b"\x4b\x38\x81\x80\x00\x01\x00\x05\x00\x00\x00\x00\x05\x67\x6d\x61\
    \x69\x6c\x03\x63\x6f\x6d\x00\x00\x0f\x00\x01\xc0\x0c\x00\x0f\x00\
    \x01\x00\x00\x0b\x55\x00\x1b\x00\x05\x0d\x67\x6d\x61\x69\x6c\x2d\
    \x73\x6d\x74\x70\x2d\x69\x6e\x01\x6c\x06\x67\x6f\x6f\x67\x6c\x65\
    \xc0\x12\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0b\x55\x00\x09\x00\x0a\
    \x04\x61\x6c\x74\x31\xc0\x29\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0b\
    \x55\x00\x09\x00\x1e\x04\x61\x6c\x74\x33\xc0\x29\xc0\x0c\x00\x0f\
    \x00\x01\x00\x00\x0b\x55\x00\x09\x00\x14\x04\x61\x6c\x74\x32\xc0\
    \x29\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0b\x55\x00\x09\x00\x28\x04\
    \x61\x6c\x74\x34\xc0\x29";

    let dummy = 
        DnsRequestHeader
        {
            header: DnsHeader {id: 0x4b38, ..Default::default()},
            payload: DnsRequestPayload
                {
                    qname: b"\x05\x67\x6d\x61\x69\x6c\x03\x63\x6f\x6d\x00".to_vec(),
                    qtype: QType::MX,
                    qclass: QClass::IN, 
                }
        };

    let now = Instant::now();
    let ans = DnsRequestAnswer::parse(pkt, &dummy);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(ans.is_ok(), true, "{}", ans.err().unwrap());
    let ans = ans.unwrap();

    assert_eq!(ans.header.id, 0x4b38);
    assert_eq!(ans.header.ancount, 5);
    assert_eq!(ans.header.arcount, 0);
    assert_eq!(ans.header.nscount, 0);
    assert_eq!(ans.header.qdcount, 1);

    assert_eq!(ans.response[0].rdata, DnsRdata::MX{ preference: 5, exchange: "gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[0].name, "gmail.com".to_string());
    assert_eq!(ans.response[0].dtype, QType::MX);
    assert_eq!(ans.response[0].class, QClass::IN);
    assert_eq!(ans.response[0].ttl, 2901);
    assert_eq!(ans.response[0].rdlength, 27);

    assert_eq!(ans.response[1].rdata, DnsRdata::MX{ preference: 10, exchange: "alt1.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[1].name, "gmail.com".to_string());
    assert_eq!(ans.response[1].dtype, QType::MX);
    assert_eq!(ans.response[1].class, QClass::IN);
    assert_eq!(ans.response[1].ttl, 2901);
    assert_eq!(ans.response[1].rdlength, 9);

    assert_eq!(ans.response[2].rdata, DnsRdata::MX{ preference: 30, exchange: "alt3.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[2].name, "gmail.com".to_string());
    assert_eq!(ans.response[2].dtype, QType::MX);
    assert_eq!(ans.response[2].class, QClass::IN);
    assert_eq!(ans.response[2].ttl, 2901);
    assert_eq!(ans.response[2].rdlength, 9);

    assert_eq!(ans.response[3].rdata, DnsRdata::MX{ preference: 20, exchange: "alt2.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[3].name, "gmail.com".to_string());
    assert_eq!(ans.response[3].dtype, QType::MX);
    assert_eq!(ans.response[3].class, QClass::IN);
    assert_eq!(ans.response[3].ttl, 2901);
    assert_eq!(ans.response[3].rdlength, 9);

    assert_eq!(ans.response[4].rdata, DnsRdata::MX{ preference: 40, exchange: "alt4.gmail-smtp-in.l.google.com".to_string() });
    assert_eq!(ans.response[4].name, "gmail.com".to_string());
    assert_eq!(ans.response[4].dtype, QType::MX);
    assert_eq!(ans.response[4].class, QClass::IN);
    assert_eq!(ans.response[4].ttl, 2901);
    assert_eq!(ans.response[4].rdlength, 9);
}


#[test]
fn test_response3()
{
    use std::time::Instant;

    let pkt = 
    b"\xb5\x9a\x81\x80\x00\x01\
    \x00\x11\x00\x04\x00\x00\x04\x6e\x69\x78\x64\x03\x6f\x72\x67\x00\
    \x00\xff\x00\x01\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x12\
    \x04\x6e\x73\x32\x34\x07\x63\x6c\x6f\x75\x64\x6e\x73\x03\x6e\x65\
    \x74\x00\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\x00\x07\x04\x6e\
    \x73\x32\x33\xc0\x2b\xc0\x0c\x00\x33\x00\x01\x00\x00\x00\x01\x00\
    \x0d\x01\x00\x00\x0a\x08\xcb\x22\x97\x04\x93\xda\x7f\x9c\xc0\x0c\
    \x00\x02\x00\x01\x00\x00\x0e\x10\x00\x07\x04\x6e\x73\x32\x32\xc0\
    \x2b\xc0\x0c\x00\x30\x00\x01\x00\x00\x0e\x10\x00\x44\x01\x01\x03\
    \x0d\xd6\x6f\x8a\xd1\x2b\x02\x12\x3f\x3e\x60\xc3\xc0\x76\xb8\xc8\
    \x38\x5e\x35\xf5\x1b\x9a\x38\xf1\x8e\xdd\x38\x2c\x8b\x3b\x27\x11\
    \x90\x1a\x39\xbc\xf0\x40\x6f\x25\xd4\xf2\x37\x3d\x3f\x65\x7b\x82\
    \x5d\x61\xb6\x96\x21\x8e\x57\xb0\x27\x2a\x81\xa9\xd7\x4b\xc6\x2f\
    \x5f\xc0\x0c\x00\x30\x00\x01\x00\x00\x0e\x10\x00\x44\x01\x00\x03\
    \x0d\xde\xe7\x1f\x6d\xda\x40\x7f\x92\x3a\xa6\xe6\x4a\x00\xdc\x46\
    \xce\x09\x5f\x21\x74\x22\x7f\x27\x52\x9e\xa3\xd6\xd9\xf8\xe9\x88\
    \x9b\xde\x92\x2a\x2a\x45\xa9\xbe\x1d\x2f\x0d\x9d\xca\x22\x44\x92\
    \x86\xe8\xc8\xe1\x7c\x4f\x7a\x56\x1c\x6b\x0c\x72\x29\x59\x54\x03\
    \x42\xc0\x0c\x00\x10\x00\x01\x00\x00\x0e\x10\x00\x34\x33\x76\x3d\
    \x73\x70\x66\x31\x20\x2b\x6d\x78\x20\x2b\x61\x3a\x6d\x61\x69\x6c\
    \x2e\x6e\x69\x78\x64\x2e\x6f\x72\x67\x20\x69\x70\x34\x3a\x32\x31\
    \x37\x2e\x32\x30\x2e\x31\x31\x32\x2e\x32\x30\x38\x20\x2d\x61\x6c\
    \x6c\xc0\x0c\x00\x0f\x00\x01\x00\x00\x0e\x10\x00\x09\x00\x0a\x04\
    \x6d\x61\x69\x6c\xc0\x0c\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\x10\
    \x00\x07\x04\x6e\x73\x32\x31\xc0\x2b\xc0\x0c\x00\x06\x00\x01\x00\
    \x00\x0e\x10\x00\x20\xc1\x78\x07\x73\x75\x70\x70\x6f\x72\x74\xc0\
    \x2b\x78\x77\x95\x35\x00\x00\x1c\x20\x00\x00\x07\x08\x00\x12\x75\
    \x00\x00\x00\x0e\x10\xc0\x0c\x00\x2e\x00\x01\x00\x00\x0e\x10\x00\
    \x5c\x00\x06\x0d\x02\x00\x00\x0e\x10\x61\xa3\x9a\x00\x61\x7c\x0d\
    \x00\x68\x47\x04\x6e\x69\x78\x64\x03\x6f\x72\x67\x00\x8f\xc7\xa6\
    \x30\x7b\xed\xe5\xc5\xd6\x5d\x63\x24\x0f\x8e\x41\x04\xda\xed\x45\
    \x43\x6d\xc6\x71\xbb\x29\x0e\x20\x40\x29\x04\x16\x29\xe1\x03\xc8\
    \x76\x87\x06\x8c\x60\x1b\xe4\x3f\xbc\xca\x04\xf1\x8e\x33\x06\x0e\
    \x50\xfa\xea\x92\x45\x3f\x06\xd5\x6c\x8f\x26\x11\x9f\xc0\x0c\x00\
    \x2e\x00\x01\x00\x00\x0e\x10\x00\x5c\x00\x02\x0d\x02\x00\x00\x0e\
    \x10\x61\xa3\x9a\x00\x61\x7c\x0d\x00\x68\x47\x04\x6e\x69\x78\x64\
    \x03\x6f\x72\x67\x00\xfc\x4e\xfc\xfe\xef\x69\x44\xac\xeb\xb4\x50\
    \x51\x1e\x4a\xa4\x29\x93\xd2\x6c\x79\x1a\x61\xe6\xde\x4f\xc2\x4f\
    \xda\x94\x3d\x84\x22\x86\xfe\x23\x8b\x7e\x84\xf0\xb0\x2d\x6b\xfa\
    \xea\xee\x3a\xf5\x46\x75\x24\x35\x92\xd0\xea\xab\x3c\x60\xd9\xbc\
    \x5b\xed\xb0\x96\xf9\xc0\x0c\x00\x2e\x00\x01\x00\x00\x0e\x10\x00\
    \x5c\x00\x0f\x0d\x02\x00\x00\x0e\x10\x61\xa3\x9a\x00\x61\x7c\x0d\
    \x00\x68\x47\x04\x6e\x69\x78\x64\x03\x6f\x72\x67\x00\x07\x1b\x42\
    \xed\x2b\xdb\x51\x32\x5a\x31\x1c\x42\xb3\x07\xd2\x85\xdb\xda\x3d\
    \x77\x06\xc8\x9f\xb5\x33\x04\x99\x45\xf1\xa8\x62\xed\x22\xb2\x08\
    \x20\x35\xc1\x72\x78\x08\x1c\x54\x94\xd8\x24\x34\x8d\x00\x04\x01\
    \xa6\xfe\x75\x74\xbc\x4c\xcb\x51\xac\x29\x51\xc8\x9e\xc0\x0c\x00\
    \x2e\x00\x01\x00\x00\x0e\x10\x00\x5c\x00\x10\x0d\x02\x00\x00\x0e\
    \x10\x61\xa3\x9a\x00\x61\x7c\x0d\x00\x68\x47\x04\x6e\x69\x78\x64\
    \x03\x6f\x72\x67\x00\xc8\x21\x2e\x0f\xee\x95\x5a\xa6\x79\x45\x11\
    \x51\x6c\xf3\xfd\x22\x69\xd3\x8f\x12\xc8\x63\xd4\x1d\x1c\x53\x46\
    \x44\xc6\xbb\xed\x3a\x7d\xa5\x2e\x23\x38\x96\xfc\x9c\x7f\xdb\x73\
    \x56\x58\x99\xa3\xe9\x39\xe0\x1c\x8d\x0f\xec\x11\xa4\x46\x44\x76\
    \x25\x41\x27\x2d\x17\xc0\x0c\x00\x2e\x00\x01\x00\x00\x0e\x10\x00\
    \x5c\x00\x30\x0d\x02\x00\x00\x0e\x10\x61\xa3\x9a\x00\x61\x7c\x0d\
    \x00\x68\x47\x04\x6e\x69\x78\x64\x03\x6f\x72\x67\x00\x78\xf4\x67\
    \xde\x5c\x2b\x23\xec\x49\x45\xee\x43\x05\x30\xdb\x27\x84\x2d\x14\
    \xc3\x46\x9c\x8c\x4b\x02\x25\xc1\x65\x55\x79\xe6\x21\x30\x8d\xad\
    \x0a\x58\xd7\xa5\xc5\xd6\xaa\xd5\x11\x97\xc2\xd3\x9b\xe1\x8c\x76\
    \x92\xd0\x64\x78\xac\x62\x23\xfc\x47\xf4\xce\x14\xfa\xc0\x0c\x00\
    \x2e\x00\x01\x00\x00\x0e\x10\x00\x5c\x00\x30\x0d\x02\x00\x00\x0e\
    \x10\x61\xa3\x9a\x00\x61\x7c\x0d\x00\xf3\xc9\x04\x6e\x69\x78\x64\
    \x03\x6f\x72\x67\x00\x4d\x20\xf8\xc2\x89\x3d\xa4\x79\xa8\x75\xfa\
    \x19\x7b\x48\x86\x55\x66\xd5\x7b\x4e\xa5\x9e\x61\x6f\x44\x9b\x2b\
    \xd1\xff\xe3\xcd\xfe\xe3\xab\x68\xe6\x0b\x27\x4c\x76\x7f\x0a\x90\
    \x2e\x4d\xe1\x70\x15\x7e\xea\x1b\xc8\x59\xac\x5b\x8c\xbc\x9d\x50\
    \xb3\x31\xd7\x83\xdc\xc0\x0c\x00\x2e\x00\x01\x00\x00\x00\x01\x00\
    \x5c\x00\x33\x0d\x02\x00\x00\x00\x00\x61\xa3\x9a\x00\x61\x7c\x0d\
    \x00\x68\x47\x04\x6e\x69\x78\x64\x03\x6f\x72\x67\x00\x1f\x27\x14\
    \xc4\x02\x52\xbf\x78\x19\x16\x74\xe8\xcf\x7c\x24\x88\x3b\x39\x2c\
    \x76\xc4\x1f\xab\xca\x19\x3b\x91\x40\x2c\x99\xa5\x3b\x40\x3a\x3c\
    \x85\x52\x4d\x17\x1b\xbb\xe4\xa2\xe6\x80\xde\x4f\x9b\x11\xa2\x4c\
    \xc6\xab\xb8\x05\xed\xee\xc2\x44\x49\x7c\x88\xc1\x69\xc0\x0c\x00\
    \x02\x00\x01\x00\x00\x0e\x10\x00\x02\xc0\x44\xc0\x0c\x00\x02\x00\
    \x01\x00\x00\x0e\x10\x00\x02\xc0\x70\xc0\x0c\x00\x02\x00\x01\x00\
    \x00\x0e\x10\x00\x02\xc1\x78\xc0\x0c\x00\x02\x00\x01\x00\x00\x0e\
    \x10\x00\x02\xc0\x26";
    
    let dummy = 
        DnsRequestHeader
        {
            header: DnsHeader {id: 0xb59a, ..Default::default()},
            payload: DnsRequestPayload
                {
                    qname: b"\x04\x6e\x69\x78\x64\x03\x6f\x72\x67\x00".to_vec(),
                    qtype: QType::ALL,
                    qclass: QClass::IN, 
                }
        };

    let now = Instant::now();
    let ans = DnsRequestAnswer::parse(pkt, &dummy);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(ans.is_ok(), true, "{}", ans.err().unwrap());
    let ans = ans.unwrap();

    assert_eq!(ans.header.id, 0xb59a);
    assert_eq!(ans.header.ancount, 17);
    assert_eq!(ans.header.arcount, 0);
    assert_eq!(ans.header.nscount, 4);
    assert_eq!(ans.header.qdcount, 1);

    let ansord = &ans.response[0];
    assert_eq!(ansord.rdata, DnsRdata::NS{ fqdn: "ns24.cloudns.net".to_string() });
    assert_eq!(ansord.name, "nixd.org".to_string());
    assert_eq!(ansord.dtype, QType::NS);
    assert_eq!(ansord.class, QClass::IN);
    assert_eq!(ansord.ttl, 3600);
    assert_eq!(ansord.rdlength, 18);

    let ansord = &ans.response[1];
    assert_eq!(ansord.rdata, DnsRdata::NS{ fqdn: "ns23.cloudns.net".to_string() });
    assert_eq!(ansord.name, "nixd.org".to_string());
    assert_eq!(ansord.dtype, QType::NS);
    assert_eq!(ansord.class, QClass::IN);
    assert_eq!(ansord.ttl, 3600);
    assert_eq!(ansord.rdlength, 7);

    let ansord = &ans.response[2];
    assert_eq!(ansord.rdata, DnsRdata::NSEC3PARAM{ data: b"\x01\x00\x00\x0a\x08\xcb\x22\x97\x04\x93\xda\x7f\x9c".to_vec() });
    assert_eq!(ansord.name, "nixd.org".to_string());
    assert_eq!(ansord.dtype, QType::NSEC3PARAM);
    assert_eq!(ansord.class, QClass::IN);
    assert_eq!(ansord.ttl, 1);
    assert_eq!(ansord.rdlength, 13);
}
