/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/// This file contains the config file parsers.

use std::borrow::Borrow;
use std::hash::{Hash, Hasher};
use std::io::Read;
use std::fs::File;
use std::net::IpAddr;

use crate::{write_error, internal_error, internal_error_map, error::*};

use crate::portable::*;
use crate::tokenizer::*;
use super::common::{RESOLV_CFG_PATH, HOST_CFG_PATH};

fn read_file(path: &str) -> CDnsResult<String>
{
    let mut file = 
        File::open(path).map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

    let mut file_content: String = String::new();

    file.read_to_string(&mut file_content).map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

    return Ok(file_content);
}

/// A resolve.conf parser
#[derive(Clone, Debug)]
pub struct ResolveConfEntry 
{
    ip: IpAddr,
    adapter_ip: Option<IpAddr>,
}

impl ResolveConfEntry
{
    pub 
    fn get_resolver_ip(&self) -> &IpAddr
    {
        return &self.ip;
    }

    pub 
    fn get_adapter_ip(&self) -> Option<&IpAddr>
    {
        return self.adapter_ip.as_ref();
    }

    pub 
    fn parse_resolve_cfg() -> CDnsResult<Vec<ResolveConfEntry>>
    {
        let file_content = read_file(RESOLV_CFG_PATH)?;
        
        return Self::parser_resolv_internal(file_content);
    }

    unsafe
    fn split_dns_ifr(field_data: &str, ifr: Option<&IfInfo>) -> CDnsResult<(IpAddr, Option<IpAddr>)>
    {
        if let Some((ip, intf)) = field_data.split_once("%")
        {   
            let ip_addr: IpAddr = 
                match ip.parse()    
                {
                    Ok(r) => r,
                    Err(e) =>
                        internal_error!(CDnsErrorType::InternalError, "parse IP: '{}' failed: '{}'", ip, e)
                };

            if ifr.is_none() == true
            {
                internal_error!(
                    CDnsErrorType::InternalError, 
                    "can not add: '{}' to list because interface data is not available!", 
                    field_data
                );
            } 

            let ifrr = ifr.unwrap();

            match ifrr.get_ifr_ip(intf, &ip_addr)?
            {
                Some(ip_ifr) => return Ok((ip_addr, Some(ip_ifr))),
                None => internal_error!(CDnsErrorType::InternalError, "interface: '{}' does not exist", intf),
            }
        }
        else
        {
            let ip_addr: IpAddr = 
                match field_data.parse()    
                {
                    Ok(r) => r,
                    Err(e) =>
                        internal_error!(CDnsErrorType::InternalError, "parse IP: '{}' failed: '{}'", field_data, e)
                };

            return Ok((ip_addr, None))
        }
    }

    fn parser_resolv_internal(file_content: String) -> CDnsResult<Vec<ResolveConfEntry>>
    {
        let mut tk = ConfTokenizer::from_str(&file_content)?;
        let mut dns_list: Vec<ResolveConfEntry> = Vec::new();
        let if_info: Option<IfInfo> = 
            unsafe
            {
                match IfInfo::get_interfaces_info()
                {
                    Ok(r) => Some(r),
                    Err(e) =>
                    {
                        write_error!("{}", e);
                        None
                    }
                }
            };

        loop
        {
            let field_name = tk.read_next()?;

            if field_name.is_none() == true
            {
                // reached EOF
                break;
            }

            let field_name = field_name.unwrap();

            if field_name == "nameserver"
            {
                // read second component
                let field_data = tk.read_next()?;

                if field_data.is_none() == true
                {
                    // reached unexpected EOF
                    write_error!("unexpected EOF near nameserver");
                    break;
                }

                let field_data = field_data.unwrap();

                // parse IP and convert Interface to its IP
                let (dns_ip, ifr_ip) = 
                    unsafe
                    {
                        match Self::split_dns_ifr(field_data, if_info.as_ref())
                        {
                            Ok((d_ip, i_ip)) => (d_ip, i_ip),
                            Err(e) =>
                            {
                                write_error!("{}", e);
                                continue;
                            }
                        }
                    };
                    

                    dns_list.push(
                    ResolveConfEntry
                    { 
                        ip: dns_ip,
                        adapter_ip: ifr_ip,
                    }
                );
                
                
            }
            else
            {
                // skip until end of line or EOF
                tk.skip_upto_eol();
            }
        }

        return Ok(dns_list);
    }
}

/// An /etc/hosts file parser
#[derive(Clone, Debug)]
pub struct HostnameEntry
{
    ip: IpAddr,
    hostnames: Vec<String>,
}

impl Eq for HostnameEntry {}

impl PartialEq for HostnameEntry 
{
    fn eq(&self, other: &HostnameEntry) -> bool 
    {
        return self.ip == other.ip;
    }
}

impl Borrow<IpAddr> for HostnameEntry
{
    fn borrow(&self) -> &IpAddr 
    {
        return &self.ip;    
    }
}

impl Hash for HostnameEntry 
{
    fn hash<H: Hasher>(&self, state: &mut H) 
    {
        self.ip.hash(state);
    }
}


impl HostnameEntry
{
    /// Returns the IP address
    pub 
    fn get_ip(&self) -> &IpAddr
    {
        return &self.ip;
    }

    /// Returns the slice with the hostnames
    pub 
    fn get_hostnames(&self) -> &[String]
    {
        return self.hostnames.as_slice();
    }

    /// Returns the iterator of the hostnames
    pub 
    fn get_hostnames_iter(&self) -> std::slice::Iter<'_, String>
    {
        return self.hostnames.iter();
    }

    /// Parses the /etc/hosts file and returns a vector of records from file
    pub 
    fn parse_host_file(output_err: bool) -> CDnsResult<Vec<HostnameEntry>>
    {
        let file_content = read_file(HOST_CFG_PATH)?;
        
        return Self::parse_host_file_internal(file_content, output_err);
    }

    fn parse_host_file_internal(file_content: String, output_err: bool) -> CDnsResult<Vec<HostnameEntry>>
    {
        let mut tk = ConfTokenizer::from_str(&file_content)?;
        let mut he_list: Vec<HostnameEntry> = Vec::new();

        loop
        {
            let field_ip = tk.read_next()?;
    
            if field_ip.is_none() == true
            {
                // reached EOF
                break;
            }

            //println!("ip: {}", field_ip.as_ref().unwrap());
    
            let ip: IpAddr = 
                match field_ip.unwrap().parse()
                {
                    Ok(r) => r,
                    Err(_e) => 
                    {
                        // skip
                        tk.skip_upto_eol();
                        continue;
                    }
                };
    
            let hostnames = tk.read_upto_eol()?;

            if hostnames.len() > 0
            {
                let he = HostnameEntry{ ip: ip, hostnames: hostnames };
                he_list.push(he);
            }
            else
            {
                if output_err == true
                {
                    write_error!("in file: '{}' IP is not defined with domain name: '{}'", HOST_CFG_PATH, ip);
                }
            }
        }

        if he_list.len() == 0
        {
            if output_err == true
            {
                write_error!("file: '{}' file is empty or damaged!", HOST_CFG_PATH);
            }
        }

        return Ok(he_list);
    }
    

}

/*
pub enum NsSwitchSources
{
    Files,
    Dns
}
/// A parser for /etc/nsswitch.conf
pub struct NsSwitchEntry
{
    section: String,
    search_order: Vec<String>
}
*/


#[test]
fn test_parser_resolv_internal_0() 
{
    let ip1: IpAddr = "192.168.2.1".parse().unwrap();
    let ip2: IpAddr = "127.0.0.1".parse().unwrap();
    let ip3: IpAddr = "8.8.8.8".parse().unwrap();
    let ip4: IpAddr = "127.0.0.1".parse().unwrap();
    let ip4_if: IpAddr = "127.0.0.1".parse().unwrap();
    let ip5: IpAddr = "fe80::1".parse().unwrap();
    let ip5_if: IpAddr = "::1".parse().unwrap();

    let test = 
    "nameserver 192.168.2.1\n\
    nameserver  127.0.0.1\n\
    lookup dns.ns.ts\n\
    nameserver  8.8.8.8\n
    nameserver 127.0.0.1%lo\n
    nameserver fe80::1%lo\n".to_string();

    let res = ResolveConfEntry::parser_resolv_internal(test);
    assert_eq!(res.is_ok(), true, "{}", res.err().unwrap());
    let res = res.unwrap();

    assert_eq!(res[0].ip, ip1);
    assert_eq!(res[1].ip, ip2);
    assert_eq!(res[2].ip, ip3);
    assert_eq!(res[3].ip, ip4);
    assert_eq!(res[3].adapter_ip.is_some(), true);
    assert_eq!(res[3].adapter_ip.as_ref().unwrap(), &ip4_if);
    assert_eq!(res[4].ip, ip5);
    assert_eq!(res[4].adapter_ip.is_some(), true);
    assert_eq!(res[4].adapter_ip.as_ref().unwrap(), &ip5_if);
}

#[test]
fn test_parser_resolv_internal() 
{
    let ip1: IpAddr = "192.168.2.1".parse().unwrap();
    let ip2: IpAddr = "127.0.0.1".parse().unwrap();
    let ip3: IpAddr = "8.8.8.8".parse().unwrap();

    let test = 
    "nameserver 192.168.2.1\n\
    nameserver  127.0.0.1\n\
    lookup dns.ns.ts\n\
    nameserver  8.8.8.8".to_string();

    let res = ResolveConfEntry::parser_resolv_internal(test);
    assert_eq!(res.is_ok(), true, "{}", res.err().unwrap());
    let res = res.unwrap();

    assert_eq!(res[0].ip, ip1);
    assert_eq!(res[1].ip, ip2);
    assert_eq!(res[2].ip, ip3);
}

#[test]
fn test_parser_resolv_internal2() 
{
    let ip1: IpAddr = "192.168.2.1".parse().unwrap();
    let ip2: IpAddr = "127.0.0.1".parse().unwrap();
    let ip3: IpAddr = "8.8.8.8".parse().unwrap();

    let test = 
    "# test 12345\n\
    # nameserver 192.168.3.1
    nameserver 192.168.2.1  \n\
    nameserver  127.0.0.1   \n\
    lookup dns.ns.ts\n\
    nameserver  8.8.8.8\n".to_string();

    let res = ResolveConfEntry::parser_resolv_internal(test);
    assert_eq!(res.is_ok(), true, "{}", res.err().unwrap());
    let res = res.unwrap();

    assert_eq!(res[0].ip, ip1);
    assert_eq!(res[1].ip, ip2);
    assert_eq!(res[2].ip, ip3);
}

#[test]
fn test_parse_host_file_0()
{
    let hosts1: Vec<&'static str> = vec!["debian-laptop"];
    let hosts2: Vec<&'static str> = vec!["localhost", "ip6-localhost", "ip6-loopback"];
    let hosts3: Vec<&'static str> = vec!["ip6-allnodes"];
    let hosts4: Vec<&'static str> = vec!["ip6-allrouters"];

    let ip1:IpAddr = "127.0.1.1".parse().unwrap();
    let ip2:IpAddr = "::1".parse().unwrap();
    let ip3:IpAddr = "ff02::1".parse().unwrap();
    let ip4:IpAddr = "ff02::2".parse().unwrap();

    let test =
    "127.0. 0.1	localhost
    127.0.1.1	debian-laptop
    
    # The following lines are desirable for IPv6 capable hosts
    ::1     localhost ip6-localhost ip6-loopback
    ff02::1 ip6-allnodes
    ff02::2 ip6-allrouters".to_string();

    let p = HostnameEntry::parse_host_file_internal(test, true);
    assert_eq!(p.is_ok(), true, "{}", p.err().unwrap());

    let p = p.unwrap();

    assert_eq!(p[0].ip, ip1);
    assert_eq!(p[1].ip, ip2);
    assert_eq!(p[2].ip, ip3);
    assert_eq!(p[3].ip, ip4);

    assert_eq!(p[0].hostnames, hosts1);
    assert_eq!(p[1].hostnames, hosts2);
    assert_eq!(p[2].hostnames, hosts3);
    assert_eq!(p[3].hostnames, hosts4);

    return;
}

#[test]
fn test_parse_host_file()
{
    let ip0:IpAddr = "127.0.0.1".parse().unwrap();
    let ip1:IpAddr = "127.0.1.1".parse().unwrap();
    let ip2:IpAddr = "::1".parse().unwrap();
    let ip3:IpAddr = "ff02::1".parse().unwrap();
    let ip4:IpAddr = "ff02::2".parse().unwrap();

    let hosts0: Vec<&'static str> = vec!["localhost"];
    let hosts1: Vec<&'static str> = vec!["debian-laptop"];
    let hosts2: Vec<&'static str> = vec!["localhost", "ip6-localhost", "ip6-loopback"];
    let hosts3: Vec<&'static str> = vec!["ip6-allnodes"];
    let hosts4: Vec<&'static str> = vec!["ip6-allrouters"];

    let test =
    "127.0.0.1	localhost
    127.0.1.1	debian-laptop
    
    # The following lines are desirable for IPv6 capable hosts
    ::1     localhost ip6-localhost ip6-loopback
    ff02::1 ip6-allnodes
    ff02::2 ip6-allrouters".to_string();

    let p = HostnameEntry::parse_host_file_internal(test, true);
    assert_eq!(p.is_ok(), true, "{}", p.err().unwrap());

    let p = p.unwrap();

    assert_eq!(p[0].ip, ip0);
    assert_eq!(p[1].ip, ip1);
    assert_eq!(p[2].ip, ip2);
    assert_eq!(p[3].ip, ip3);
    assert_eq!(p[4].ip, ip4);

    assert_eq!(p[0].hostnames, hosts0);
    assert_eq!(p[1].hostnames, hosts1);
    assert_eq!(p[2].hostnames, hosts2);
    assert_eq!(p[3].hostnames, hosts3);
    assert_eq!(p[4].hostnames, hosts4);
}

#[test]
fn test_parse_host_file_2()
{
    let ip0:IpAddr = "127.0.0.1".parse().unwrap();
    let ip1:IpAddr = "127.0.1.1".parse().unwrap();
    let ip2:IpAddr = "::1".parse().unwrap();
    let ip3:IpAddr = "ff02::1".parse().unwrap();
    let ip4:IpAddr = "ff02::2".parse().unwrap();

    let hosts0: Vec<&'static str> = vec!["localhost", "localdomain", "domain.local"];
    let hosts1: Vec<&'static str> = vec!["debian-laptop", "test123.domain.tld"];
    let hosts2: Vec<&'static str> = vec!["localhost", "ip6-localhost", "ip6-loopback"];
    let hosts3: Vec<&'static str> = vec!["ip6-allnodes"];
    let hosts4: Vec<&'static str> = vec!["ip6-allrouters"];

    let test =
    "127.0.0.1	localhost localdomain domain.local
    127.0.1.1	debian-laptop test123.domain.tld
    
    # The following lines are desirable for IPv6 capable hosts
    #
    #
    ::1     localhost ip6-localhost ip6-loopback
    ff02::1 ip6-allnodes
    ff02::2 ip6-allrouters
    ".to_string();

    let p = HostnameEntry::parse_host_file_internal(test, true);
    assert_eq!(p.is_ok(), true, "{}", p.err().unwrap());

    let p = p.unwrap();

    assert_eq!(p[0].ip, ip0);
    assert_eq!(p[1].ip, ip1);
    assert_eq!(p[2].ip, ip2);
    assert_eq!(p[3].ip, ip3);
    assert_eq!(p[4].ip, ip4);

    assert_eq!(p[0].hostnames, hosts0);
    assert_eq!(p[1].hostnames, hosts1);
    assert_eq!(p[2].hostnames, hosts2);
    assert_eq!(p[3].hostnames, hosts3);
    assert_eq!(p[4].hostnames, hosts4);
}

