/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

//! cdns-rs
//! 
//! An implementation of client side DNS query library - a resolver, a set of 
//! routines that provide access to the Internet Domain Name System (DNS). 
//! 
//! This library provides both:
//! sync and async crates to query remote DNS servers and local hosts file.
//! 
//! This code will be considered stable only when the major part of verion number
//! will reach v 1.x.x
//! 
//! # Limitations
//! 
//! - This library reads /etc/hosts and partially understands /etc/resolv.conf i.e
//! only 'nameserver' option is parsed at the moment.
//! - This library does not support TCP query at the moment.
//! - The nsswitch.conf is also ignored as this crate at the moment  is not pretending 
//!     to be a glibc/libc replacement.
//! - Does not support DNS-over-TLS at the moment
//! - Does not support DNSSEC at the moment
//! - /etc/resolve.conf is limited to 'nameserver'
//! 
//! # Supports
//! 
//! Supported GNU/Linux and *BSD (incl OSX, but not guaranteed)
//! 
//! # Layout
//! 
//! Dirs: 
//! - a_sync - contains all async code
//! - sync - contains all sync code
//! Files:
//! - portable.rs - contains the code which requres porting to other OS. 
//!     This code usually incompatiable with different UNIX-type OSes.
//! - tokenizer.rs - contains simple tokenizer for parsing config files.
//! - caches.rs - contains a cached in memory config files like /etc/hosts.
//!     If file was modified, the cache controller attempts to reload file during
//!     library call from main program. 
//! - cfg_parsers.rs - contains the confgiration files parsers.
//! - common.rs - this file contains all code related to DNS protocol i.e structures,
//!     parsing.
//! - mutex.rs - a local implementaion of mutex for dafu usage in multithreading env.
//!     Derived from rust's mutex implementation.
//! - network.rs - a code which deals with networking part, i.e msg exchange with DNS
//!     server.
//! - poison.rs - a mutex poison detection derived from rust's code.
//! - query.rs - file contains a code which combines `common.rs`, `networking.rs` and cache
//!     managment. Also convers raw result to a structured data tables.
//! - shortcuts.rs - a code for faster prototyping i.e MX records requests or A or AAAA.
//! - error.rs - an error wrapper and mapper
//! 
//! Features: 
//! - feature = "use_async" for asynchronious code (use cdns_rs::a_sync::{...};)
//! - feature = "use_sync" for synchronious code (use cdns_rs::sync::{...};)
//! - feature = "stderr_output" to enable output of soft errors to stderr
//!
//! All features can be used simultaniously.
//! # Usage
//! 
//! cdns-rs = {version = "0.1", default-features = false, features = ["use_sync"]}
//! 
//! By default, both `use_async` and `use_sync` features are enabled.


extern crate rand;
#[macro_use] extern crate lazy_static;
extern crate libc;
#[macro_use] extern crate bitflags;

#[cfg(feature = "use_sync")]
extern crate byteorder;
#[cfg(feature = "use_sync")]
extern crate crossbeam_utils;
#[cfg(feature = "use_sync")]
pub mod sync;

#[cfg(feature = "use_async")]
extern crate async_recursion;
#[cfg(feature = "use_async")]
extern crate tokio;
#[cfg(feature = "use_async")]
extern crate tokio_byteorder;
#[cfg(feature = "use_async")]
pub mod a_sync;

pub mod portable;
#[macro_use] pub mod error;
pub mod tokenizer;
#[macro_use] pub mod log; 

pub use lazy_static::*;
use error::*;
use log::*;
