/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/


use std::collections::LinkedList;
use std::convert::{TryFrom, TryInto};
use std::net::{IpAddr, Ipv4Addr, Ipv6Addr, SocketAddr};
use std::sync::Arc;
use std::{fmt, str};
use tokio::time::Duration;
use tokio::time::Instant;

use crate::error::*;
use crate::{write_error, internal_error, internal_error_map};

use super::cfg_parsers::ResolveConfEntry;
use super::common::*;
use super::caches::CACHE;
use super::network::NetworkTap;

#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum QueryMode
{
    /// In defualt mode, on the first successfull response, the library returns the result
    /// to main program. In case of NXDOMAIN error, it stops returning error ignoring the rest
    /// resolvers.
    DefaultMode,

    /// In `ForceNxQueryAll` mode, in case of NXDOMAIN, the library will try to query the rest
    /// nameservers, if any
    ForceNxQueryAll,

    /// In `ForceQueryAll` mode, the library will query all available nameservers and return all results.
    /// In case of error, the result will also be returned.
    ForceQueryAll,
}

impl Default for QueryMode
{
    fn default() -> Self 
    {
        return Self::DefaultMode;    
    }
}


#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub struct QuerySetup
{
    /// Sets the realtime counter to measure delay
    pub(crate) measure_time: bool, 

    /// Forces to ignore hosts file
    pub(crate) ign_hosts: bool, 

    /// Sets the query mode
    pub(crate) qmode: QueryMode,

    /// Sets the timeout duration for reply awaiting.
    pub(crate) timeout: Option<Duration>,
}

impl Default for QuerySetup
{
    fn default() -> Self 
    {
        return Self { measure_time: false, ign_hosts: false, timeout: Some(Duration::from_secs(1)), qmode: QueryMode::default() };
    }
}

impl QuerySetup
{
    pub 
    fn new(measure_time: bool, ign_hosts: bool, qmode: QueryMode, timeout: Option<Duration>) -> Self
    {
        return Self { measure_time, ign_hosts, qmode, timeout };
    }
}


#[derive(Clone, Debug)]
pub enum QDnsQueryRec
{
    /// Answer from the DNS server
    Ok(Vec<DnsResponsePayload>),
    /// Server side error
    ServFail,
    /// Query does not exist, but meaningfull when `aa` is true
    NxDomain,
    /// A name server refuses to perform the specified operation
    Refused,
    /// A name server does not support the requested kind of query
    NotImpl,
    /// A fromat error
    FormError,
}

impl QDnsQueryRec
{
    pub 
    fn is_ok(&self) -> bool
    {
        match *self
        {
            Self::Ok(_) => return true,
            _ => return false,
        }
    }
}

impl fmt::Display for QDnsQueryRec
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::Ok(ref quries) => 
            {
                for q in quries.iter()
                {
                    writeln!(f, "{}", q)?;
                }
            },
            Self::ServFail =>
            {
                writeln!(f, "SERVFAIL")?;
            },
            Self::NxDomain =>
            {
                writeln!(f, "NXDOMAIN")?;
            },
            Self::Refused => 
            {   
                writeln!(f, "REFUSED")?;
            },
            Self::NotImpl => 
            {
                writeln!(f, "NOT IMPLEMENTED")?;
            },
            Self::FormError =>
            {
                writeln!(f, "FORMAT ERROR")?;
            }
        }

        return Ok(());
    }
}

/// A structure which describes the query properties and contains the
/// results.
#[derive(Clone, Debug)]
pub struct QDnsQuery
{
    /// A realtime time elapsed for query
    elapsed: Option<Duration>,
    /// Server which performed the response and port number
    server: String,
    /// Authoritative Answer
    aa: bool,
    /// Authoratives section
    authoratives: Vec<DnsResponsePayload>,
    /// Responses
    rec: QDnsQueryRec,
}

impl fmt::Display for QDnsQuery
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        write!(f, "Source: {} ", self.server)?;
        if let Some(ref el) = self.elapsed
        {
            write!(f, "{:.2?} ", el)?;
        }

        if self.aa == true
        {
            writeln!(f, "Authoritative answer")?;
        }
        else
        {
            writeln!(f, "Non-Authoritative answer")?;
        }

        writeln!(f, "Authoritatives: ")?;

        for a in self.authoratives.iter()
        {
            writeln!(f, "{}", a)?;
        }

        writeln!(f, "\nAnswer: ")?;

        return writeln!(f, "{}", self.rec);
    }
    
}

impl QDnsQuery
{
    /// Clones the responses which contains non empty successfull responses from DNS
    /// server.
    pub 
    fn collect_ok_responses(&self) -> Vec<DnsRdata>
    {
        let mut out: Vec<DnsRdata> = Vec::new();

        match &self.rec
        {
            QDnsQueryRec::Ok(recs) =>
            {
                for r in recs
                {
                    if r.rdata.is_some() == true
                    {
                        out.push(r.rdata.clone());
                    }
                }
            },
            _ => {}
        }

        return out;
    }

    /// Creates references to responses which contains non empty successfull 
    /// responses from DNS server.
    pub 
    fn collect_ok_responses_ref(&self) -> Vec<&DnsRdata>
    {
        let mut out: Vec<&DnsRdata> = Vec::new();

        match &self.rec
        {
            QDnsQueryRec::Ok(recs) =>
            {
                for r in recs
                {
                    if r.rdata.is_some() == true
                    {
                        out.push(&r.rdata);
                    }
                }
            },
            _ => {}
        }

        return out;
    }
    
    pub 
    fn is_authorative(&self) -> bool
    {
        return self.aa;
    }

    pub 
    fn get_elapsed_time(&self) -> Option<&Duration>
    {
        return self.elapsed.as_ref();
    }

    pub 
    fn get_server(&self) -> &String
    {
        return &self.server;
    }

    pub 
    fn get_authoratives(&self) -> &[DnsResponsePayload]
    {
        return self.authoratives.as_slice();
    }

    pub 
    fn get_responses(&self) -> &QDnsQueryRec
    {
        return &self.rec;
    }
}

impl QDnsQuery
{
    /// Constructs instance like it is from 'local' source but not from DNS server.
    pub 
    fn from_local(req_pl: Vec<DnsResponsePayload>, now: Option<&Instant>) -> QDnsQuery
    {
        let elapsed = 
            match now
            {
                Some(n) => Some(n.elapsed()),
                None => None
            };

        return 
            Self
            {
                elapsed: elapsed,
                server: HOST_CFG_PATH.to_string(),
                aa: true,
                authoratives: Vec::new(),
                rec: QDnsQueryRec::Ok(req_pl),
            };
    }

    /// Constructs an instance from the remote response.
    pub 
    fn from_response(server: &SocketAddr, ans: DnsRequestAnswer, now: Option<&Instant>) -> CDnsResult<Self>
    {
        // parsing header
        let aa = ans.header.status.contains(StatusBits::AUTH_ANSWER);

        let qr: QDnsQueryRec =
            if ans.header.status.contains(StatusBits::RESP_NOERROR) == true
            {
                /*let resps: Vec<QueryRec> = 
                    ans.response.into_iter().map(|a| a.into()).collect();
                */
                QDnsQueryRec::Ok(ans.response)
            }
            else if ans.header.status.contains(StatusBits::RESP_FORMERR) == true
            {
                QDnsQueryRec::FormError
            }
            else if ans.header.status.contains(StatusBits::RESP_NOT_IMPL) == true
            {
                QDnsQueryRec::NotImpl
            }
            else if ans.header.status.contains(StatusBits::RESP_NXDOMAIN) == true
            {
                QDnsQueryRec::NxDomain
            }
            else if ans.header.status.contains(StatusBits::RESP_REFUSED) == true
            {
                QDnsQueryRec::Refused
            }
            else if ans.header.status.contains(StatusBits::RESP_SERVFAIL) == true
            {
                QDnsQueryRec::ServFail
            }
            else
            {
                internal_error!(CDnsErrorType::DnsResponse, "response status bits unknwon result: '{}'", ans.header.status.bits());
            };

        let elapsed = 
            match now
            {
                Some(n) => Some(n.elapsed()),
                None => None
            };

        return Ok(
            Self
            {
                elapsed: elapsed,
                server: server.to_string(),
                aa: aa,
                authoratives: ans.authoratives,
                rec: qr,
            }
        );
    }
}

/*
#[derive(Debug)]
pub enum QDnsQueriesRes
{
    DnsResultSingle
    {
        res: QDnsQuery,
    },
    DnsResultMultiple
    {
        res: Vec<QDnsQuery>,
    },
    DnsNotAvailable
}

impl QDnsQueriesRes
{
    pub 
    fn is_results(&self) -> bool
    {
        match *self
        {
            Self::DnsNotAvailable => return false,
            _ => return true,
        }
    }
}

impl From<LinkedList<QDnsQuery>> for QDnsQueriesRes
{
    fn from(mut responses: LinkedList<QDnsQuery>) -> Self 
    {
        if responses.len() == 1
        {
            return QDnsQueriesRes::DnsResultSingle{ res: responses.pop_back().unwrap() };
        }
        else if responses.len() > 1
        {
            return QDnsQueriesRes::DnsResultMultiple{ res: responses.into_iter().map(|h| h).collect() };
        }
        else
        {
            return QDnsQueriesRes::DnsNotAvailable;
        }
    }
}

impl fmt::Display for QDnsQueriesRes
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::DnsResultSingle{ ref res } =>
            {
                writeln!(f, "{}", res)?;
            },
            Self::DnsResultMultiple{ ref res } =>
            {
                for r in res.iter()
                {
                    write!(f, "{}", r)?;
                }
            },
            Self::DnsNotAvailable => write!(f, "No DNS server available")?
        }

        return Ok(());
    }
}*/

#[derive(Debug)]
pub enum QDnsQueriesRes
{
    DnsOk
    {
        res: Vec<QDnsQuery>,
    },
    DnsNotAvailable
}

impl QDnsQueriesRes
{
    pub 
    fn into_inner(self) -> Option<Vec<QDnsQuery>>
    {
        match self
        {
            Self::DnsNotAvailable => None,
            Self::DnsOk{ res } => return Some(res),
        }
    }
    
    pub 
    fn is_results(&self) -> bool
    {
        match *self
        {
            Self::DnsNotAvailable => return false,
            _ => return true,
        }
    }

    pub 
    fn len(&self) -> usize
    {
        if let Self::DnsOk{res} = self
        {
            return res.len();
        }
        else
        {
            return 0;
        }
    }

    pub 
    fn list_results(&self) -> Option<std::slice::Iter<'_, QDnsQuery>>
    {
        if let Self::DnsOk{res} = self
        {
            return Some(res.iter());
        }
        else
        {
            return None;
        }
    }
}

impl From<QDnsQuery> for QDnsQueriesRes
{
    fn from(query: QDnsQuery) -> Self 
    {
        return QDnsQueriesRes::DnsOk{ res: vec![query] };
    }
}

impl From<LinkedList<QDnsQuery>> for QDnsQueriesRes
{
    fn from(responses: LinkedList<QDnsQuery>) -> Self 
    {
        if responses.len() > 0
        {
            return QDnsQueriesRes::DnsOk{ res: responses.into_iter().map(|h| h).collect() };
        }
        else
        {
            return QDnsQueriesRes::DnsNotAvailable;
        }
    }
}

impl fmt::Display for QDnsQueriesRes
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result 
    {
        match *self
        {
            Self::DnsOk{ ref res } =>
            {
                for r in res.iter()
                {
                    write!(f, "{}", r)?;
                }
            },
            Self::DnsNotAvailable => write!(f, "No DNS server available")?
        }

        return Ok(());
    }
}

/// An structure for the target in request.
/// Implements From for [IpAddr], [Ipv4Addr], [Ipv6Addr], [str]
/// In case if IP address is in the human readable 'string' format
/// it will be stored as Name. And later the code will try to convert it
/// to IP address before returing as string.
pub enum QDnsName<'temp>
{
    //Ip(&'temp IpAddr),
    IpV4(&'temp Ipv4Addr),
    IpV6(&'temp Ipv6Addr),
    Name(&'temp str),
}

impl<'temp>  QDnsName<'temp>
{
    /// Answers if current type is IpV4.
    /// But does not performs check if Name is an IP in str format
    pub
    fn is_ipv4(&self) -> bool
    {
        match *self
        {
            Self::IpV4(_) => return true,
            _ => return false,
        }
    }

    /// Answers if current type is IpV6.
    /// But does not performs check if Name is an IP in str format
    pub 
    fn is_ipv6(&self) -> bool
    {
        match *self
        {
            Self::IpV6(_) => return true,
            _ => return false,
        }
    }

    /// Answers if current type is IpV4 or IpV6.
    /// But does not performs check if Name is an IP in str format
    pub 
    fn is_ip(&self) -> bool
    {
        match *self
        {
            Self::IpV4(_) => return true,
            Self::IpV6(_) => return true,
            _ => return false,
        }
    }

    /// Returns the type of the IP address version in [QType] format.
    pub 
    fn get_ip_qtype(&self) -> Option<QType>
    {
        match *self
        {
            Self::IpV4(_) => return Some(QType::A),
            Self::IpV6(_) => return Some(QType::AAAA),
            Self::Name(name) =>
            {
                if let Ok(_) = name.parse::<Ipv4Addr>()
                {   
                    return Some(QType::A);
                }
                else if let Ok(_) = name.parse::<Ipv6Addr>()
                {
                    return Some(QType::AAAA);
                }
                else
                {
                    return None;
                }
            }
        }
    }
}

impl<'temp> PartialEq<str> for QDnsName<'temp> 
{
    fn eq(&self, other: &str) -> bool 
    {
        match *self
        {
            Self::Name(name) => return name == other,
            Self::IpV4(ip) =>
            {
                if let Ok(other_ip) = other.parse::<Ipv4Addr>()
                {   
                    return &other_ip == ip;
                }
                
                return false;
            },
            Self::IpV6(ip) =>
            {
                if let Ok(other_ip) = other.parse::<Ipv6Addr>()
                {
                    return &other_ip == ip;
                }

                return false;
            }
        }
    }
}

impl<'temp> From<&'temp IpAddr> for QDnsName<'temp>
{
    fn from(ip: &'temp IpAddr) -> Self 
    {
        match *ip
        {
            IpAddr::V4(ref ip) => return Self::IpV4(ip),
            IpAddr::V6(ref ip) => return Self::IpV6(ip),
        }
    }
}

impl<'temp> From<&'temp Ipv4Addr> for QDnsName<'temp>
{
    fn from(ip: &'temp Ipv4Addr) -> Self 
    {
        return Self::IpV4(ip);
    }
}

impl<'temp> From<&'temp Ipv6Addr> for QDnsName<'temp>
{
    fn from(ip: &'temp Ipv6Addr) -> Self 
    {
        return Self::IpV6(ip);
    }
}

impl<'temp> From<&'temp str> for QDnsName<'temp>
{
    fn from(name: &'temp str) -> Self 
    {
        return Self::Name(name);
    }
}

impl<'temp> TryInto<Vec<u8>> for QDnsName<'temp>
{
    type Error = CDnsError;
    fn try_into(self) -> Result<Vec<u8>, Self::Error> 
    {
        match self
        {
            Self::IpV4(ip) =>
            {
                return ipv4_pkt(ip);
            },
            Self::IpV6(ip) =>
            {
                return ipv6_pkt(ip);
            },
            Self::Name(name) =>
            {
                if let Ok(ip) = name.parse::<Ipv4Addr>()
                {   
                    return ipv4_pkt(&ip);
                }
                else if let Ok(ip) = name.parse::<Ipv6Addr>()
                {
                    return ipv6_pkt(&ip);
                }
                else
                {
                    return name2pkt(name);
                }
            }
        }
    }
}

impl<'temp> TryFrom<&QDnsName<'temp>> for IpAddr
{
    type Error = CDnsError;
    fn try_from(value: &QDnsName<'temp>) -> Result<Self, Self::Error> 
    {
        match *value
        {
            QDnsName::IpV4(ip) =>
            {
                return Ok(IpAddr::V4(ip.clone()));
            },
            QDnsName::IpV6(ip) =>
            {
                return Ok(IpAddr::V6(ip.clone()));
            },
            QDnsName::Name(name) =>
            {
                if let Ok(ip) = name.parse::<Ipv4Addr>()
                {   
                    return Ok(IpAddr::V4(ip.clone()));
                }
                else if let Ok(ip) = name.parse::<Ipv6Addr>()
                {
                    return Ok(IpAddr::V6(ip.clone()));
                }
                else
                {
                    internal_error!(CDnsErrorType::InternalError, "not ip address!")
                }
            }
        }
    }
}


pub struct QDns{}

impl QDns
{
    /// Constructs the request from input.
    fn construct_lookup(req: &mut DnsRequestHeader, name: QDnsName, qtype: QType) -> CDnsResult<()>
    {
        //let rng = rand::thread_rng();
        // preparing status register
        let mut status: StatusBits = StatusBits::empty();
        status = (status & !StatusBits::OPCODE_STANDARD) | StatusBits::RECURSION_DESIRED;

        // constructing request structure
        //let mut req: DnsRequestHeader = DnsRequestHeader{ ..Default::default() };

        req.header.id = rand::random();
        req.header.status = status;
        req.header.qdcount = 1;

        // creating payload request
        req.payload = DnsRequestPayload::new(name.try_into()?, qtype, QClass::IN);

        
        /*let pkt = req.to_bytes().map_err(|e|
            internal_error_map!(CDnsErrorType::InternalError, "{}", e)
        )?;*/

        return Ok(());
    }

    /// Performs search of FQDN by IP (reverse) in cache of the /etc/hosts. 
    /// If file was modified then the cache will be updated.
    async 
    fn search_hosts_by_ip(name: &QDnsName<'_>, now: Option<&Instant>) -> CDnsResult<Option<QDnsQuery>>
    {
        let hlist = CACHE.clone_host_list().await?;
        let ip: IpAddr = name.try_into()?;

        match hlist.get(&ip)
        {
            Some(r) => 
            {   
                return Ok(
                    Some(
                        QDnsQuery::from_local(
                            DnsResponsePayload::new_local(QType::PTR, r)?, 
                            now
                        )
                    )
                );
            },
            None => 
                return Ok(None),
        }
    }

    /// Performs search of IP by FQDN in cache of the /etc/hosts. 
    /// /// If file was modified then the cache will be updated.
    async 
    fn search_hosts_by_fqdn(name: &QDnsName<'_>, now: Option<&Instant>) -> CDnsResult<Option<QDnsQuery>>
    {
        let hlist = CACHE.clone_host_list().await?;

        for h in hlist.iter()
        {
            for fqdn in h.get_hostnames_iter()
            {
                if name == fqdn.as_str()
                {
                    if let Some(qtype) = name.get_ip_qtype()
                    {
                        return Ok(
                            Some(
                                QDnsQuery::from_local(
                                    DnsResponsePayload::new_local(qtype, h)?, 
                                    now
                                )
                            )
                        );
                    }
                    else 
                    {
                        continue;    
                    }
                }
            }
        }

        return Ok(None);
    }

    async 
    fn process_request1(
        pkt: &[u8], 
        request: &DnsRequestHeader, 
        resolver: &ResolveConfEntry, 
        opts: &QuerySetup, 
        now: Option<&Instant>
    ) -> CDnsResult<QDnsQuery>
    {
        let mut rcvbuf = vec![0_u8; 2048];

        let mut tap =  
                NetworkTap::new_udp(
                    resolver.get_resolver_ip(), 
                    resolver.get_adapter_ip(), 
                    opts.timeout.clone()
                ).await?;


        tap.send(pkt).await?;

        tap.recv(rcvbuf.as_mut_slice()).await?;

        // parsing response to structure
        let ans = 
            DnsRequestAnswer::parse(rcvbuf.as_slice(), &request).await?;

        //println!("debug: {:?}", ans.response);

        let qres = QDnsQuery::from_response(tap.get_remote_addr(), ans, now);

        return qres;
    }

    async 
    fn process_request(request: DnsRequestHeader, opts: QuerySetup, now: Option<Instant>) -> CDnsResult<QDnsQueriesRes>
    {
        // send to server, now obtaining DNS server list
        let resolvers = CACHE.clone_resolve_list().await?;
        let pkt = request.to_bytes().await?;

        let mut responses: LinkedList<QDnsQuery> = LinkedList::new();

        for resolver in resolvers.iter()
        { 
            let qres = 
                match Self::process_request1(pkt.as_slice(), &request, resolver.as_ref(), &opts, now.as_ref()).await
                {
                    Ok(r) => r,
                    Err(e) =>
                    {
                        write_error!("{}", e);
                        continue;
                    }
                };

                match qres.rec
                {
                    QDnsQueryRec::Ok(_) =>
                    {
                        responses.push_back(qres);
    
                        match opts.qmode
                        {   
                            QueryMode::DefaultMode | QueryMode::ForceNxQueryAll => break,
                            QueryMode::ForceQueryAll => {}
                        }
                        
                    },
                    QDnsQueryRec::NxDomain =>
                    {
                        let aa = qres.aa;
    
                        responses.push_back(qres);
    
                        // check if this is answer from authorative server
                        if aa == false
                        {
                            // the response is not from authorative server
                            
                            continue;
                        }
                        else if opts.qmode == QueryMode::ForceQueryAll
                        {
                            continue;
                        }
    
                        // return result
                        break;
                    },
                    _ =>
                    {
                        responses.push_back(qres);
    
                        continue;
                    }
                    
                } // match
        }

        return Ok( QDnsQueriesRes::from(responses) );
    }

    async 
    fn process_request2(
        pkt: Arc<Vec<u8>>, 
        request: Arc<DnsRequestHeader>, 
        resolver: Arc<ResolveConfEntry>, 
        opts: QuerySetup, 
        now: Option<&Instant>
    ) -> CDnsResult<QDnsQuery>
    {
        let mut rcvbuf = vec![0_u8; 2048];

        let mut tap =  
                NetworkTap::new_udp(
                    resolver.get_resolver_ip(), 
                    resolver.get_adapter_ip(), 
                    opts.timeout.clone()
                ).await?;


        tap.send(pkt.as_slice()).await?;

        tap.recv(rcvbuf.as_mut_slice()).await?;

        // parsing response to structure
        let ans = 
            DnsRequestAnswer::parse(rcvbuf.as_slice(), &request).await?;

        //println!("debug: {:?}", ans.response);

        let qres = QDnsQuery::from_response(tap.get_remote_addr(), ans, now);

        return qres;
    }

    /// Processing created request by sending it to the DNS servers. This function
    /// does not perform lookups in local databases.
    async 
    fn process_request_simultanious(request: DnsRequestHeader, opts: QuerySetup, now: Option<Instant>) -> CDnsResult<QDnsQueriesRes>
    {
        // send to server, now obtaining DNS server list
        let resolvers = CACHE.clone_resolve_list().await?;
        let pkt = Arc::new(request.to_bytes().await?);
        let c_request = Arc::new(request);

        let mut responses: LinkedList<QDnsQuery> = LinkedList::new();

        let mut handlers = Vec::with_capacity(resolvers.len());

        // starting quering the nameservers
        for resolver in resolvers.iter()
        { 
            let c_resolver = resolver.clone();
            let c_pkt = pkt.clone();
            let cc_request = c_request.clone();
            handlers.push(
                tokio::task::spawn(async move 
                    { 
                        Self::process_request2(c_pkt, cc_request, c_resolver, opts.clone(), now.as_ref()).await 
                    }
                )
            );            
        } // for

        // wait for every task to complete
        for h in handlers
        {
            match h.await
            {
                Ok(Ok(r)) => 
                    responses.push_back(r),
                Ok(Err(e)) => 
                    write_error!("{}", e),
                Err(e) =>
                    write_error!("{}", e),
            }
        }

        return Ok( QDnsQueriesRes::from(responses) );
    }

    /// Perfroms the `qtype` request, except PTR with `opts`.
    /// 
    /// # Arguments
    /// 
    /// * `name` - [QDnsName] an object to request
    /// 
    /// * `qtype` - [QType] a type of the request (question)
    /// 
    /// * `opts` - [QuerySetup] a settings
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] with [QDnsQueriesRes] which may contain data
    /// 
    /// * [CDnsResult] with error
    /// 
    /// Should not panic. MT-safe.
    async 
    fn lookup_optional(name: QDnsName<'_>, qtype: QType, opts: QuerySetup) -> CDnsResult<QDnsQueriesRes>
    {
        // at the moment this lib does not use nsswitch to read order
        // check hosts file firstly
        let now = 
            if opts.measure_time == true
            {
                Some(Instant::now())
            }
            else
            {
                None
            };

        if opts.ign_hosts == false
        {
            match Self::search_hosts_by_fqdn(&name, now.as_ref()).await
            {
                Ok(r) => 
                {
                    if r.is_some() == true
                    {
                        return Ok( QDnsQueriesRes::from(r.unwrap()) );
                    }
                },
                Err(e) =>
                {
                    write_error!("{}", e);
                }
            }
        }

        // nothing was found in /etc/hosts

        let mut req: DnsRequestHeader = DnsRequestHeader{ ..Default::default() };

        // constructs packet
        Self::construct_lookup(&mut req, name, qtype)?;
     
        match opts.qmode
        {
            QueryMode::ForceQueryAll => 
                return Self::process_request_simultanious(req, opts, now).await,
            _ => return Self::process_request(req, opts, now).await,
        }
        
    }

    /// Performes a query of the requested type with requested settings.
    /// 
    /// # Arguments
    /// 
    /// * `name` - [QDnsName] an item to look i.e IP or Domain name
    /// 
    /// * `qtypr` - [QType] a request type
    /// 
    /// * `opts` - [QuerySetup] an additional setup
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] with [QDnsQueriesRes] which may contain data
    /// 
    /// * [CDnsResult] with error
    /// 
    /// Should not panic. MT-safe.
    pub async 
    fn query_optional(name: QDnsName<'_>, qtype: QType, opts: QuerySetup) -> CDnsResult<QDnsQueriesRes>
    {
        if QType::PTR == qtype
        {
            return Self::reverse_lookup_optional(name, opts).await;
        }
        else
        {
            return Self::lookup_optional(name, qtype, opts).await;
        }
    }

    /// Performes a query of the requested type with default settings.
    /// 
    /// # Arguments
    /// 
    /// * `name` - [QDnsName] an item to look i.e IP or Domain name
    /// 
    /// * `qtypr` - [QType] a request type
    /// 
    /// # Returns
    /// 
    /// * [CDnsResult] with [QDnsQueriesRes] which may contain data
    /// 
    /// * [CDnsResult] with error
    /// 
    /// Should not panic. MT-safe.
    pub async 
    fn query(name: QDnsName<'_>, qtype: QType) -> CDnsResult<QDnsQueriesRes>
    {
        return Self::query_optional(name, qtype, QuerySetup::default()).await;
    }

    pub async 
    fn reverse_lookup_optional(name: QDnsName<'_>, opts: QuerySetup) -> CDnsResult<QDnsQueriesRes>
    {
        // at the moment this lib does not use nsswitch to read order
        // check hosts file firstly
        let now = 
            if opts.measure_time == true
            {
                Some(Instant::now())
            }
            else
            {
                None
            };

        if opts.ign_hosts == false
        {
            match Self::search_hosts_by_ip(&name, now.as_ref()).await
            {
                Ok(r) => 
                {
                    if r.is_some() == true
                    {
                        return Ok( QDnsQueriesRes::from(r.unwrap()) );
                    }
                },
                Err(e) =>
                {
                    write_error!("{}", e);
                }
            }
        }

        // nothing was found in /etc/hosts

        let mut req: DnsRequestHeader = DnsRequestHeader{ ..Default::default() };

        // constructs packet
        Self::construct_lookup(&mut req, name, QType::PTR)?;
     
        match opts.qmode
        {
            QueryMode::ForceQueryAll => 
                return Self::process_request_simultanious(req, opts, now).await,
            _ => return Self::process_request(req, opts, now).await,
        }
    }

    pub async 
    fn reverse_lookup(name: QDnsName<'_>) -> CDnsResult<QDnsQueriesRes>
    {
       return Self::reverse_lookup_optional(name, QuerySetup::default()).await;
    }
}


#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_request()
{
    use tokio::time::Instant;
    
    let ipp = IpAddr::V4(Ipv4Addr::new(8, 8, 8, 8));
    let test = QDnsName::from(&ipp);

    let mut req: DnsRequestHeader = DnsRequestHeader{ ..Default::default() };

    let now = Instant::now();

    let pkt = QDns::construct_lookup(&mut req, test, QType::PTR);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(pkt.is_ok(), true, "{}", pkt.err().unwrap());

    let pkt = req.to_bytes().await;
    assert_eq!(pkt.is_ok(), true);
    let pkt = pkt.unwrap();

    let ctrl = 
    b"\x15\xc8\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x01\x38\x01\x38\
    \x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\
    \x61\x00\x00\x0c\x00\x01";

    assert_eq!(&pkt[2..], &ctrl[2..]);
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_request_1()
{
    use tokio::time::Instant;
    
    let ipp = IpAddr::V4("100.150.111.80".parse().unwrap());
    let test = QDnsName::from(&ipp);
    
    let mut req: DnsRequestHeader = DnsRequestHeader{ ..Default::default() };

    let now = Instant::now();

    let pkt = QDns::construct_lookup(&mut req, test, QType::PTR);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(pkt.is_ok(), true, "{}", pkt.err().unwrap());

    let pkt = req.to_bytes().await;
    assert_eq!(pkt.is_ok(), true);
    let pkt = pkt.unwrap();

    let ctrl = 
    b"\x74\xa1\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x02\x38\x30\x03\
    \x31\x31\x31\x03\x31\x35\x30\x03\x31\x30\x30\x07\x69\x6e\x2d\x61\
    \x64\x64\x72\x04\x61\x72\x70\x61\x00\x00\x0c\x00\x01";
    
    assert_eq!(&pkt[2..], &ctrl[2..]);
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_request6()
{
    use tokio::time::Instant;
    
    let ipp =  IpAddr::V6("2a00:1450:4003:802::200e".parse().unwrap());
    let test = QDnsName::from(&ipp);

    let mut req: DnsRequestHeader = DnsRequestHeader{ ..Default::default() };

    let now = Instant::now();

    let pkt = QDns::construct_lookup(&mut req, test, QType::PTR);
    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(pkt.is_ok(), true, "{}", pkt.err().unwrap());

    let pkt = req.to_bytes().await;
    assert_eq!(pkt.is_ok(), true);
    let pkt = pkt.unwrap();

    let ctrl = 
    b"\xee\xec\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x01\x65\x01\x30\
    \x01\x30\x01\x32\x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\
    \x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\x01\x30\x01\x32\x01\x30\
    \x01\x38\x01\x30\x01\x33\x01\x30\x01\x30\x01\x34\x01\x30\x01\x35\
    \x01\x34\x01\x31\x01\x30\x01\x30\x01\x61\x01\x32\x03\x69\x70\x36\
    \x04\x61\x72\x70\x61\x00\x00\x0c\x00\x01";

    assert_eq!(&pkt[2..], &ctrl[2..]);
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_ip2pkt()
{
    use tokio::time::Instant;
    
    let test = IpAddr::V4(Ipv4Addr::new(8, 8, 8, 8));

    let now = Instant::now();

    let res = ip2pkt(&test);

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_ok(), true, "err: {}", res.err().unwrap());

    let res = res.unwrap();
    let ctrl = b"\x01\x38\x01\x38\x01\x38\x01\x38\x07\x69\x6e\x2d\x61\x64\x64\x72\x04\x61\x72\x70\x61\x00";

    assert_eq!(res.as_slice(), ctrl);
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_byte2hexchar()
{
    assert_eq!(byte2hexchar(1), 0x31);
    assert_eq!(byte2hexchar(9), 0x39);
    assert_eq!(byte2hexchar(10), 'a' as u8);
    assert_eq!(byte2hexchar(15), 'f' as u8);
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn reverse_lookup_test()
{
    use tokio::time::Instant;
    
    let ipp: IpAddr = "8.8.8.8".parse().unwrap();
    let test = QDnsName::from(&ipp);

    let now = Instant::now();
    let res = 
        QDns::reverse_lookup_optional(test, QuerySetup::new(true, false, QueryMode::ForceQueryAll, Some(Duration::from_secs(3)))).await;

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_ok(), true);
    
    let res = res.unwrap();

    println!("{}", res);

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            for dq in res
            {
                match dq.rec
                {
                    QDnsQueryRec::Ok(r) =>
                    {
                        assert_eq!(r.len(), 1);
                        assert_eq!(r[0].rdata, DnsRdata::PTR{ fqdn: "dns.google".to_string() });
                    },
                    _ => assert_eq!(true, false, "QDnsQueryRec expected OK"),
                }
            }
        },
        _ => assert_eq!(true, false, "expected DnsResultSingle"),
    }
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn reverse_lookup_hosts_test()
{
    use tokio::time::Instant;

    let ipp: IpAddr = "127.0.0.1".parse().unwrap();
    let test = QDnsName::from(&ipp);
    
    let now = Instant::now();
    let res = 
        QDns::reverse_lookup_optional(test, QuerySetup::new(true, false, QueryMode::ForceQueryAll, Some(Duration::from_secs(3)))).await;

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_ok(), true);
    
    let res = res.unwrap();

    println!("{}", res);

    match res
    {
        QDnsQueriesRes::DnsOk{ res } =>
        {
            let rec = &res[0];

            assert_eq!(rec.server.as_str(), "/etc/hosts");
            
            match &rec.rec
            {
                QDnsQueryRec::Ok(r) =>
                {
                    assert_eq!(r.len(), 1);
                    assert_eq!(r[0].rdata, DnsRdata::PTR{ fqdn: "localhost".to_string() });
                },
                _ => assert_eq!(true, false, "QDnsQueryRec expected OK"),
            }
        },
        _ => assert_eq!(true, false, "expected DnsResultSingle"),
    }
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn reverse_lookup_a()
{
    use tokio::time::Instant;

    let test = QDnsName::from("dns.google");
    
    let now = Instant::now();
    let res = 
        QDns::query_optional(test, QType::A, QuerySetup::new(true, false, QueryMode::ForceQueryAll, Some(Duration::from_secs(3)))).await;

    let elapsed = now.elapsed();
    println!("Elapsed: {:.2?}", elapsed);

    assert_eq!(res.is_ok(), true);
    
    let res = res.unwrap();
    println!("{}", res);
}
