/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/// This file contains a networking code.

use std::net::{IpAddr, Ipv4Addr, Ipv6Addr, SocketAddr};

use tokio::io::ErrorKind;
use tokio::net::UdpSocket;
use tokio::time::{self, Duration};


use crate::{internal_error, internal_error_map};

use crate::error::*;

/// A types of the communication channels which are supported.
/// It is used for data transmission between DNS client and DNS server.
pub enum NetworkTap
{
    /// A UDP stream
    UdpStream{ sock: UdpSocket, remote_addr: SocketAddr, timeout: Duration },
}

impl NetworkTap
{
    const IPV4_BIND_ALL: IpAddr = IpAddr::V4(Ipv4Addr::new(0, 0, 0, 0));
    const IPV6_BIND_ALL: IpAddr = IpAddr::V6(Ipv6Addr::new(0, 0, 0, 0, 0, 0, 0, 0));

    pub async
    fn new_udp(resolver_ip: &IpAddr, bind_ip: Option<&IpAddr>, timeout: Option<Duration>) -> CDnsResult<Self>
    {
        let bind_addr = 
            match resolver_ip
            {
                IpAddr::V4(_) => 
                    SocketAddr::from((bind_ip.unwrap_or(&Self::IPV4_BIND_ALL).clone(), 0)),
                IpAddr::V6(_) => 
                    SocketAddr::from((bind_ip.unwrap_or(&Self::IPV6_BIND_ALL).clone(), 0)),
            };

        //println!("debug: trying to bind: '{}'", bind_addr);

        let socket = 
            UdpSocket::bind(bind_addr).await.map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

        // socket timeout can not be set in async mode

        // setting address and port
        let remote_dns_host = SocketAddr::from((resolver_ip.clone(), 53));

        socket.connect(&remote_dns_host).await.map_err(|e| internal_error_map!(CDnsErrorType::InternalError, "{}", e))?;

        return Ok(
            Self::UdpStream
            { 
                sock: socket, 
                remote_addr: remote_dns_host,
                timeout: timeout.unwrap_or(Duration::from_secs(1)),
            }
        );
    }


    /// Reads the remote host's address and port and returns it.
    /// 
    /// # Returns
    /// 
    /// * [SocketAddr] a reference to remote host address and port
    pub 
    fn get_remote_addr(&self) -> &SocketAddr
    {
        match *self
        {
            Self::UdpStream{ ref remote_addr, .. } => return remote_addr,
        }
    }

    /// Sends data over channel synchroniosly
    pub async
    fn send(&mut self, sndbuf: &[u8]) -> CDnsResult<()>
    {
         // sending request
         let n = 
            match *self
            {
                Self::UdpStream{ ref mut sock, timeout, .. } =>
                {
                    let tm_res = 
                        time::timeout(timeout, sock.send(sndbuf)).await;

                    match tm_res
                    {
                        Ok(snd_res) => 
                            snd_res.map_err(|e| internal_error_map!(CDnsErrorType::IoError, "{}", e))?,
                        Err(e) =>
                            internal_error!(
                                CDnsErrorType::DnsResponse, 
                                "request send timeout from: '{}' elapsed: '{}'", 
                                self.get_remote_addr(), e
                            ),
                    }
                    
                }
            };
         
         //println!("debug: r = {}", n);

         return Ok(());
    }

    /// Receives data from channel synchroniosly
    pub async
    fn recv(&mut self, rcvbuf: &mut [u8]) -> CDnsResult<usize>
    {
        match *self
        {
            Self::UdpStream{ ref mut sock, ref remote_addr, timeout } =>
            {
                loop
                {
                    let tm_res = 
                        time::timeout(timeout, sock.recv_from(rcvbuf)).await;

                    let rcv_res = 
                        match tm_res
                        {
                            Err(e) =>
                                internal_error!(
                                    CDnsErrorType::DnsResponse, 
                                    "request receive timeout from: '{}' elapsed: '{}'", 
                                    self.get_remote_addr(), e
                                ),
                            Ok(r) => r
                        };

                    match rcv_res
                    {
                        Ok((rcv_len, rcv_src)) =>
                        {
                            // this should not fail because socket is "connected"
                            if &rcv_src != remote_addr
                            {
                                internal_error!(
                                    CDnsErrorType::DnsResponse, 
                                    "received answer from unknown host: '{}' exp: '{}'", 
                                    remote_addr, 
                                    rcv_src
                                );
                            }

                            return Ok(rcv_len);
                        },
                        Err(ref e) if e.kind() == ErrorKind::WouldBlock =>
                        {
                            // can WouldBlock happen in async mode?
                            continue;
                        },
                        Err(ref e) if e.kind() == ErrorKind::Interrupted =>
                        {
                            continue;
                        },
                        Err(e) =>
                        {
                            // exit with error
                            internal_error!(CDnsErrorType::IoError, "{}", e); 
                        }
                    } // match
                } // loop
            } // UdpStream
        } // match
    } 
}