/*-
* cdns-rs - a simple sync/async DNS query library
* Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/// This file contains configuration caches.

use std::collections::HashSet;
use std::sync::Arc;
use std::time::SystemTime;
use tokio::fs;

use tokio::sync::Mutex;

use crate::{write_error, internal_error, internal_error_map, error::*};

use super::cfg_parsers::*;
use super::common::{RESOLV_CFG_PATH, HOST_CFG_PATH};

lazy_static!{
    pub static ref CACHE: CachesController = CachesController::new();
}

#[derive(Clone, Debug)]
struct CacheTime<'cache>
{
    path: &'cache str,
    last_modif: SystemTime,
}

impl<'cache> CacheTime<'cache>
{
    async 
    fn new(path: &'cache str) -> CDnsResult<CacheTime<'cache>>
    {
        return Ok(CacheTime { path: path, last_modif: Self::get_last_modified(path).await? });
    }

    fn fake(path: &'cache str) -> Self
    {
        return Self { path: path, last_modif: SystemTime::now() };
    }

    async 
    fn get_last_modified(path: &'cache str) -> CDnsResult<SystemTime>
    {
        let metadata = fs::metadata(path).await.map_err(|e| 
            internal_error_map!(CDnsErrorType::InternalError, "fs::metadata::modified() not supported on this platform: '{}'", e)
        )?;

        let last_modif = 
            metadata.modified().map_err(|e| 
                internal_error_map!(CDnsErrorType::InternalError, "fs::metadata::modified() not supported on this platform: '{}'", e)
            )?;

        return Ok(last_modif);
    }

    async 
    fn check_modified(&mut self) -> CDnsResult<bool>
    {
        // in case if get_last_modified() will return Err, return what was cached prevously
        let last_modif = 
            match Self::get_last_modified(self.path).await
            {
                Ok(t) => t,
                Err(e) => internal_error!(CDnsErrorType::InternalError, "{}", e),
            };

        return Ok(self.last_modif != last_modif);
        /*{
            // reload
            match ResolveConfEntry::parse_resolve_cfg()
            {
                Ok(r) => 
                    self.cache = Arc::new(r),
                Err(e) =>
                    write_error!("{}", e),
            }
             
        }
        
        return;*/
    }
}

#[derive(Clone, Debug)]
pub struct ConfigCacheResolve
{
    last_modif: CacheTime<'static>,
    cache: Arc<Vec<Arc<ResolveConfEntry>>>,
}

impl Default for ConfigCacheResolve
{
    fn default() -> Self 
    {
        return Self {last_modif: CacheTime::fake(RESOLV_CFG_PATH), cache: Arc::new(Vec::new()) };
    }
}

impl ConfigCacheResolve
{
    async 
    fn new() -> CDnsResult<Self>
    {
        return Ok(
            Self 
            {
                last_modif: CacheTime::new(RESOLV_CFG_PATH).await?, 
                cache: Self::read_config().await?
            }
        );
    }

    async 
    fn read_config() -> CDnsResult<Arc<Vec<Arc<ResolveConfEntry>>>>
    {
        return Ok(Arc::new(ResolveConfEntry::parse_resolve_cfg().await?));
    }

    async 
    fn check_modified(&mut self) -> CDnsResult<()>
    {
        // in case if get_last_modified() will return Err, save what was cached prevously

        if self.last_modif.check_modified().await? == true
        {
            // reload
            match ResolveConfEntry::parse_resolve_cfg().await
            {
                Ok(r) => 
                    self.cache = Arc::new(r),
                Err(e) =>
                    return Err(e),
            }
             
        }
        
        return Ok(());
    }

    fn clone_cache(&self) -> Arc<Vec<Arc<ResolveConfEntry>>>
    {
        return self.cache.clone();
    }
}

#[derive(Clone, Debug)]
pub struct ConfigCacheHosts
{
    last_modif: CacheTime<'static>,
    cache: Arc<HashSet<HostnameEntry>>,
}

impl Default for ConfigCacheHosts
{
    fn default() -> Self 
    {
        return Self {last_modif: CacheTime::fake(HOST_CFG_PATH), cache: Arc::new(HashSet::new()) };
    }
}

impl ConfigCacheHosts
{
    async 
    fn new() -> CDnsResult<Self>
    {
        return Ok(
            Self {last_modif: CacheTime::new(HOST_CFG_PATH).await?, cache: Self::read_config().await? }
        );
    }

    async 
    fn read_config() -> CDnsResult<Arc<HashSet<HostnameEntry>>>
    {
        return Ok(
            Arc::new(
            HostnameEntry::parse_host_file(true).await?.into_iter().map(|f| f).collect()
            )
        );
    }

    async 
    fn check_modified(&mut self) -> CDnsResult<()>
    {
        // in case if get_last_modified() will return Err, save what was cached prevously

        if self.last_modif.check_modified().await? == true
        {
            // reload
            match HostnameEntry::parse_host_file(true).await
            {
                Ok(r) =>
                    self.cache = Arc::new(r.into_iter().map(|f| f).collect()),
                Err(e) =>
                    return Err(e),
            }
        }
        

        return Ok(());
    }

    fn clone_cache(&self) -> Arc<HashSet<HostnameEntry>>
    {
        return self.cache.clone();
    }
}

pub struct CachesController
{
    resolv_cache: Mutex<ConfigCacheResolve>,
    host_cache: Mutex<ConfigCacheHosts>
}

unsafe impl Sync for CachesController{}
unsafe impl Send for CachesController{}

impl CachesController
{ 
    fn new() -> Self
    {
        /*let resolve_cache = ConfigCacheResolve::new().await;

        let host_cache = ConfigCacheHosts::new().await;

        if resolve_cache.is_err() == true
        {
            write_error!("{}", resolve_cache.as_ref().err().unwrap());
        }
        
        if host_cache.is_err() == true
        {
            write_error!("{}", host_cache.as_ref().err().unwrap());
        }*/

        return CachesController
            { 
                resolv_cache: Mutex::new(ConfigCacheResolve::default()),
                host_cache: Mutex::new(ConfigCacheHosts::default()),
            };
    }

    pub async
    fn is_resolve_cached(&self) -> bool
    {
        return !self.resolv_cache.lock().await.cache.is_empty();
    }

    pub async 
    fn is_host_cached(&self) -> bool
    {
        return !self.host_cache.lock().await.cache.is_empty();
    }

    async 
    fn try_cache_resolve(&self) -> CDnsResult<Arc<Vec<Arc<ResolveConfEntry>>>>
    {
        let mut mutx_resolv = self.resolv_cache.lock().await;
        mutx_resolv.check_modified().await?;

        return Ok(mutx_resolv.clone_cache());
    }

    pub async 
    fn clone_resolve_list(&self) -> CDnsResult<Arc<Vec<Arc<ResolveConfEntry>>>>
    {
        // clone from cache and/or reload cache
        match self.try_cache_resolve().await
        {
            Ok(r) => return Ok(r),
            Err(e) => 
            {
                write_error!("{}", e);
                // try to read directly
                return ConfigCacheResolve::read_config().await;
            }
        }
    }

    async 
    fn try_cache_host(&self) -> CDnsResult<Arc<HashSet<HostnameEntry>>>
    {
        let mut mutx_host = self.host_cache.lock().await;
        mutx_host.check_modified().await?;

        return Ok(mutx_host.clone_cache());
    }

    pub async 
    fn clone_host_list(&self) -> CDnsResult<Arc<HashSet<HostnameEntry>>>
    {
        match self.try_cache_host().await
        {
            Ok(r) => return Ok(r),
            Err(e) =>
            {
                write_error!("{}", e);
                // try to read directly
                return ConfigCacheHosts::read_config().await;
            }
        }
    }
}


#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_init()
{
    let res = CACHE.clone_resolve_list().await;
    assert_eq!(res.is_ok(), true, "{}", res.err().unwrap());

    let res = res.unwrap();

    println!("{:?}", res);

    let res = CACHE.clone_host_list().await;
    assert_eq!(res.is_ok(), true, "{}", res.err().unwrap());

    let res = res.unwrap();

    println!("{:?}", res);
}
