use std::path::Path;

use anyhow::{Context, Result};
use clap::Parser;
use owo_colors::OwoColorize;

use crate::config_file::ConfigFile;

#[derive(Debug, Parser)]
#[clap(
    about = "Query about config file data used by CLI",
    arg_required_else_help = true
)]
#[allow(clippy::struct_excessive_bools)]
pub(crate) struct Config {
    #[clap(long = "directory", short = 'd', help = "Query about directory data")]
    directory: bool,
    #[clap(
        long = "ignore",
        short = 'i',
        help = "Query about ignored file name data"
    )]
    ignore: bool,
    #[clap(long = "location", short = 'l', help = "Return config file location")]
    location: bool,
    #[clap(long = "print", short = 'p', help = "Display config file content")]
    print: bool,
}

impl Config {
    pub(super) fn run(&self, config_file: &ConfigFile, config_file_location: &Path) -> Result<()> {
        if self.directory {
            let read_directory = config_file.directory();
            for name in read_directory {
                println!("{}", name);
            }
        }
        if self.ignore {
            let read_ignore_file_name = config_file.ignore_file_name();
            for name in read_ignore_file_name {
                println!("{}", name);
            }
        }
        if self.location {
            println!(
                "{}: {:?}",
                "Config file location".blue(),
                config_file_location
            );
        }
        if self.print {
            let content = toml::to_string_pretty(config_file)
                .context("Failed to convert struct to pretty toml")?;
            println!("{}", content);
        }
        Ok(())
    }
}
