use crate::sonar::TextRange;
use std::{
    collections::BTreeMap,
    fs::File,
    io::{BufRead as _, BufReader},
    path::{Path, PathBuf},
};
use tracing::instrument;

#[derive(Debug)]
pub struct PackageRange {
    pub range: TextRange,
    pub name_range: TextRange,
    pub version_range: TextRange,
}

#[derive(Debug)]
pub struct Lockfile {
    pub lockfile_path: PathBuf,
    pub dependencies: BTreeMap<String, PackageRange>,
}

enum StateParser {
    // on '[[package]]'
    Package {
        start_line: usize,
    },
    // on 'name = "package-name"'
    Name {
        start_line: usize,
        package_name: String,
        name_range: TextRange,
    },
    // on 'version = "1.2.3"'
    Version {
        start_line: usize,
        package_name: String,
        name_range: TextRange,
        version_range: TextRange,
    },
    // on empty line
    End,
}

impl TryFrom<&Path> for Lockfile {
    type Error = eyre::Error;

    #[instrument(level = "debug")]
    fn try_from(lockfile_path: &Path) -> Result<Self, Self::Error> {
        let lockfile_path = lockfile_path.canonicalize()?;
        let mut dependencies = BTreeMap::new();
        let file = File::open(&lockfile_path)?;
        let reader = BufReader::new(file);
        use StateParser::*;
        let mut state = End;
        for (line_num, line) in reader.lines().enumerate() {
            let line_num = line_num + 1;
            let line = line?;
            let create_range = |line: &str| {
                let start_column = line.find('"').unwrap_or(0);
                let end_column = line.rfind('"').unwrap_or_else(|| line.len() - 1);
                TextRange {
                    start_line: line_num,
                    end_line: line_num,
                    start_column,
                    end_column,
                }
            };
            match state {
                End if line == "[[package]]" => {
                    state = Package {
                        start_line: line_num,
                    };
                }
                Package { start_line } if line.starts_with("name = ") => {
                    let name_range = create_range(&line);
                    state = Name {
                        start_line,
                        package_name: line.replace("name = ", "").replace('"', ""),
                        name_range,
                    };
                }
                Name {
                    start_line,
                    package_name,
                    name_range,
                } if line.starts_with("version = ") => {
                    let version_range = create_range(&line);
                    state = Version {
                        start_line,
                        package_name,
                        name_range,
                        version_range,
                    };
                }
                Version {
                    start_line,
                    package_name,
                    name_range,
                    version_range,
                } if line.is_empty() => {
                    let range = TextRange {
                        start_line,
                        end_line: line_num,
                        start_column: 0,
                        end_column: 0,
                    };
                    let package_range = PackageRange {
                        range,
                        name_range,
                        version_range,
                    };
                    dependencies.insert(package_name, package_range);
                    state = End;
                }
                s => state = s,
            }
        }
        let lockfile = Self {
            lockfile_path,
            dependencies,
        };
        Ok(lockfile)
    }
}
