use crate::{cargo::Lockfile, sonar};
use eyre::{Context, Result};
use rustsec::{Report, Vulnerability};
use std::{fs::File, path::PathBuf};
use tracing::{error, info};

const AUDIT_ENGINE: &str = "audit";

pub struct Audit<'ranges> {
    json: PathBuf,
    lockfile: &'ranges Lockfile,
}

impl<'ranges> Audit<'ranges> {
    pub fn new<P>(json: P, lockfile: &'ranges Lockfile) -> Self
    where
        P: Into<PathBuf>,
    {
        Self {
            json: json.into(),
            lockfile,
        }
    }

    // FIXME: reporting on 'Cargo.toml' is a bad idea because the vulnerability might exist on a nested dependency
    // it might be possible to use 'Cargo.lock' instead (or use 'cargo-deny' to obtain a graph of dependency and find the root dependency)
    fn to_issue(&self, vulnerability: &Vulnerability) -> sonar::Issue {
        let advisory = &vulnerability.advisory;
        let rule_id = Some(advisory.id.to_string());
        let message = format!(
            "{} (see https://github.com/rustsec/advisory-db/blob/main/crates/{}/{}.md)",
            advisory.title, advisory.package, advisory.id
        );
        let file_path = self.lockfile.lockfile_path.to_string_lossy().to_string();
        let dependency = self.lockfile.dependencies.get(advisory.package.as_str());
        let text_range = dependency
            .map(|ranges| &ranges.range)
            .map(sonar::TextRange::clone)
            .unwrap_or_else(|| {
                error!("failed to find a corresponding text range for this vulnerability. This is probably a bug in `cargo-sonar`, if you can please report an issue at https://gitlab.com/woshilapin/cargo-sonar/-/issues, we'd be happy to try and fix it.");
                sonar::TextRange {
                    start_line: 1,
                    end_line: 1,
                    start_column: 0,
                    end_column: 1,
                }
            });
        let primary_location = sonar::Location {
            message,
            file_path,
            text_range,
        };
        let secondary_locations = Vec::new();
        sonar::Issue {
            engine_id: AUDIT_ENGINE.to_string(),
            rule_id,
            severity: sonar::Severity::Major,
            r#type: sonar::IssueType::Vulnerability,
            primary_location,
            secondary_locations,
        }
    }
    pub fn to_issues(&self, report: Report) -> sonar::Report {
        report
            .vulnerabilities
            .list
            .iter()
            .map(|vulnerability| self.to_issue(vulnerability))
            .collect()
    }
}

impl std::convert::TryInto<sonar::Report> for Audit<'_> {
    type Error = eyre::Error;

    fn try_into(self) -> Result<sonar::Report> {
        let file = File::open(&self.json).with_context(|| {
            format!(
                "failed to open 'cargo-audit' report from '{:?}' file",
                self.json
            )
        })?;
        let report = serde_json::from_reader::<_, Report>(file)
            .context("failed to be parsed as a 'rustsec::report::Report'")?;
        let issues = self.to_issues(report);
        info!("{} sonar issues created", issues.len());
        Ok(issues)
    }
}
