#![doc = include_str!("../README.md")]

#[cfg(feature = "audit")]
mod audit;
#[cfg(feature = "audit")]
mod cargo;
mod cli;
#[cfg(feature = "clippy")]
mod clippy;
mod sonar;

use clap::Parser;
use eyre::{Context as _, Result};
use tracing::{info, Level};
use tracing_subscriber::{prelude::*, EnvFilter};

fn init_tracer() {
    let default_level = Level::INFO;
    let rust_log =
        std::env::var(EnvFilter::DEFAULT_ENV).unwrap_or_else(|_| default_level.to_string());
    let env_filter_subscriber = EnvFilter::try_new(rust_log).unwrap_or_else(|e| {
        eprintln!(
            "invalid {}, falling back to level '{}' - {}",
            EnvFilter::DEFAULT_ENV,
            default_level,
            e,
        );
        EnvFilter::new(default_level.to_string())
    });
    tracing_subscriber::registry()
        .with(tracing_subscriber::fmt::layer())
        .with(env_filter_subscriber)
        .init();
}

fn main() -> Result<()> {
    init_tracer();
    color_eyre::install()?;
    let options = cli::Command::parse();

    #[cfg(feature = "audit")]
    let cargo_ranges = {
        let cargo_toml = std::env::current_dir()?.join("Cargo.toml");
        cargo::CargoRanges::try_from(cargo_toml.as_path())?
    };
    let mut sonar_report = sonar::Report::default();
    #[cfg(feature = "audit")]
    if options.audit {
        use crate::audit::Audit;
        let audit_report: sonar::Report =
            Audit::new(options.audit_path, &cargo_ranges).try_into()?;
        sonar_report.extend(audit_report);
    }
    #[cfg(feature = "clippy")]
    if options.clippy {
        use crate::clippy::Clippy;
        let clippy_report: sonar::Report = Clippy::new(options.clippy_path).try_into()?;
        sonar_report.extend(clippy_report);
    }
    let file = std::fs::File::create("sonar.json").context("failed to create 'sonar.json' file")?;
    info!("{} sonar issues created", sonar_report.len());
    #[cfg(not(debug_assertions))]
    let writer = serde_json::to_writer;
    #[cfg(debug_assertions)]
    let writer = serde_json::to_writer_pretty;
    writer(file, &sonar_report).context("failed to write sonar issues to 'sonar.json' file")?;
    Ok(())
}
