mod audit;
mod cargo;
mod clippy;
mod sonar;
use crate::{audit::Audit, cargo::CargoRanges, clippy::Clippy, sonar::Issues};

use eyre::{Context as _, Result};
use tracing::{info, Level};
use tracing_subscriber::{prelude::*, EnvFilter};

fn init_tracer() {
    let default_level = Level::INFO;
    let rust_log =
        std::env::var(EnvFilter::DEFAULT_ENV).unwrap_or_else(|_| default_level.to_string());
    let env_filter_subscriber = EnvFilter::try_new(rust_log).unwrap_or_else(|e| {
        eprintln!(
            "invalid {}, falling back to level '{}' - {}",
            EnvFilter::DEFAULT_ENV,
            default_level,
            e,
        );
        EnvFilter::new(default_level.to_string())
    });
    tracing_subscriber::registry()
        .with(tracing_subscriber::fmt::layer())
        .with(env_filter_subscriber)
        .init();
}

fn main() -> Result<()> {
    init_tracer();
    color_eyre::install()?;
    let cargo_ranges =
        CargoRanges::try_from(std::env::current_dir()?.join("Cargo.toml").as_path())?;
    let clippy_issues: Issues = Clippy::new("clippy.json").try_into()?;
    let audit_issues: Issues = Audit::new("audit.json", &cargo_ranges).try_into()?;

    let mut issues = clippy_issues;
    issues.extend(audit_issues);

    let file = std::fs::File::create("sonar.json").context("failed to create 'sonar.json' file")?;
    info!("{} sonar issues created", issues.len());
    #[cfg(not(debug_assertions))]
    let writer = serde_json::to_writer;
    #[cfg(debug_assertions)]
    let writer = serde_json::to_writer_pretty;
    writer(file, &issues).context("failed to write sonar issues to 'sonar.json' file")?;
    Ok(())
}
